/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.collect.AbstractIterator;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Iterables;
import com.google.common.primitives.Doubles;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.columniterator.OnDiskAtomIterator;
import org.apache.cassandra.db.compaction.AbstractCompactionStrategy;
import org.apache.cassandra.db.compaction.AbstractCompactionTask;
import org.apache.cassandra.db.compaction.CompactionManager;
import org.apache.cassandra.db.compaction.LeveledCompactionTask;
import org.apache.cassandra.db.compaction.LeveledManifest;
import org.apache.cassandra.db.compaction.OperationType;
import org.apache.cassandra.db.compaction.SizeTieredCompactionStrategyOptions;
import org.apache.cassandra.db.lifecycle.LifecycleTransaction;
import org.apache.cassandra.dht.Range;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.io.sstable.ISSTableScanner;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LeveledCompactionStrategy
extends AbstractCompactionStrategy {
    private static final Logger logger = LoggerFactory.getLogger(LeveledCompactionStrategy.class);
    private static final String SSTABLE_SIZE_OPTION = "sstable_size_in_mb";
    @VisibleForTesting
    final LeveledManifest manifest;
    private final int maxSSTableSizeInMB;

    public LeveledCompactionStrategy(ColumnFamilyStore cfs, Map<String, String> options) {
        super(cfs, options);
        int configuredMaxSSTableSize = 160;
        SizeTieredCompactionStrategyOptions localOptions = new SizeTieredCompactionStrategyOptions(options);
        if (options != null && options.containsKey(SSTABLE_SIZE_OPTION)) {
            configuredMaxSSTableSize = Integer.parseInt(options.get(SSTABLE_SIZE_OPTION));
            if (!Boolean.getBoolean("cassandra.tolerate_sstable_size")) {
                if (configuredMaxSSTableSize >= 1000) {
                    logger.warn("Max sstable size of {}MB is configured for {}.{}; having a unit of compaction this large is probably a bad idea", new Object[]{configuredMaxSSTableSize, cfs.name, cfs.getColumnFamilyName()});
                }
                if (configuredMaxSSTableSize < 50) {
                    logger.warn("Max sstable size of {}MB is configured for {}.{}.  Testing done for CASSANDRA-5727 indicates that performance improves up to 160MB", new Object[]{configuredMaxSSTableSize, cfs.name, cfs.getColumnFamilyName()});
                }
            }
        }
        this.maxSSTableSizeInMB = configuredMaxSSTableSize;
        this.manifest = new LeveledManifest(cfs, this.maxSSTableSizeInMB, localOptions);
        logger.trace("Created {}", (Object)this.manifest);
    }

    public int getLevelSize(int i) {
        return this.manifest.getLevelSize(i);
    }

    public int[] getAllLevelSize() {
        return this.manifest.getAllLevelSize();
    }

    @Override
    public synchronized AbstractCompactionTask getNextBackgroundTask(int gcBefore) {
        OperationType op;
        LeveledManifest.CompactionCandidate candidate;
        LifecycleTransaction txn;
        do {
            if ((candidate = this.manifest.getCompactionCandidates()) == null) {
                SSTableReader sstable = this.findDroppableSSTable(gcBefore);
                if (sstable == null) {
                    logger.trace("No compaction necessary for {}", (Object)this);
                    return null;
                }
                candidate = new LeveledManifest.CompactionCandidate(Collections.singleton(sstable), sstable.getSSTableLevel(), this.getMaxSSTableBytes());
                op = OperationType.TOMBSTONE_COMPACTION;
                continue;
            }
            op = OperationType.COMPACTION;
        } while ((txn = this.cfs.getTracker().tryModify(candidate.sstables, OperationType.COMPACTION)) == null);
        LeveledCompactionTask newTask = new LeveledCompactionTask(this.cfs, txn, candidate.level, gcBefore, candidate.maxSSTableBytes, false);
        newTask.setCompactionType(op);
        return newTask;
    }

    @Override
    public synchronized Collection<AbstractCompactionTask> getMaximalTask(int gcBefore, boolean splitOutput) {
        Iterable<SSTableReader> sstables = this.manifest.getAllSSTables();
        Iterable<SSTableReader> filteredSSTables = LeveledCompactionStrategy.filterSuspectSSTables(sstables);
        if (Iterables.isEmpty(sstables)) {
            return null;
        }
        LifecycleTransaction txn = this.cfs.getTracker().tryModify(filteredSSTables, OperationType.COMPACTION);
        if (txn == null) {
            return null;
        }
        return Arrays.asList(new LeveledCompactionTask(this.cfs, txn, 0, gcBefore, this.getMaxSSTableBytes(), true));
    }

    @Override
    public AbstractCompactionTask getUserDefinedTask(Collection<SSTableReader> sstables, int gcBefore) {
        throw new UnsupportedOperationException("LevelDB compaction strategy does not allow user-specified compactions");
    }

    @Override
    public AbstractCompactionTask getCompactionTask(LifecycleTransaction txn, int gcBefore, long maxSSTableBytes) {
        assert (txn.originals().size() > 0);
        int level = -1;
        for (SSTableReader sstable : txn.originals()) {
            if (level == -1) {
                level = sstable.getSSTableLevel();
            }
            if (level == sstable.getSSTableLevel()) continue;
            level = 0;
        }
        return new LeveledCompactionTask(this.cfs, txn, level, gcBefore, maxSSTableBytes, false);
    }

    @Override
    public Collection<Collection<SSTableReader>> groupSSTablesForAntiCompaction(Collection<SSTableReader> ssTablesToGroup) {
        int groupSize = 2;
        HashMap sstablesByLevel = new HashMap();
        for (SSTableReader sstable : ssTablesToGroup) {
            Integer level = sstable.getSSTableLevel();
            if (!sstablesByLevel.containsKey(level)) {
                sstablesByLevel.put(level, new ArrayList());
            }
            ((Collection)sstablesByLevel.get(level)).add(sstable);
        }
        ArrayList<Collection<SSTableReader>> groupedSSTables = new ArrayList<Collection<SSTableReader>>();
        for (Collection levelOfSSTables : sstablesByLevel.values()) {
            ArrayList<SSTableReader> currGroup = new ArrayList<SSTableReader>();
            for (SSTableReader sstable : levelOfSSTables) {
                currGroup.add(sstable);
                if (currGroup.size() != groupSize) continue;
                groupedSSTables.add(currGroup);
                currGroup = new ArrayList();
            }
            if (currGroup.size() == 0) continue;
            groupedSSTables.add(currGroup);
        }
        return groupedSSTables;
    }

    @Override
    public int getEstimatedRemainingTasks() {
        return this.manifest.getEstimatedTasks();
    }

    @Override
    public long getMaxSSTableBytes() {
        return (long)this.maxSSTableSizeInMB * 1024L * 1024L;
    }

    @Override
    public AbstractCompactionStrategy.ScannerList getScanners(Collection<SSTableReader> sstables, Range<Token> range) {
        ArrayListMultimap byLevel = ArrayListMultimap.create();
        for (SSTableReader sSTableReader : sstables) {
            byLevel.get((Object)sSTableReader.getSSTableLevel()).add(sSTableReader);
        }
        ArrayList<ISSTableScanner> scanners = new ArrayList<ISSTableScanner>(sstables.size());
        try {
            for (Integer level : byLevel.keySet()) {
                if (level <= 0) {
                    for (SSTableReader sstable : byLevel.get((Object)level)) {
                        scanners.add(sstable.getScanner(range, CompactionManager.instance.getRateLimiter()));
                    }
                    continue;
                }
                List<SSTableReader> intersecting = LeveledScanner.intersecting(byLevel.get((Object)level), range);
                if (intersecting.isEmpty()) continue;
                scanners.add(new LeveledScanner(intersecting, range));
            }
        }
        catch (Throwable throwable) {
            try {
                new AbstractCompactionStrategy.ScannerList(scanners).close();
            }
            catch (Throwable t2) {
                throwable.addSuppressed(t2);
            }
            throw throwable;
        }
        return new AbstractCompactionStrategy.ScannerList(scanners);
    }

    @Override
    public void replaceSSTables(Collection<SSTableReader> removed, Collection<SSTableReader> added) {
        this.manifest.replace(removed, added);
    }

    @Override
    public void addSSTable(SSTableReader added) {
        this.manifest.add(added);
    }

    @Override
    public void removeSSTable(SSTableReader sstable) {
        this.manifest.remove(sstable);
    }

    public String toString() {
        return String.format("LCS@%d(%s)", this.hashCode(), this.cfs.name);
    }

    private SSTableReader findDroppableSSTable(final int gcBefore) {
        for (int i = this.manifest.getLevelCount(); i >= 0; --i) {
            SSTableReader sstable;
            SortedSet<SSTableReader> sstables = this.manifest.getLevelSorted(i, new Comparator<SSTableReader>(){

                @Override
                public int compare(SSTableReader o1, SSTableReader o2) {
                    double r1 = o1.getEstimatedDroppableTombstoneRatio(gcBefore);
                    double r2 = o2.getEstimatedDroppableTombstoneRatio(gcBefore);
                    return -1 * Doubles.compare((double)r1, (double)r2);
                }
            });
            if (sstables.isEmpty()) continue;
            Set<SSTableReader> compacting = this.cfs.getTracker().getCompacting();
            Iterator iterator = sstables.iterator();
            while (iterator.hasNext() && !((sstable = (SSTableReader)iterator.next()).getEstimatedDroppableTombstoneRatio(gcBefore) <= (double)this.tombstoneThreshold)) {
                if (compacting.contains(sstable) || sstable.isMarkedSuspect() || !this.worthDroppingTombstones(sstable, gcBefore)) continue;
                return sstable;
            }
        }
        return null;
    }

    public static Map<String, String> validateOptions(Map<String, String> options) throws ConfigurationException {
        Map<String, String> uncheckedOptions = AbstractCompactionStrategy.validateOptions(options);
        String size = options.containsKey(SSTABLE_SIZE_OPTION) ? options.get(SSTABLE_SIZE_OPTION) : "1";
        try {
            int ssSize = Integer.parseInt(size);
            if (ssSize < 1) {
                throw new ConfigurationException(String.format("%s must be larger than 0, but was %s", SSTABLE_SIZE_OPTION, ssSize));
            }
        }
        catch (NumberFormatException ex) {
            throw new ConfigurationException(String.format("%s is not a parsable int (base10) for %s", size, SSTABLE_SIZE_OPTION), ex);
        }
        uncheckedOptions.remove(SSTABLE_SIZE_OPTION);
        uncheckedOptions = SizeTieredCompactionStrategyOptions.validateOptions(options, uncheckedOptions);
        return uncheckedOptions;
    }

    private static class LeveledScanner
    extends AbstractIterator<OnDiskAtomIterator>
    implements ISSTableScanner {
        private final Range<Token> range;
        private final List<SSTableReader> sstables;
        private final Iterator<SSTableReader> sstableIterator;
        private final long totalLength;
        private ISSTableScanner currentScanner;
        private long positionOffset;

        public LeveledScanner(Collection<SSTableReader> sstables, Range<Token> range) {
            this.range = range;
            this.sstables = new ArrayList<SSTableReader>(sstables.size());
            long length = 0L;
            for (SSTableReader sstable : sstables) {
                this.sstables.add(sstable);
                long estimatedKeys = sstable.estimatedKeys();
                double estKeysInRangeRatio = 1.0;
                if (estimatedKeys > 0L && range != null) {
                    estKeysInRangeRatio = (double)sstable.estimatedKeysForRanges(Collections.singleton(range)) / (double)estimatedKeys;
                }
                length = (long)((double)length + (double)sstable.uncompressedLength() * estKeysInRangeRatio);
            }
            this.totalLength = length;
            Collections.sort(this.sstables, SSTableReader.sstableComparator);
            this.sstableIterator = this.sstables.iterator();
            assert (this.sstableIterator.hasNext());
            this.currentScanner = this.sstableIterator.next().getScanner(range, CompactionManager.instance.getRateLimiter());
        }

        public static List<SSTableReader> intersecting(Collection<SSTableReader> sstables, Range<Token> range) {
            ArrayList<SSTableReader> filtered = new ArrayList<SSTableReader>();
            for (SSTableReader sstable : sstables) {
                Range<Token> sstableRange = new Range<Token>(sstable.first.getToken(), sstable.last.getToken());
                if (range != null && !sstableRange.intersects(range)) continue;
                filtered.add(sstable);
            }
            return filtered;
        }

        protected OnDiskAtomIterator computeNext() {
            if (this.currentScanner == null) {
                return (OnDiskAtomIterator)this.endOfData();
            }
            try {
                while (true) {
                    if (this.currentScanner.hasNext()) {
                        return (OnDiskAtomIterator)this.currentScanner.next();
                    }
                    this.positionOffset += this.currentScanner.getLengthInBytes();
                    this.currentScanner.close();
                    if (!this.sstableIterator.hasNext()) {
                        this.currentScanner = null;
                        return (OnDiskAtomIterator)this.endOfData();
                    }
                    this.currentScanner = this.sstableIterator.next().getScanner(this.range, CompactionManager.instance.getRateLimiter());
                }
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }

        @Override
        public void close() throws IOException {
            if (this.currentScanner != null) {
                this.currentScanner.close();
            }
        }

        @Override
        public long getLengthInBytes() {
            return this.totalLength;
        }

        @Override
        public long getCurrentPosition() {
            return this.positionOffset + (this.currentScanner == null ? 0L : this.currentScanner.getCurrentPosition());
        }

        @Override
        public String getBackingFiles() {
            return Joiner.on((String)", ").join(this.sstables);
        }
    }
}

