/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.driver.core;

import com.datastax.driver.core.Cluster;
import com.datastax.driver.core.Host;
import com.datastax.driver.core.LatencyTracker;
import com.datastax.driver.core.Statement;
import com.datastax.driver.core.exceptions.BootstrappingException;
import com.datastax.driver.core.exceptions.DriverInternalError;
import com.datastax.driver.core.exceptions.OverloadedException;
import com.datastax.driver.core.exceptions.QueryValidationException;
import com.datastax.driver.core.exceptions.UnavailableException;
import com.datastax.driver.core.exceptions.UnpreparedException;
import java.util.Set;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.HdrHistogram.Histogram;
import org.HdrHistogram.Recorder;
import org.apache.flink.cassandra.shaded.com.google.common.annotations.Beta;
import org.apache.flink.cassandra.shaded.com.google.common.base.Preconditions;
import org.apache.flink.cassandra.shaded.com.google.common.collect.ImmutableSet;
import org.apache.flink.cassandra.shaded.com.google.common.collect.MapMaker;
import org.apache.flink.cassandra.shaded.com.google.common.util.concurrent.Futures;
import org.apache.flink.cassandra.shaded.com.google.common.util.concurrent.ListenableFuture;
import org.apache.flink.cassandra.shaded.com.google.common.util.concurrent.SettableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Beta
public class PerHostPercentileTracker
implements LatencyTracker {
    private static final Logger logger = LoggerFactory.getLogger(PerHostPercentileTracker.class);
    private final ConcurrentMap<Host, Recorder> recorders;
    private final ConcurrentMap<Host, CachedHistogram> cachedHistograms;
    private final long highestTrackableLatencyMillis;
    private final int numberOfSignificantValueDigits;
    private final int minRecordedValues;
    private final long intervalMs;
    private static final Set<Class<? extends Exception>> EXCLUDED_EXCEPTIONS = ImmutableSet.of(UnavailableException.class, OverloadedException.class, BootstrappingException.class, UnpreparedException.class, QueryValidationException.class);

    private PerHostPercentileTracker(long highestTrackableLatencyMillis, int numberOfSignificantValueDigits, int numberOfHosts, int minRecordedValues, long intervalMs) {
        this.highestTrackableLatencyMillis = highestTrackableLatencyMillis;
        this.numberOfSignificantValueDigits = numberOfSignificantValueDigits;
        this.minRecordedValues = minRecordedValues;
        this.intervalMs = intervalMs;
        this.recorders = new MapMaker().initialCapacity(numberOfHosts).makeMap();
        this.cachedHistograms = new MapMaker().initialCapacity(numberOfHosts).makeMap();
    }

    public static Builder builderWithHighestTrackableLatencyMillis(long highestTrackableLatencyMillis) {
        return new Builder(highestTrackableLatencyMillis);
    }

    @Override
    public void update(Host host, Statement statement, Exception exception, long newLatencyNanos) {
        if (!this.shouldConsiderNewLatency(statement, exception)) {
            return;
        }
        long latencyMs = TimeUnit.NANOSECONDS.toMillis(newLatencyNanos);
        try {
            this.getRecorder(host).recordValue(latencyMs);
        }
        catch (ArrayIndexOutOfBoundsException e) {
            logger.warn("Got request with latency of {} ms, which exceeds the configured maximum trackable value {}", (Object)latencyMs, (Object)this.highestTrackableLatencyMillis);
        }
    }

    public long getLatencyAtPercentile(Host host, double percentile) {
        Preconditions.checkArgument(percentile >= 0.0 && percentile < 100.0, "percentile must be between 0.0 and 100 (was %f)");
        Histogram histogram = this.getLastIntervalHistogram(host);
        if (histogram == null || histogram.getTotalCount() < (long)this.minRecordedValues) {
            return -1L;
        }
        return histogram.getValueAtPercentile(percentile);
    }

    private Recorder getRecorder(Host host) {
        Recorder recorder = (Recorder)this.recorders.get(host);
        if (recorder == null) {
            recorder = new Recorder(this.highestTrackableLatencyMillis, this.numberOfSignificantValueDigits);
            Recorder old = this.recorders.putIfAbsent(host, recorder);
            if (old != null) {
                recorder = old;
            } else {
                this.cachedHistograms.putIfAbsent(host, CachedHistogram.empty());
            }
        }
        return recorder;
    }

    private Histogram getLastIntervalHistogram(Host host) {
        try {
            Histogram staleHistogram;
            Recorder recorder;
            SettableFuture<Histogram> future;
            CachedHistogram newEntry;
            CachedHistogram entry;
            do {
                if ((entry = (CachedHistogram)this.cachedHistograms.get(host)) == null) {
                    return null;
                }
                long age = System.currentTimeMillis() - entry.timestamp;
                if (age < this.intervalMs) {
                    return (Histogram)entry.histogram.get();
                }
                recorder = (Recorder)this.recorders.get(host);
                staleHistogram = (Histogram)entry.histogram.get(0L, TimeUnit.MILLISECONDS);
            } while (!this.cachedHistograms.replace(host, entry, newEntry = new CachedHistogram(future = SettableFuture.create())));
            Histogram newHistogram = recorder.getIntervalHistogram(staleHistogram);
            future.set(newHistogram);
            return newHistogram;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            return null;
        }
        catch (ExecutionException e) {
            throw new DriverInternalError("Unexpected error", e.getCause());
        }
        catch (TimeoutException e) {
            throw new DriverInternalError("Unexpected timeout while getting histogram", e);
        }
    }

    private boolean shouldConsiderNewLatency(Statement statement, Exception exception) {
        if (exception == null) {
            return true;
        }
        return !EXCLUDED_EXCEPTIONS.contains(exception.getClass());
    }

    @Override
    public void onRegister(Cluster cluster) {
    }

    @Override
    public void onUnregister(Cluster cluster) {
    }

    static class CachedHistogram {
        final ListenableFuture<Histogram> histogram;
        final long timestamp;

        CachedHistogram(ListenableFuture<Histogram> histogram) {
            this.histogram = histogram;
            this.timestamp = System.currentTimeMillis();
        }

        static CachedHistogram empty() {
            return new CachedHistogram(Futures.immediateFuture(null));
        }
    }

    public static class Builder {
        private final long highestTrackableLatencyMillis;
        private int numberOfSignificantValueDigits = 3;
        private int minRecordedValues = 1000;
        private int numberOfHosts = 16;
        private long intervalMs = TimeUnit.MINUTES.toMillis(5L);

        Builder(long highestTrackableLatencyMillis) {
            this.highestTrackableLatencyMillis = highestTrackableLatencyMillis;
        }

        public Builder withNumberOfSignificantValueDigits(int numberOfSignificantValueDigits) {
            this.numberOfSignificantValueDigits = numberOfSignificantValueDigits;
            return this;
        }

        public Builder withMinRecordedValues(int minRecordedValues) {
            this.minRecordedValues = minRecordedValues;
            return this;
        }

        public Builder withNumberOfHosts(int numberOfHosts) {
            this.numberOfHosts = numberOfHosts;
            return this;
        }

        public Builder withInterval(long interval, TimeUnit unit) {
            this.intervalMs = TimeUnit.MILLISECONDS.convert(interval, unit);
            return this;
        }

        public PerHostPercentileTracker build() {
            return new PerHostPercentileTracker(this.highestTrackableLatencyMillis, this.numberOfSignificantValueDigits, this.numberOfHosts, this.minRecordedValues, this.intervalMs);
        }
    }
}

