/*
 * Decompiled with CFR 0.152.
 */
package org.junit.platform.reporting.legacy.xml;

import java.io.Writer;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.text.MessageFormat;
import java.text.NumberFormat;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.TreeSet;
import java.util.function.Function;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;
import org.junit.platform.commons.util.ExceptionUtils;
import org.junit.platform.commons.util.StringUtils;
import org.junit.platform.engine.TestExecutionResult;
import org.junit.platform.engine.reporting.ReportEntry;
import org.junit.platform.launcher.TestIdentifier;
import org.junit.platform.launcher.TestPlan;
import org.junit.platform.reporting.legacy.LegacyReportingUtils;
import org.junit.platform.reporting.legacy.xml.XmlReportData;

class XmlReportWriter {
    private static final Pattern CDATA_SPLIT_PATTERN = Pattern.compile("(?<=]])(?=>)");
    private final XmlReportData reportData;

    XmlReportWriter(XmlReportData reportData) {
        this.reportData = reportData;
    }

    void writeXmlReport(TestIdentifier rootDescriptor, Writer out) throws XMLStreamException {
        TestPlan testPlan = this.reportData.getTestPlan();
        Map<TestIdentifier, AggregatedTestResult> tests = testPlan.getDescendants(rootDescriptor).stream().filter(testIdentifier -> this.shouldInclude(testPlan, (TestIdentifier)testIdentifier)).collect(Collectors.toMap(Function.identity(), this::toAggregatedResult));
        this.writeXmlReport(rootDescriptor, tests, out);
    }

    private AggregatedTestResult toAggregatedResult(TestIdentifier testIdentifier) {
        if (this.reportData.wasSkipped(testIdentifier)) {
            return AggregatedTestResult.skipped();
        }
        return AggregatedTestResult.nonSkipped(this.reportData.getResults(testIdentifier));
    }

    private boolean shouldInclude(TestPlan testPlan, TestIdentifier testIdentifier) {
        return testIdentifier.isTest() || testPlan.getChildren(testIdentifier).isEmpty();
    }

    private void writeXmlReport(TestIdentifier testIdentifier, Map<TestIdentifier, AggregatedTestResult> tests, Writer out) throws XMLStreamException {
        XMLOutputFactory factory = XMLOutputFactory.newInstance();
        XMLStreamWriter xmlWriter = factory.createXMLStreamWriter(out);
        xmlWriter.writeStartDocument("UTF-8", "1.0");
        this.newLine(xmlWriter);
        this.writeTestsuite(testIdentifier, tests, xmlWriter);
        xmlWriter.writeEndDocument();
        xmlWriter.flush();
        xmlWriter.close();
    }

    private void writeTestsuite(TestIdentifier testIdentifier, Map<TestIdentifier, AggregatedTestResult> tests, XMLStreamWriter writer) throws XMLStreamException {
        NumberFormat numberFormat = NumberFormat.getInstance(Locale.US);
        writer.writeStartElement("testsuite");
        this.writeSuiteAttributes(testIdentifier, tests.values(), numberFormat, writer);
        this.newLine(writer);
        this.writeSystemProperties(writer);
        for (Map.Entry<TestIdentifier, AggregatedTestResult> entry : tests.entrySet()) {
            this.writeTestcase(entry.getKey(), entry.getValue(), numberFormat, writer);
        }
        this.writeOutputElement("system-out", this.formatNonStandardAttributesAsString(testIdentifier), writer);
        writer.writeEndElement();
        this.newLine(writer);
    }

    private void writeSuiteAttributes(TestIdentifier testIdentifier, Collection<AggregatedTestResult> testResults, NumberFormat numberFormat, XMLStreamWriter writer) throws XMLStreamException {
        this.writeAttributeSafely(writer, "name", testIdentifier.getDisplayName());
        this.writeTestCounts(testResults, writer);
        this.writeAttributeSafely(writer, "time", this.getTime(testIdentifier, numberFormat));
        this.writeAttributeSafely(writer, "hostname", this.getHostname().orElse("<unknown host>"));
        this.writeAttributeSafely(writer, "timestamp", DateTimeFormatter.ISO_LOCAL_DATE_TIME.format(this.getCurrentDateTime()));
    }

    private void writeTestCounts(Collection<AggregatedTestResult> testResults, XMLStreamWriter writer) throws XMLStreamException {
        Map counts = testResults.stream().map(it -> ((AggregatedTestResult)it).type).collect(Collectors.groupingBy(Function.identity(), Collectors.counting()));
        long total = counts.values().stream().mapToLong(Long::longValue).sum();
        this.writeAttributeSafely(writer, "tests", String.valueOf(total));
        this.writeAttributeSafely(writer, "skipped", counts.getOrDefault((Object)AggregatedTestResult.Type.SKIPPED, 0L).toString());
        this.writeAttributeSafely(writer, "failures", counts.getOrDefault((Object)AggregatedTestResult.Type.FAILURE, 0L).toString());
        this.writeAttributeSafely(writer, "errors", counts.getOrDefault((Object)AggregatedTestResult.Type.ERROR, 0L).toString());
    }

    private void writeSystemProperties(XMLStreamWriter writer) throws XMLStreamException {
        writer.writeStartElement("properties");
        this.newLine(writer);
        Properties systemProperties = System.getProperties();
        for (String propertyName : new TreeSet<String>(systemProperties.stringPropertyNames())) {
            writer.writeEmptyElement("property");
            this.writeAttributeSafely(writer, "name", propertyName);
            this.writeAttributeSafely(writer, "value", systemProperties.getProperty(propertyName));
            this.newLine(writer);
        }
        writer.writeEndElement();
        this.newLine(writer);
    }

    private void writeTestcase(TestIdentifier testIdentifier, AggregatedTestResult testResult, NumberFormat numberFormat, XMLStreamWriter writer) throws XMLStreamException {
        writer.writeStartElement("testcase");
        this.writeAttributeSafely(writer, "name", this.getName(testIdentifier));
        this.writeAttributeSafely(writer, "classname", this.getClassName(testIdentifier));
        this.writeAttributeSafely(writer, "time", this.getTime(testIdentifier, numberFormat));
        this.newLine(writer);
        this.writeSkippedOrErrorOrFailureElement(testIdentifier, testResult, writer);
        ArrayList<String> systemOutElements = new ArrayList<String>();
        ArrayList<String> systemErrElements = new ArrayList<String>();
        systemOutElements.add(this.formatNonStandardAttributesAsString(testIdentifier));
        this.collectReportEntries(testIdentifier, systemOutElements, systemErrElements);
        this.writeOutputElements("system-out", systemOutElements, writer);
        this.writeOutputElements("system-err", systemErrElements, writer);
        writer.writeEndElement();
        this.newLine(writer);
    }

    private String getName(TestIdentifier testIdentifier) {
        return testIdentifier.getLegacyReportingName();
    }

    private String getClassName(TestIdentifier testIdentifier) {
        return LegacyReportingUtils.getClassName(this.reportData.getTestPlan(), testIdentifier);
    }

    private void writeSkippedOrErrorOrFailureElement(TestIdentifier testIdentifier, AggregatedTestResult testResult, XMLStreamWriter writer) throws XMLStreamException {
        if (testResult.type == AggregatedTestResult.Type.SKIPPED) {
            this.writeSkippedElement(this.reportData.getSkipReason(testIdentifier), writer);
        } else {
            Map<AggregatedTestResult.Type, List<Optional<Throwable>>> throwablesByType = testResult.getThrowablesByType();
            for (AggregatedTestResult.Type type : EnumSet.of(AggregatedTestResult.Type.FAILURE, AggregatedTestResult.Type.ERROR)) {
                for (Optional throwable : throwablesByType.getOrDefault((Object)type, Collections.emptyList())) {
                    this.writeErrorOrFailureElement(type, throwable.orElse(null), writer);
                }
            }
        }
    }

    private void writeSkippedElement(String reason, XMLStreamWriter writer) throws XMLStreamException {
        if (StringUtils.isNotBlank((String)reason)) {
            writer.writeStartElement("skipped");
            this.writeCDataSafely(writer, reason);
            writer.writeEndElement();
        } else {
            writer.writeEmptyElement("skipped");
        }
        this.newLine(writer);
    }

    private void writeErrorOrFailureElement(AggregatedTestResult.Type type, Throwable throwable, XMLStreamWriter writer) throws XMLStreamException {
        String elementName;
        String string = elementName = type == AggregatedTestResult.Type.FAILURE ? "failure" : "error";
        if (throwable != null) {
            writer.writeStartElement(elementName);
            this.writeFailureAttributesAndContent(throwable, writer);
            writer.writeEndElement();
        } else {
            writer.writeEmptyElement(elementName);
        }
        this.newLine(writer);
    }

    private void writeFailureAttributesAndContent(Throwable throwable, XMLStreamWriter writer) throws XMLStreamException {
        if (throwable.getMessage() != null) {
            this.writeAttributeSafely(writer, "message", throwable.getMessage());
        }
        this.writeAttributeSafely(writer, "type", throwable.getClass().getName());
        this.writeCDataSafely(writer, ExceptionUtils.readStackTrace((Throwable)throwable));
    }

    private void collectReportEntries(TestIdentifier testIdentifier, List<String> systemOutElements, List<String> systemErrElements) {
        List<ReportEntry> entries = this.reportData.getReportEntries(testIdentifier);
        if (!entries.isEmpty()) {
            ArrayList<String> systemOutElementsForCapturedOutput = new ArrayList<String>();
            StringBuilder formattedReportEntries = new StringBuilder();
            for (int i = 0; i < entries.size(); ++i) {
                ReportEntry reportEntry = entries.get(i);
                LinkedHashMap<String, String> keyValuePairs = new LinkedHashMap<String, String>(reportEntry.getKeyValuePairs());
                this.removeIfPresentAndAddAsSeparateElement(keyValuePairs, "stdout", systemOutElementsForCapturedOutput);
                this.removeIfPresentAndAddAsSeparateElement(keyValuePairs, "stderr", systemErrElements);
                if (keyValuePairs.isEmpty()) continue;
                this.buildReportEntryDescription(reportEntry.getTimestamp(), keyValuePairs, i + 1, formattedReportEntries);
            }
            systemOutElements.add(formattedReportEntries.toString().trim());
            systemOutElements.addAll(systemOutElementsForCapturedOutput);
        }
    }

    private void removeIfPresentAndAddAsSeparateElement(Map<String, String> keyValuePairs, String key, List<String> elements) {
        String value = keyValuePairs.remove(key);
        if (value != null) {
            elements.add(value);
        }
    }

    private void buildReportEntryDescription(LocalDateTime timestamp, Map<String, String> keyValuePairs, int entryNumber, StringBuilder result) {
        result.append(MessageFormat.format("Report Entry #{0} (timestamp: {1})\n", entryNumber, DateTimeFormatter.ISO_LOCAL_DATE_TIME.format(timestamp)));
        keyValuePairs.forEach((key, value) -> result.append(MessageFormat.format("\t- {0}: {1}\n", key, value)));
    }

    private String getTime(TestIdentifier testIdentifier, NumberFormat numberFormat) {
        return numberFormat.format(this.reportData.getDurationInSeconds(testIdentifier));
    }

    private Optional<String> getHostname() {
        try {
            return Optional.ofNullable(InetAddress.getLocalHost().getHostName());
        }
        catch (UnknownHostException e) {
            return Optional.empty();
        }
    }

    private LocalDateTime getCurrentDateTime() {
        return LocalDateTime.now(this.reportData.getClock()).withNano(0);
    }

    private String formatNonStandardAttributesAsString(TestIdentifier testIdentifier) {
        return "unique-id: " + testIdentifier.getUniqueId() + "\ndisplay-name: " + testIdentifier.getDisplayName();
    }

    private void writeOutputElements(String elementName, List<String> elements, XMLStreamWriter writer) throws XMLStreamException {
        for (String content : elements) {
            this.writeOutputElement(elementName, content, writer);
        }
    }

    private void writeOutputElement(String elementName, String content, XMLStreamWriter writer) throws XMLStreamException {
        writer.writeStartElement(elementName);
        this.writeCDataSafely(writer, "\n" + content + "\n");
        writer.writeEndElement();
        this.newLine(writer);
    }

    private void writeAttributeSafely(XMLStreamWriter writer, String name, String value) throws XMLStreamException {
        writer.writeAttribute(name, XmlReportWriter.escapeIllegalChars(value));
    }

    private void writeCDataSafely(XMLStreamWriter writer, String data) throws XMLStreamException {
        for (String safeDataPart : CDATA_SPLIT_PATTERN.split(XmlReportWriter.escapeIllegalChars(data))) {
            writer.writeCData(safeDataPart);
        }
    }

    static String escapeIllegalChars(String text) {
        if (text.codePoints().allMatch(XmlReportWriter::isAllowedXmlCharacter)) {
            return text;
        }
        StringBuilder result = new StringBuilder(text.length() * 2);
        text.codePoints().forEach(codePoint -> {
            if (XmlReportWriter.isAllowedXmlCharacter(codePoint)) {
                result.appendCodePoint(codePoint);
            } else {
                result.append("&#").append(codePoint).append(';');
            }
        });
        return result.toString();
    }

    private static boolean isAllowedXmlCharacter(int codePoint) {
        return codePoint == 9 || codePoint == 10 || codePoint == 13 || codePoint >= 32 && codePoint <= 55295 || codePoint >= 57344 && codePoint <= 65533 || codePoint >= 65536 && codePoint <= 0x10FFFF;
    }

    private void newLine(XMLStreamWriter xmlWriter) throws XMLStreamException {
        xmlWriter.writeCharacters("\n");
    }

    private static boolean isFailure(TestExecutionResult result) {
        Optional throwable = result.getThrowable();
        return throwable.isPresent() && throwable.get() instanceof AssertionError;
    }

    static class AggregatedTestResult {
        private static final AggregatedTestResult SKIPPED_RESULT = new AggregatedTestResult(Type.SKIPPED, Collections.emptyList());
        private final Type type;
        private final List<TestExecutionResult> executionResults;

        public static AggregatedTestResult skipped() {
            return SKIPPED_RESULT;
        }

        public static AggregatedTestResult nonSkipped(List<TestExecutionResult> executionResults) {
            Type type = executionResults.stream().map(x$0 -> Type.from(x$0)).max(Comparator.naturalOrder()).orElse(Type.SUCCESS);
            return new AggregatedTestResult(type, executionResults);
        }

        private AggregatedTestResult(Type type, List<TestExecutionResult> executionResults) {
            this.type = type;
            this.executionResults = executionResults;
        }

        public Map<Type, List<Optional<Throwable>>> getThrowablesByType() {
            return this.executionResults.stream().collect(Collectors.groupingBy(x$0 -> Type.from(x$0), Collectors.mapping(TestExecutionResult::getThrowable, Collectors.toList())));
        }

        static enum Type {
            SUCCESS,
            SKIPPED,
            FAILURE,
            ERROR;


            private static Type from(TestExecutionResult executionResult) {
                if (executionResult.getStatus() == TestExecutionResult.Status.FAILED) {
                    return XmlReportWriter.isFailure(executionResult) ? FAILURE : ERROR;
                }
                return SUCCESS;
            }
        }
    }
}

