/*
 * Decompiled with CFR 0.152.
 */
package reactor.core;

import java.util.UUID;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.event.registry.CachingRegistry;
import reactor.event.registry.Registration;
import reactor.event.registry.Registry;
import reactor.event.selector.HeaderResolver;
import reactor.event.selector.Selector;
import reactor.function.Consumer;
import reactor.function.support.CancelConsumerException;
import reactor.function.support.SingleUseConsumer;
import reactor.support.NamedDaemonThreadFactory;
import reactor.util.Assert;
import reactor.util.UUIDUtils;

public class HashWheelTimer {
    private static final Logger LOG = LoggerFactory.getLogger(HashWheelTimer.class);
    private final Registry<Consumer<Long>> tasks = new CachingRegistry<Consumer<Long>>(false);
    private final int resolution;
    private final Thread loop;

    public HashWheelTimer() {
        this(50);
    }

    public HashWheelTimer(final int resolution) {
        this.resolution = resolution;
        this.loop = new NamedDaemonThreadFactory("hash-wheel-timer").newThread(new Runnable(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void run() {
                while (!Thread.currentThread().isInterrupted()) {
                    long now = HashWheelTimer.now(resolution);
                    for (Registration reg : HashWheelTimer.this.tasks.select(now)) {
                        try {
                            if (reg.isCancelled() || reg.isPaused()) continue;
                            ((Consumer)reg.getObject()).accept(now);
                        }
                        catch (CancelConsumerException cce) {
                            reg.cancel();
                        }
                        catch (Throwable t) {
                            LOG.error(t.getMessage(), t);
                        }
                        finally {
                            if (!reg.isCancelAfterUse()) continue;
                            reg.cancel();
                        }
                    }
                    try {
                        Thread.sleep(resolution);
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                    }
                }
            }
        });
        this.loop.start();
    }

    public Registration<? extends Consumer<Long>> schedule(Consumer<Long> consumer, long period, TimeUnit timeUnit, long delayInMilliseconds) {
        Assert.isTrue(!this.loop.isInterrupted(), "Cannot submit tasks to this timer as it has been cancelled.");
        return this.tasks.register(new PeriodSelector(TimeUnit.MILLISECONDS.convert(period, timeUnit), delayInMilliseconds, this.resolution), consumer);
    }

    public Registration<? extends Consumer<Long>> schedule(Consumer<Long> consumer, long period, TimeUnit timeUnit) {
        return this.schedule(consumer, period, timeUnit, 0L);
    }

    public Registration<? extends Consumer<Long>> submit(Consumer<Long> consumer, long delay, TimeUnit timeUnit) {
        Assert.isTrue(!this.loop.isInterrupted(), "Cannot submit tasks to this timer as it has been cancelled.");
        long ms = TimeUnit.MILLISECONDS.convert(delay, timeUnit);
        return this.tasks.register(new PeriodSelector(ms, ms, this.resolution), new SingleUseConsumer<Long>(consumer)).cancelAfterUse();
    }

    public HashWheelTimer submit(Consumer<Long> consumer) {
        this.submit(consumer, this.resolution, TimeUnit.MILLISECONDS);
        return this;
    }

    public void cancel() {
        this.loop.interrupt();
    }

    private static long now(int resolution) {
        return (long)(Math.ceil(System.currentTimeMillis() / (long)resolution) * (double)resolution);
    }

    private static class PeriodSelector
    implements Selector {
        private final UUID uuid = UUIDUtils.create();
        private final long period;
        private final long delay;
        private final long createdMillis;
        private final int resolution;

        private PeriodSelector(long period, long delay, int resolution) {
            this.period = period;
            this.delay = delay;
            this.resolution = resolution;
            this.createdMillis = HashWheelTimer.now(resolution);
        }

        @Override
        public UUID getId() {
            return this.uuid;
        }

        @Override
        public Object getObject() {
            return this.period;
        }

        @Override
        public boolean matches(Object key) {
            long now = (Long)key;
            long period = (long)(Math.ceil((now - this.createdMillis) / (long)this.resolution) * (double)this.resolution);
            return period >= this.delay && period % this.period == 0L;
        }

        @Override
        public HeaderResolver getHeaderResolver() {
            return null;
        }
    }
}

