/*
 * Decompiled with CFR 0.152.
 */
package reactor.io;

import java.io.IOException;
import java.io.InputStream;
import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.CoderResult;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import javax.annotation.concurrent.NotThreadSafe;
import reactor.function.Supplier;
import reactor.util.Assert;

@NotThreadSafe
public class Buffer
implements Comparable<Buffer>,
Iterable<Byte>,
ReadableByteChannel,
WritableByteChannel {
    public static int SMALL_BUFFER_SIZE = Integer.parseInt(System.getProperty("reactor.io.defaultBufferSize", "16384"));
    public static int MAX_BUFFER_SIZE = Integer.parseInt(System.getProperty("reactor.io.maxBufferSize", "16384000"));
    private static final Charset UTF8 = Charset.forName("UTF-8");
    private final boolean dynamic;
    private ByteBuffer buffer;
    private CharsetDecoder decoder;
    private CharBuffer chars;
    private int position;
    private int limit;

    public Buffer() {
        this.dynamic = true;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public Buffer(int atLeast, boolean fixed) {
        if (fixed) {
            if (atLeast > MAX_BUFFER_SIZE) throw new IllegalArgumentException("Requested buffer size exceeds maximum allowed (" + MAX_BUFFER_SIZE + ")");
            this.buffer = ByteBuffer.allocate(atLeast);
        } else {
            this.ensureCapacity(atLeast);
        }
        this.dynamic = !fixed;
    }

    public Buffer(Buffer bufferToCopy) {
        this.dynamic = bufferToCopy.dynamic;
        this.buffer = bufferToCopy.buffer.duplicate();
    }

    public Buffer(ByteBuffer bufferToStartWith) {
        this.dynamic = true;
        this.buffer = bufferToStartWith;
    }

    public static Buffer wrap(byte[] bytes) {
        return new Buffer(bytes.length, true).append(bytes).flip();
    }

    public static Buffer wrap(String str, boolean fixed) {
        return new Buffer(str.length(), fixed).append(str).flip();
    }

    public static Buffer wrap(String str) {
        return Buffer.wrap(str, true);
    }

    public static Integer parseInt(Buffer b, int start, int end) {
        b.snapshot();
        b.buffer.limit(end);
        b.buffer.position(start);
        Integer i = Buffer.parseInt(b);
        b.reset();
        return i;
    }

    public static Integer parseInt(Buffer b) {
        if (b.remaining() == 0) {
            return null;
        }
        b.snapshot();
        int len = b.remaining();
        int num = 0;
        int dec = 1;
        int i = b.position + len;
        while (i > b.position) {
            char c = (char)b.buffer.get(--i);
            num += Character.getNumericValue(c) * dec;
            dec *= 10;
        }
        b.reset();
        return num;
    }

    public static Long parseLong(Buffer b, int start, int end) {
        int origPos = b.buffer.position();
        int origLimit = b.buffer.limit();
        b.buffer.position(start);
        b.buffer.limit(end);
        Long l = Buffer.parseLong(b);
        b.buffer.position(origPos);
        b.buffer.limit(origLimit);
        return l;
    }

    public static Long parseLong(Buffer b) {
        if (b.remaining() == 0) {
            return null;
        }
        ByteBuffer bb = b.buffer;
        int origPos = bb.position();
        int len = bb.remaining();
        long num = 0L;
        int dec = 1;
        int i = len;
        while (i > 0) {
            char c = (char)bb.get(--i);
            num += (long)(Character.getNumericValue(c) * dec);
            dec *= 10;
        }
        bb.position(origPos);
        return num;
    }

    public boolean isDynamic() {
        return this.dynamic;
    }

    public int position() {
        return null == this.buffer ? 0 : this.buffer.position();
    }

    public Buffer position(int pos) {
        if (null != this.buffer) {
            this.buffer.position(pos);
        }
        return this;
    }

    public Buffer limit(int limit) {
        if (null != this.buffer) {
            this.buffer.limit(limit);
        }
        return this;
    }

    public Buffer skip(int len) {
        if (len < 0) {
            throw new IllegalArgumentException("len must >= 0");
        }
        if (null != this.buffer) {
            int pos = this.buffer.position();
            this.buffer.position(pos + len);
        }
        return this;
    }

    public int limit() {
        return null == this.buffer ? 0 : this.buffer.limit();
    }

    public int capacity() {
        return null == this.buffer ? SMALL_BUFFER_SIZE : this.buffer.capacity();
    }

    public int remaining() {
        return null == this.buffer ? SMALL_BUFFER_SIZE : this.buffer.remaining();
    }

    public Buffer clear() {
        if (null != this.buffer) {
            this.buffer.position(0);
            this.buffer.limit(this.buffer.capacity());
        }
        return this;
    }

    public Buffer compact() {
        if (null != this.buffer) {
            this.buffer.compact();
        }
        return this;
    }

    public Buffer flip() {
        if (null != this.buffer) {
            this.buffer.flip();
        }
        return this;
    }

    public Buffer rewind() {
        if (null != this.buffer) {
            this.buffer.rewind();
        }
        return this;
    }

    public Buffer rewind(int len) {
        if (len < 0) {
            throw new IllegalArgumentException("len must >= 0");
        }
        if (null != this.buffer) {
            int pos = this.buffer.position();
            this.buffer.position(pos - len);
        }
        return this;
    }

    public Buffer duplicate() {
        return new Buffer(this.buffer.duplicate());
    }

    public Buffer copy() {
        this.snapshot();
        Buffer b = new Buffer(this.buffer.remaining(), false);
        b.append(this.buffer);
        this.reset();
        return b.flip();
    }

    public Buffer prepend(Buffer b) {
        if (null == b) {
            return this;
        }
        return this.prepend(b.buffer);
    }

    public Buffer prepend(String s) {
        if (null == s) {
            return this;
        }
        return this.prepend(s.getBytes());
    }

    public Buffer prepend(byte[] bytes) {
        this.shift(bytes.length);
        this.buffer.put(bytes);
        this.reset();
        return this;
    }

    public Buffer prepend(ByteBuffer b) {
        if (null == b) {
            return this;
        }
        this.shift(b.remaining());
        this.buffer.put(b);
        this.reset();
        return this;
    }

    public Buffer prepend(byte b) {
        this.shift(1);
        this.buffer.put(b);
        this.reset();
        return this;
    }

    public Buffer prepend(char c) {
        this.shift(2);
        this.buffer.putChar(c);
        this.reset();
        return this;
    }

    public Buffer prepend(short s) {
        this.shift(2);
        this.buffer.putShort(s);
        this.reset();
        return this;
    }

    public Buffer prepend(int i) {
        this.shift(4);
        this.buffer.putInt(i);
        this.reset();
        return this;
    }

    public Buffer prepend(long l) {
        this.shift(8);
        this.buffer.putLong(l);
        this.reset();
        return this;
    }

    public Buffer append(String s) {
        this.ensureCapacity(s.length());
        this.buffer.put(s.getBytes());
        return this;
    }

    public Buffer append(short s) {
        this.ensureCapacity(2);
        this.buffer.putInt(s);
        return this;
    }

    public Buffer append(int i) {
        this.ensureCapacity(4);
        this.buffer.putInt(i);
        return this;
    }

    public Buffer append(long l) {
        this.ensureCapacity(8);
        this.buffer.putLong(l);
        return this;
    }

    public Buffer append(char c) {
        this.ensureCapacity(2);
        this.buffer.putChar(c);
        return this;
    }

    public Buffer append(ByteBuffer ... buffers) {
        for (ByteBuffer bb : buffers) {
            this.ensureCapacity(bb.remaining());
            this.buffer.put(bb);
        }
        return this;
    }

    public Buffer append(Buffer ... buffers) {
        for (Buffer b : buffers) {
            int pos = null == this.buffer ? 0 : this.buffer.position();
            int len = b.remaining();
            this.ensureCapacity(len);
            this.buffer.put(b.byteBuffer());
            this.buffer.position(pos + len);
        }
        return this;
    }

    public Buffer append(byte b) {
        this.ensureCapacity(1);
        this.buffer.put(b);
        return this;
    }

    public Buffer append(byte[] b) {
        this.ensureCapacity(b.length);
        this.buffer.put(b);
        return this;
    }

    public byte first() {
        this.snapshot();
        if (this.position > 0) {
            this.buffer.position(0);
        }
        byte b = this.buffer.get();
        this.reset();
        return b;
    }

    public byte last() {
        int pos = this.buffer.position();
        int limit = this.buffer.limit();
        this.buffer.position(limit - 1);
        byte b = this.buffer.get();
        this.buffer.position(pos);
        return b;
    }

    public byte read() {
        if (null != this.buffer) {
            return this.buffer.get();
        }
        throw new BufferUnderflowException();
    }

    public Buffer read(byte[] b) {
        if (null != this.buffer) {
            this.buffer.get(b);
        }
        return this;
    }

    public short readShort() {
        if (null != this.buffer) {
            return this.buffer.getShort();
        }
        throw new BufferUnderflowException();
    }

    public int readInt() {
        if (null != this.buffer) {
            return this.buffer.getInt();
        }
        throw new BufferUnderflowException();
    }

    public float readFloat() {
        if (null != this.buffer) {
            return this.buffer.getFloat();
        }
        throw new BufferUnderflowException();
    }

    public double readDouble() {
        if (null != this.buffer) {
            return this.buffer.getDouble();
        }
        throw new BufferUnderflowException();
    }

    public long readLong() {
        if (null != this.buffer) {
            return this.buffer.getLong();
        }
        throw new BufferUnderflowException();
    }

    public char readChar() {
        if (null != this.buffer) {
            return this.buffer.getChar();
        }
        throw new BufferUnderflowException();
    }

    public void snapshot() {
        this.position = this.buffer.position();
        this.limit = this.buffer.limit();
    }

    public Buffer reset() {
        this.buffer.limit(this.limit);
        this.buffer.position(this.position);
        return this;
    }

    @Override
    public Iterator<Byte> iterator() {
        return new Iterator<Byte>(){

            @Override
            public boolean hasNext() {
                return Buffer.this.buffer.remaining() > 0;
            }

            @Override
            public Byte next() {
                return Buffer.this.buffer.get();
            }

            @Override
            public void remove() {
            }
        };
    }

    @Override
    public int read(ByteBuffer dst) throws IOException {
        this.snapshot();
        if (dst.remaining() < this.limit) {
            this.buffer.limit(dst.remaining());
        }
        int pos = dst.position();
        dst.put(this.buffer);
        this.buffer.limit(this.limit);
        return dst.position() - pos;
    }

    @Override
    public int write(ByteBuffer src) throws IOException {
        int pos = src.position();
        this.append(src);
        return src.position() - pos;
    }

    @Override
    public boolean isOpen() {
        return this.isDynamic();
    }

    @Override
    public void close() throws IOException {
        this.clear();
    }

    public String asString() {
        if (null != this.buffer) {
            return this.decode();
        }
        return null;
    }

    public String substring(int start, int end) {
        this.snapshot();
        this.buffer.limit(end > start ? end : this.limit);
        this.buffer.position(start);
        String s = this.asString();
        this.reset();
        return s;
    }

    public byte[] asBytes() {
        if (null != this.buffer) {
            this.snapshot();
            byte[] b = new byte[this.buffer.remaining()];
            this.buffer.get(b);
            this.reset();
            return b;
        }
        return null;
    }

    public InputStream inputStream() {
        return new BufferInputStream();
    }

    public Buffer slice(int start, int len) {
        this.snapshot();
        ByteBuffer bb = ByteBuffer.allocate(len);
        this.buffer.position(start);
        bb.put(this.buffer);
        this.reset();
        bb.flip();
        return new Buffer(bb);
    }

    public Iterable<View> split(int delimiter) {
        return this.split(new ArrayList<View>(), delimiter, false);
    }

    public Iterable<View> split(List<View> views, int delimiter) {
        return this.split(views, delimiter, false);
    }

    public Iterable<View> split(int delimiter, boolean stripDelimiter) {
        return this.split(new ArrayList<View>(), delimiter, stripDelimiter);
    }

    public Iterable<View> split(List<View> views, int delimiter, boolean stripDelimiter) {
        this.snapshot();
        int start = this.position;
        for (byte b : this) {
            if (b != delimiter) continue;
            int end = stripDelimiter ? this.buffer.position() - 1 : this.buffer.position();
            views.add(this.createView(start, end));
            start = end + (stripDelimiter ? 1 : 0);
        }
        if (start != this.buffer.position()) {
            this.buffer.position(start);
        }
        this.reset();
        return views;
    }

    public Iterable<View> split(Buffer delimiter) {
        return this.split(new ArrayList<View>(), delimiter, false);
    }

    public Iterable<View> split(Buffer delimiter, boolean stripDelimiter) {
        return this.split(new ArrayList<View>(), delimiter, stripDelimiter);
    }

    public Iterable<View> split(List<View> views, Buffer delimiter, boolean stripDelimiter) {
        this.snapshot();
        byte[] delimBytes = delimiter.asBytes();
        if (delimBytes.length == 0) {
            return Collections.emptyList();
        }
        int start = this.position;
        for (byte b : this) {
            if (b != delimBytes[0]) continue;
            int end = -1;
            for (int i = 1; i < delimBytes.length; ++i) {
                if (this.read() != delimBytes[i]) {
                    end = -1;
                    break;
                }
                end = stripDelimiter ? this.buffer.position() - delimBytes.length : this.buffer.position();
            }
            if (end <= 0) continue;
            views.add(this.createView(start, end));
            start = end + (stripDelimiter ? delimBytes.length : 0);
        }
        if (start != this.buffer.position()) {
            this.buffer.position(start);
        }
        this.reset();
        return views;
    }

    public int indexOf(byte b) {
        return this.indexOf(b, this.buffer.position(), this.buffer.remaining());
    }

    public int indexOf(byte b, int start, int end) {
        this.snapshot();
        if (this.buffer.position() != start) {
            this.buffer.position(start);
        }
        int pos = -1;
        while (this.buffer.hasRemaining() && this.buffer.position() < end) {
            if (this.buffer.get() != b) continue;
            pos = this.buffer.position();
            break;
        }
        this.reset();
        return pos;
    }

    public View createView() {
        this.snapshot();
        return new View(this.position, this.limit);
    }

    public View createView(int start, int end) {
        this.snapshot();
        return new View(start, end);
    }

    public List<View> slice(int ... positions) {
        Assert.notNull(positions, "Positions cannot be null.");
        if (positions.length == 0) {
            return Collections.emptyList();
        }
        this.snapshot();
        ArrayList<View> views = new ArrayList<View>();
        int len = positions.length;
        for (int i = 0; i < len; ++i) {
            int start = positions[i];
            int end = i + 1 < len ? positions[++i] : this.limit;
            views.add(this.createView(start, end));
            this.reset();
        }
        return views;
    }

    public ByteBuffer byteBuffer() {
        return this.buffer;
    }

    public String toString() {
        return null != this.buffer ? this.buffer.toString() : "<EMPTY>";
    }

    @Override
    public int compareTo(Buffer buffer) {
        return null != buffer ? this.buffer.compareTo(buffer.buffer) : -1;
    }

    private synchronized void ensureCapacity(int atLeast) {
        if (null == this.buffer) {
            this.buffer = ByteBuffer.allocate(SMALL_BUFFER_SIZE);
            return;
        }
        int pos = this.buffer.position();
        int cap = this.buffer.capacity();
        if (this.dynamic && this.buffer.remaining() < atLeast) {
            if (this.buffer.limit() < cap) {
                if (pos + atLeast > cap) {
                    this.expand();
                    cap = this.buffer.capacity();
                }
                this.buffer.limit(Math.min(pos + atLeast, cap));
            } else {
                this.expand();
            }
        } else if (pos + SMALL_BUFFER_SIZE > MAX_BUFFER_SIZE) {
            throw new BufferOverflowException();
        }
    }

    private void expand() {
        this.snapshot();
        ByteBuffer newBuff = ByteBuffer.allocate(this.buffer.limit() + SMALL_BUFFER_SIZE);
        this.buffer.flip();
        newBuff.put(this.buffer);
        this.buffer = newBuff;
        this.reset();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private String decode() {
        if (null == this.decoder) {
            this.decoder = UTF8.newDecoder();
        }
        this.snapshot();
        try {
            if (null == this.chars || this.chars.remaining() < this.buffer.remaining()) {
                this.chars = CharBuffer.allocate(this.buffer.remaining());
            } else {
                this.chars.rewind();
            }
            this.decoder.reset();
            CoderResult cr = this.decoder.decode(this.buffer, this.chars, true);
            if (cr.isUnderflow()) {
                this.decoder.flush(this.chars);
            }
            this.chars.flip();
            String string = this.chars.toString();
            return string;
        }
        finally {
            this.reset();
        }
    }

    private void shift(int right) {
        ByteBuffer currentBuffer;
        if (null == this.buffer) {
            this.ensureCapacity(right);
            currentBuffer = this.buffer;
        } else {
            currentBuffer = this.buffer.slice();
        }
        int len = this.buffer.remaining();
        int pos = this.buffer.position();
        this.ensureCapacity(right + len);
        this.buffer.position(pos + right);
        this.buffer.put(currentBuffer);
        this.buffer.position(pos);
        this.snapshot();
    }

    public class View
    implements Supplier<Buffer> {
        private final int start;
        private final int end;

        private View(int start, int end) {
            this.start = start;
            this.end = end;
        }

        public int getStart() {
            return this.start;
        }

        public int getEnd() {
            return this.end;
        }

        @Override
        public Buffer get() {
            Buffer.this.buffer.limit(this.end);
            Buffer.this.buffer.position(this.start);
            return Buffer.this;
        }
    }

    private class BufferInputStream
    extends InputStream {
        ByteBuffer buffer;

        private BufferInputStream() {
            this.buffer = Buffer.this.buffer.slice();
        }

        @Override
        public int read(byte[] b) throws IOException {
            int pos = this.buffer.position();
            this.buffer.get(b);
            this.syncPos();
            return this.buffer.position() - pos;
        }

        @Override
        public int read(byte[] b, int off, int len) throws IOException {
            if (null == this.buffer || this.buffer.remaining() == 0) {
                return -1;
            }
            byte[] bytes = Buffer.this.asBytes();
            int bytesLen = bytes.length;
            System.arraycopy(bytes, 0, b, off, bytesLen);
            if (len < bytesLen) {
                this.buffer.position(Buffer.this.position + len);
            }
            this.syncPos();
            return bytesLen;
        }

        @Override
        public long skip(long n) throws IOException {
            if (n < (long)this.buffer.remaining()) {
                throw new IOException(new BufferUnderflowException());
            }
            int pos = this.buffer.position();
            this.buffer.position((int)((long)pos + n));
            this.syncPos();
            return this.buffer.position() - pos;
        }

        @Override
        public int available() throws IOException {
            return this.buffer.remaining();
        }

        @Override
        public void close() throws IOException {
            this.buffer.position(this.buffer.limit());
            this.syncPos();
        }

        @Override
        public synchronized void mark(int readlimit) {
            this.buffer.mark();
            int pos = this.buffer.position();
            int max = this.buffer.capacity() - pos;
            int newLimit = Math.min(max, pos + readlimit);
            this.buffer.limit(newLimit);
        }

        @Override
        public synchronized void reset() throws IOException {
            this.buffer.reset();
            this.syncPos();
        }

        @Override
        public boolean markSupported() {
            return true;
        }

        @Override
        public int read() throws IOException {
            byte b = this.buffer.get();
            this.syncPos();
            return b;
        }

        private void syncPos() {
            Buffer.this.buffer.position(this.buffer.position());
        }
    }
}

