package com.easesource.edge.northbound.device.spitesam.util;

import com.easesource.edge.core.common.util.convert.JsonConvertUtils;
import com.easesource.edge.periphery.Spitesam;
import com.easesource.edge.southbound.model.tesam.base.TesamMethodEnum;
import com.easesource.edge.southbound.model.tesam.base.TesamRxResponse;
import com.easesource.edge.southbound.model.tesam.base.TesamTxRequest;
import com.google.common.collect.Maps;
import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.LinkedHashMap;

public final class SpitesamOperationUtils {

    private static final Logger logger = LoggerFactory.getLogger(SpitesamOperationUtils.class);
    private static final String DEFAULT_DEVICES = "/dev/spidev2.0";
    private static final int DEFAULT_MODE = 3;
    private static final int DEFAULT_SPEED = 25000;

    public static byte[] transmit(String device, byte[] txBuf, int rxLength) {
        device = device == null ? DEFAULT_DEVICES : device;
        return transmit(device, DEFAULT_MODE, DEFAULT_SPEED, txBuf, rxLength);
    }

    public static byte[] transmit(String device, int mode, int speed, byte[] txBuf, int rxLength) {
        String str = Hex.encodeHexString(txBuf);
        try(final Spitesam spitesam = new Spitesam(device, mode, speed)) {
            if(txBuf != null && txBuf.length > 0) {
                logger.info("tx data    : " + Hex.encodeHexString(txBuf));
                final byte[] rxBuf = new byte[rxLength];
                Spitesam.spitesamTransfer(spitesam.getHandle(), txBuf, rxBuf, txBuf.length);
                logger.info("rx data    : " + Hex.encodeHexString(rxBuf));
                return rxBuf;
            }
            else {
                logger.error("txBuf is empty.");
            }
        }
        catch(RuntimeException e) {
            logger.error(e.getMessage(), e.fillInStackTrace());
        }
        return null;
    }

    public static byte calcXorAndNot(byte[] datas) {
        byte lrc = 0;
        for(byte b : datas) {
            lrc = (byte) (lrc ^ b);
        }
        return (byte) ~lrc;
    }

    public static String getRxDataFollowLrc(String rxDataWithoutLrc) {
        try {
            byte[] datas = Hex.decodeHex(rxDataWithoutLrc);
            byte lrc = calcXorAndNot(datas);
            byte[] lrcs = new byte[1];
            lrcs[0] = lrc;
            byte[] newBytes = new byte[datas.length + 1];
            System.arraycopy(datas, 0, newBytes, 0, datas.length);
            System.arraycopy(lrcs, 0, newBytes, datas.length, lrcs.length);
            return Hex.encodeHexString(newBytes, false);
        }
        catch(DecoderException e) {
            logger.error(e.getMessage(), e.fillInStackTrace());
        }
        return rxDataWithoutLrc;
    }

    public static TesamRxResponse request(TesamTxRequest tesamTxRequest) {
        return request(tesamTxRequest, null);
    }

    public static TesamRxResponse request(TesamTxRequest tesamTxRequest, String device) {

        if(logger.isInfoEnabled()) {
            logger.info(" TesamTxRequest    : " + JsonConvertUtils.convertToString(tesamTxRequest));
        }

        TesamRxResponse tesamRxResponse = new TesamRxResponse();
        if(tesamTxRequest == null) {
            logger.error(" TesamTxRequest is null. ");
            return null;
        }

        String id = tesamTxRequest.getId();
        String method = tesamTxRequest.getMethod();
        String type = tesamTxRequest.getType();
        LinkedHashMap<String, String> params = tesamTxRequest.getParams();
        LinkedHashMap<String, String> datas = tesamTxRequest.getDatas();

        tesamRxResponse.setId(id);
        tesamRxResponse.setMethod(method);

        String payload = "55803600FF0000B6";
        int length = 128;
        int resDataLength = 0;
        int[] resDataLenDivided = {};
        LinkedHashMap<String, String> resDataMap = Maps.newLinkedHashMap();
        if(StringUtils.equals(method, TesamMethodEnum.GET_UC_SIGN.getCode())) {
            // 获取服务器签名信息（建立应用连接）
            StringBuilder sbData = new StringBuilder();
            String dataStart = "80020000";
            sbData.append(dataStart);
            int lc = 0;
            String p2 = "";
            String s2 = "";
            if(datas != null && StringUtils.isNotBlank(datas.get("p2")) && StringUtils.isNotBlank(datas.get("s2"))) {
                p2 = datas.get("p2");
                s2 = datas.get("s2");
            }
            else if(datas != null && StringUtils.isNotBlank(datas.get("d1")) && StringUtils.isNotBlank(datas.get("d2"))) {
                p2 = datas.get("d1");
                s2 = datas.get("d2");
            }
            try {
                lc = Hex.decodeHex(p2).length + Hex.decodeHex(s2).length;
            }
            catch(DecoderException e) {
                logger.error(e.getMessage(), e.fillInStackTrace());
            }
            sbData.append(StringUtils.leftPad(Integer.toHexString(lc), 4, '0'));
            sbData.append(p2);
            sbData.append(s2);
            payload = "55" + getRxDataFollowLrc(sbData.toString());
            length = 128;
            resDataLength = 112;
            resDataLenDivided = new int[] { 48, 64 };
        }
        else if(StringUtils.equals(method, TesamMethodEnum.HANDLE_SECTITY_DATA_DOWN.getCode())) {
            // 安全传输数据处理（主站到终端）
            StringBuilder sbData = new StringBuilder();
            if(StringUtils.equals(type, "step4")) {
                // 5.3 安全传输数据处理（主站到终端） 第4步
                if(params != null && StringUtils.isNotBlank(params.get("sid.indi"))
                    && StringUtils.isNotBlank(params.get("sid.fdata"))) {
                    String sidIndi = params.get("sid.indi");
                    String sidFdata = params.get("sid.fdata");
                    String data = "";
                    String mac = "";
                    if(datas != null && StringUtils.isNotBlank(datas.get("data"))) {
                        data = datas.get("data");
                    }
                    else if(datas != null && StringUtils.isNotBlank(datas.get("d1"))) {
                        data = datas.get("d1");
                    }
                    if(datas != null && StringUtils.isNotBlank(datas.get("mac"))) {
                        mac = datas.get("mac");
                    }
                    else if(datas != null && StringUtils.isNotBlank(datas.get("d2"))) {
                        mac = datas.get("d2");
                    }
                    sbData.append(sidIndi);
                    sbData.append(sidFdata);
                    sbData.append(data);
                    sbData.append(mac);
                    payload = "55" + getRxDataFollowLrc(sbData.toString());
                    length = 4096;
                    //resDataLength = -1;
                    //resDataLenDivided = new int[] {};
                }
            }
            else if(StringUtils.equals(type, "step7")) {
                // 5.3 安全传输数据处理（主站到终端） 第7步
                String p2 = "11";
                if(params != null && StringUtils.isNotBlank(params.get("p2"))) {
                    switch(params.get("p2")) {
                        case "11":
                            // 明文+MAC方式：11
                            p2 = "11";
                            break;
                        case "96":
                            // 密文：96
                            p2 = "96";
                            break;
                        case "97":
                            // 密文+MAC：97
                            p2 = "97";
                            break;
                        default:
                            break;
                    }
                }

                String dataStart = "801C00" + p2;
                sbData.append(dataStart);
                int lc = 0;
                String data = "";
                if(datas != null && StringUtils.isNotBlank(datas.get("data"))) {
                    data = datas.get("data");
                }
                else if(datas != null && StringUtils.isNotBlank(datas.get("d1"))) {
                    data = datas.get("d1");
                }
                if(StringUtils.isNotBlank(data)) {
                    try {
                        lc = Hex.decodeHex(data).length;
                    }
                    catch(DecoderException e) {
                        logger.error(e.getMessage(), e.fillInStackTrace());
                    }
                }
                sbData.append(StringUtils.leftPad(Integer.toHexString(lc), 4, '0'));
                sbData.append(data);
                payload = "55" + getRxDataFollowLrc(sbData.toString());
                length = 4096;
                //resDataLength = -1;
                //resDataLenDivided = new int[] {};
            }
        }
        else if(StringUtils.equals(method, TesamMethodEnum.HANDLE_SECTITY_DATA_UP.getCode())) {
            // 安全传输数据处理（终端到主站）
            StringBuilder sbData = new StringBuilder();
            if(StringUtils.equals(type, "step1")) {
                // 5.5.1 上报（终端主动上报） 第1步
                String dataStart = "80140103";
                sbData.append(dataStart);
                int lc = 0;
                String plain = "";
                if(datas != null && StringUtils.isNotBlank(datas.get("plain"))) {
                    plain = datas.get("plain");
                }
                else if(datas != null && StringUtils.isNotBlank(datas.get("d1"))) {
                    plain = datas.get("d1");
                }
                try {
                    lc = Hex.decodeHex(plain).length;
                }
                catch(DecoderException e) {
                    logger.error(e.getMessage(), e.fillInStackTrace());
                }
                sbData.append(StringUtils.leftPad(Integer.toHexString(lc), 4, '0'));
                sbData.append(plain);
                payload = "55" + getRxDataFollowLrc(sbData.toString());
                length = 64;
                resDataLength = 16;
                resDataLenDivided = new int[] { 12, 4 };
            }
            else if(StringUtils.equals(type, "step4")) {
                // 5.5.1 上报（终端主动上报） 第4步
                if(params != null && StringUtils.isNotBlank(params.get("sid.indi"))
                    && StringUtils.isNotBlank(params.get("sid.fdata"))) {
                    String sidIndi = params.get("sid.indi");
                    String sidFdata = params.get("sid.fdata");
                    String data = "";
                    String mac = "";
                    if(datas != null && StringUtils.isNotBlank(datas.get("data"))) {
                        data = datas.get("data");
                    }
                    else if(datas != null && StringUtils.isNotBlank(datas.get("d1"))) {
                        data = datas.get("d1");
                    }
                    if(datas != null && StringUtils.isNotBlank(datas.get("mac"))) {
                        mac = datas.get("mac");
                    }
                    else if(datas != null && StringUtils.isNotBlank(datas.get("d2"))) {
                        mac = datas.get("d2");
                    }
                    sbData.append(sidIndi);
                    sbData.append(sidFdata);
                    sbData.append(data);
                    sbData.append(mac);
                    payload = "55" + getRxDataFollowLrc(sbData.toString());
                    length = 4096;
                    //resDataLength = -1;
                    //resDataLenDivided = new int[] {};
                }
            }
        }
        else if(StringUtils.equals(method, TesamMethodEnum.HANDLE_APP_LAYER_PROTO.getCode())) {
            // 应用层协议处理
            StringBuilder sbData = new StringBuilder();
            if(StringUtils.equals(type, "action")) {
                // 操作
                if(params != null && StringUtils.isNotBlank(params.get("sid.indi"))
                    && StringUtils.isNotBlank(params.get("sid.fdata"))) {
                    String sidIndi = params.get("sid.indi");
                    String sidFdata = params.get("sid.fdata");
                    String data = "";
                    String mac = "";
                    if(datas != null && StringUtils.isNotBlank(datas.get("data"))) {
                        data = datas.get("data");
                    }
                    else if(datas != null && StringUtils.isNotBlank(datas.get("d1"))) {
                        data = datas.get("d1");
                    }
                    if(datas != null && StringUtils.isNotBlank(datas.get("mac"))) {
                        mac = datas.get("mac");
                    }
                    else if(datas != null && StringUtils.isNotBlank(datas.get("d2"))) {
                        mac = datas.get("d2");
                    }
                    sbData.append(sidIndi);
                    sbData.append(sidFdata);
                    sbData.append(data);
                    sbData.append(mac);
                    payload = "55" + getRxDataFollowLrc(sbData.toString());
                    length = 64;
                }
            }
            else {
                // 读取
                String dataStart = "800E4002";
                sbData.append(dataStart);
                int lc = 0;
                String rn = "";
                String plain = "";
                if(datas != null && StringUtils.isNotBlank(datas.get("rn"))) {
                    rn = datas.get("rn");
                }
                else if(datas != null && StringUtils.isNotBlank(datas.get("d1"))) {
                    rn = datas.get("d1");
                }
                if(datas != null && StringUtils.isNotBlank(datas.get("plain"))) {
                    plain = datas.get("plain");
                }
                else if(datas != null && StringUtils.isNotBlank(datas.get("d2"))) {
                    plain = datas.get("d2");
                }
                try {
                    lc = Hex.decodeHex(rn).length + Hex.decodeHex(plain).length;
                }
                catch(DecoderException e) {
                    logger.error(e.getMessage(), e.fillInStackTrace());
                }
                sbData.append(StringUtils.leftPad(Integer.toHexString(lc), 4, '0'));
                sbData.append(rn);
                sbData.append(plain);
                payload = "55" + getRxDataFollowLrc(sbData.toString());
                length = 64;
                resDataLength = 4;
                resDataLenDivided = new int[] { 4 };
            }
        }
        try {
            byte[] rxBuf = transmit(device, Hex.decodeHex(payload), length);
            if(rxBuf != null && rxBuf.length >= 5) {
                if(rxBuf[0] == 0x55) {
                    byte sw1 = rxBuf[1];
                    byte sw2 = rxBuf[2];
                    logger.info(" sw1 : " + Hex.encodeHexString(new byte[] { sw1 }) + ", sw2 : " + Hex.encodeHexString(new byte[] { sw2 }));
                    String sw = Hex.encodeHexString(new byte[] { sw1, sw2 });
                    byte len1 = rxBuf[3];
                    byte len2 = rxBuf[4];
                    int len = Integer.parseInt(Hex.encodeHexString(new byte[] { len1, len2}), 16);
                    logger.info(" sw    : " + sw + ", len   : " + len);
                    if(StringUtils.equals(sw, "9000")) {
                        // 成功
                        if(len > 0 && rxBuf.length > 5 + len) {
                            byte[] dataBytes = new byte[len];
                            System.arraycopy(rxBuf, 5, dataBytes, 0, len);
                            logger.info(" data  : " + Hex.encodeHexString(dataBytes));
                            logger.info(" lrc   : " + Hex.encodeHexString(new byte[] { rxBuf[5 + len] }));
                            if(resDataLenDivided.length > 0 && len >= resDataLength) {
                                int index = 0;
                                for(int i = 0; i < resDataLenDivided.length; i++) {
                                    byte[] dataBytesTemp = new byte[resDataLenDivided[i]];
                                    System.arraycopy(dataBytes, index, dataBytesTemp, 0, dataBytesTemp.length);
                                    index += dataBytesTemp.length;
                                    resDataMap.put("d" + String.valueOf(i + 1), Hex.encodeHexString(dataBytesTemp));
                                }
                            }
                            else {
                                resDataMap.put("d1", Hex.encodeHexString(dataBytes));
                            }
                            tesamRxResponse.setSw(sw);
                            tesamRxResponse.setLen(len);
                            tesamRxResponse.setDataBytes(dataBytes);
                            tesamRxResponse.setLrc(rxBuf[5 + len]);
                            tesamRxResponse.setDatas(resDataMap);
                        }
                        else {
                            tesamRxResponse.setSw(sw);
                            tesamRxResponse.setLen(len);
                        }
                    }
                    else {
                        // 失败
                        tesamRxResponse.setSw(StringUtils.defaultIfBlank(sw, "0000"));
                        tesamRxResponse.setLen(len);
                    }
                }
            }
        }
        catch(DecoderException | NumberFormatException e) {
            logger.error(e.getMessage(), e.fillInStackTrace());
        }

        if(logger.isInfoEnabled()) {
            logger.info(" TesamRxResponse   : " + JsonConvertUtils.convertToString(tesamRxResponse));
        }

        return tesamRxResponse;

    }

    /*public static void main(String... args) {
        TesamRxResponse tesamRxResponse = new TesamRxResponse();
        byte[] rxBuf = null;
        int[] resDataLenDivided = {};
        LinkedHashMap<String, String> resDataMap = Maps.newLinkedHashMap();
        try {
            rxBuf = Hex.decodeHex("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");
        }
        catch(DecoderException e) {
            e.printStackTrace();
        }
        if(rxBuf != null && rxBuf.length >= 5) {
            if(rxBuf[0] == 0x55) {
                byte sw1 = rxBuf[1];
                byte sw2 = rxBuf[2];
                logger.info(" sw1 : " + Hex.encodeHexString(new byte[] { sw1 }) + ", sw2 : " + Hex.encodeHexString(new byte[] { sw2 }));
                String sw = Hex.encodeHexString(new byte[] { sw1, sw2 });
                byte len1 = rxBuf[3];
                byte len2 = rxBuf[4];
                int len = Integer.parseInt(Hex.encodeHexString(new byte[] { len1, len2}), 16);
                logger.info(" sw    : " + sw + ", len   : " + len);
                if(rxBuf.length > 5 + len) {
                    byte[] dataBytes = new byte[len];
                    System.arraycopy(rxBuf, 5, dataBytes, 0, len);
                    logger.info(" data  : " + Hex.encodeHexString(dataBytes));
                    logger.info(" lrc   : " + Hex.encodeHexString(new byte[] { rxBuf[5 + len] }));
                    if(resDataLenDivided.length > 0) {
                        int index = 0;
                        for(int i = 0; i < resDataLenDivided.length; i++) {
                            byte[] dataBytesTemp = new byte[resDataLenDivided[i]];
                            System.arraycopy(dataBytes, index, dataBytesTemp, 0, dataBytesTemp.length);
                            index += dataBytesTemp.length;
                            resDataMap.put("d" + String.valueOf(i + 1), Hex.encodeHexString(dataBytesTemp));
                        }
                    }
                    else {
                        resDataMap.put("d1", Hex.encodeHexString(dataBytes));
                    }
                    tesamRxResponse.setSw(sw);
                    tesamRxResponse.setLen(len);
                    tesamRxResponse.setDataBytes(dataBytes);
                    tesamRxResponse.setLrc(rxBuf[5 + len]);
                    tesamRxResponse.setDatas(resDataMap);
                }
            }
        }
    }*/

}
