package com.easesource.edge.northbound.service.gpio;

import com.easesource.edge.core.common.util.convert.JsonConvertUtils;
import com.easesource.edge.core.mqbus.mqtt.*;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.yaml.snakeyaml.Yaml;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;

public class MainApp {

    private static final Logger logger = LoggerFactory.getLogger(MainApp.class);

    @SuppressWarnings("all")
    public static void main(String... args) {
        if(logger.isInfoEnabled()) {
            logger.info(" >>>>>>> start southbound service (gpio) >>>>>>> ");
        }
        Map<String, Object> appMaps = Maps.newLinkedHashMap();
        Yaml yaml = new Yaml();
        try {
            InputStream in = MainApp.class.getClassLoader().getResourceAsStream("application.yml");
            appMaps = yaml.load(in);
            logger.info(" appMaps : " + JsonConvertUtils.convertToString(appMaps));
        }
        catch(Exception e) {
            logger.error(e.getMessage(), e.fillInStackTrace());
        }

        MqttClientConfig receiverClientConfig = new MqttClientConfig();
        MqttConnectConfig receiverConnectConfig = new MqttConnectConfig();
        MqttSubscribesConfig receiverSubscribesConfig = new MqttSubscribesConfig();
        MqttClientConfig senderClientConfig = new MqttClientConfig();
        MqttConnectConfig senderConnectConfig = new MqttConnectConfig();
        try {
            Map<String, Object> mqttProfileMap = (Map<String, Object>) appMaps.get("mqtt");
            // receiver config
            receiverClientConfig.setServerUri((String) mqttProfileMap.get("server-uri"));
            receiverClientConfig.setClientId((String) ((Map) mqttProfileMap.get("receiver")).get("client-id"));
            receiverConnectConfig.setUsername((String) mqttProfileMap.get("username"));
            receiverConnectConfig.setPassword((String) mqttProfileMap.get("password"));
            ArrayList<String> receiverSubscribesTopics = (ArrayList) ((Map) mqttProfileMap.get("receiver")).get("sub-topics");
            ArrayList<Integer> receiverSubscribesQoss = (ArrayList) ((Map) mqttProfileMap.get("receiver")).get("sub-qoss");
            if(receiverSubscribesTopics != null && receiverSubscribesQoss != null && receiverSubscribesTopics.size() == receiverSubscribesQoss.size()) {
                receiverSubscribesConfig.setTopics(receiverSubscribesTopics.toArray(new String[receiverSubscribesTopics.size()]));
                receiverSubscribesConfig.setQoss(receiverSubscribesQoss.stream().mapToInt(Integer::valueOf).toArray());
            }

            // sender config
            senderClientConfig.setServerUri((String) mqttProfileMap.get("server-uri"));
            senderClientConfig.setClientId((String) ((Map) mqttProfileMap.get("sender")).get("client-id"));
            senderConnectConfig.setUsername((String) mqttProfileMap.get("username"));
            senderConnectConfig.setPassword((String) mqttProfileMap.get("password"));
        }
        catch(Exception e) {
            logger.error(e.getMessage(), e.fillInStackTrace());
            return;
        }

        MqttReceiveHandler southboundMqttReceiveHandler = null;
        try {
            southboundMqttReceiveHandler = new MqttReceiveHandler(receiverClientConfig, receiverConnectConfig, receiverSubscribesConfig, new SouthboundMqttGpioCallback());
        }
        catch(Exception e) {
            logger.error(e.getMessage(), e.fillInStackTrace());
        }

        if(southboundMqttReceiveHandler == null) {
            logger.error("southboundMqttLoopback is null.");
            return;
        }

        ListeningExecutorService servicePool = MoreExecutors.listeningDecorator(Executors.newFixedThreadPool(1));
        ListenableFuture<Integer> future = servicePool.submit(southboundMqttReceiveHandler);
        future.addListener(() -> {
            try {
                logger.info(Thread.currentThread().getName() + " >> 异步处理成功.... " + future.get());
            }
            catch (InterruptedException | ExecutionException e) {
                logger.error(e.getMessage(), e.fillInStackTrace());
            }
        }, MoreExecutors.directExecutor());

        try {
            MqttSender southboundMqttSender = new MqttSender(senderClientConfig, senderConnectConfig);
            southboundMqttSender.sendToMqtt("hello", "/device/gpio/do1");
        }
        catch(Exception e) {
            logger.error(e.getMessage(), e.fillInStackTrace());
        }

        if(logger.isInfoEnabled()) {
            logger.info(" <<<<<<<< end southbound service (gpio) <<<<<<<< ");
        }
    }

}
