package cn.yunrui.bizservices.ebikemgmt.dao.impl;

import cn.yunrui.bizservices.ebikemgmt.dao.ChargeDao;
import cn.yunrui.bizservices.ebikemgmt.entity.ChargeRequest;
import cn.yunrui.bizservices.ebikemgmt.entity.ChargeResponse;
import cn.yunrui.mqttclient.ebikesrv.common.utils.JsonConvertUtils;
import cn.yunrui.mqttclient.ebikesrv.common.utils.MapUtils;
import cn.yunrui.mqttclient.ebikesrv.common.utils.ObjectUtils;
import cn.yunrui.mqttclient.ebikesrv.common.utils.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;

import java.math.BigDecimal;
import java.sql.ResultSetMetaData;
import java.sql.Types;
import java.util.*;

/**
 *
 */
public class ChargeDaoImpl implements ChargeDao {

    private final Logger logger = LoggerFactory.getLogger(getClass());

    private final RowMapper<Map<String, Object>> mapRowMapper = (rs, rowNum) -> {
        ResultSetMetaData rsmd = rs.getMetaData();
        Map<String, Object> map = new HashMap<>();
        for(int i = 1; i <= rsmd.getColumnCount(); i++) {
            if(rs.getObject(i) == null) {
                map.put(rsmd.getColumnLabel(i).trim(), null);
            }
            else {
                switch(rsmd.getColumnType(i)) {
                    case Types.INTEGER:
                        map.put(rsmd.getColumnLabel(i).trim(), rs.getInt(i));
                        break;
                    case Types.BIGINT:
                        map.put(rsmd.getColumnLabel(i).trim(), rs.getLong(i));
                        break;
                    case Types.FLOAT:
                        map.put(rsmd.getColumnLabel(i).trim(), rs.getFloat(i));
                        break;
                    case Types.REAL:
                        map.put(rsmd.getColumnLabel(i).trim(), rs.getFloat(i));
                        break;
                    case Types.DOUBLE:
                        map.put(rsmd.getColumnLabel(i).trim(), rs.getDouble(i));
                        break;
                    case Types.DECIMAL:
                        if(rsmd.getScale(i) == 0) {
                            map.put(rsmd.getColumnLabel(i).trim(), rs.getBigDecimal(i).longValue());
                        }
                        else {
                            map.put(rsmd.getColumnLabel(i).trim(), rs.getBigDecimal(i).doubleValue());
                        }
                        break;
                    case Types.NUMERIC:
                        if(rsmd.getScale(i) == 0) {
                            map.put(rsmd.getColumnLabel(i).trim(), rs.getBigDecimal(i).longValue());
                        }
                        else {
                            map.put(rsmd.getColumnLabel(i).trim(), rs.getBigDecimal(i).doubleValue());
                        }
                        break;
                    case Types.DATE:
                        map.put(rsmd.getColumnLabel(i).trim(), new Date(rs.getDate(i).getTime()));
                        break;
                    case Types.TIME:
                        map.put(rsmd.getColumnLabel(i).trim(), new Date(rs.getTime(i).getTime()));
                        break;
                    case Types.TIMESTAMP:
                        map.put(rsmd.getColumnLabel(i).trim(), new Date(rs.getTimestamp(i).getTime()));
                        break;
                    case Types.CHAR:
                        map.put(rsmd.getColumnLabel(i).trim(), rs.getString(i));
                        break;
                    case Types.NCHAR:
                        map.put(rsmd.getColumnLabel(i).trim(), rs.getString(i));
                        break;
                    case Types.VARCHAR:
                        map.put(rsmd.getColumnLabel(i).trim(), rs.getString(i));
                        break;
                    case Types.NVARCHAR:
                        map.put(rsmd.getColumnLabel(i).trim(), rs.getString(i));
                        break;
                    case Types.LONGVARCHAR:
                        map.put(rsmd.getColumnLabel(i).trim(), rs.getString(i));
                        break;
                    case Types.LONGNVARCHAR:
                        map.put(rsmd.getColumnLabel(i).trim(), rs.getString(i));
                        break;
                    default:
                        map.put(rsmd.getColumnLabel(i).trim(), rs.getString(i));
                        break;
                }
            }
        }
        return map;
    };

    private final RowMapper<ChargeRequest> crRowMapper = (rs, rowNum) -> {
        ChargeRequest chargeRequest = new ChargeRequest();
        chargeRequest.setOrgNo(rs.getString("subburo"));
        chargeRequest.setChargeplugNo(rs.getString("chargeplugNo"));
        chargeRequest.setChargeUserNo(rs.getString("chargeUserNo"));
        chargeRequest.setChargeUserType(rs.getString("chargeUserType"));
        chargeRequest.setType("start");
        chargeRequest.setMoney(rs.getDouble("money"));
        chargeRequest.setChargeTradeOrderId(rs.getString("chargeTradeOrderId"));
        chargeRequest.setEnterUserType(rs.getString("enterUserType"));
        chargeRequest.setEnterUserId(rs.getString("enterUserId"));
        return chargeRequest;
    };

    private final JdbcTemplate ebikeJdbcTemplate;

    public ChargeDaoImpl(JdbcTemplate ebikeJdbcTemplate) {
        this.ebikeJdbcTemplate = ebikeJdbcTemplate;
    }

    @Override
    public boolean checkIsCharging(Long chargeplugId) {
        String sql = "SELECT COUNT(chargeplugId) AS CNT FROM ebike.ebike_chargerecord WHERE chargeplugId = ? ";
        int cnt = ebikeJdbcTemplate.queryForObject(sql, new Object[] {chargeplugId}, Integer.class);
        return cnt > 0;
    }

    @Override
    public ChargeRequest buildChargeRequestPaySuccessWithnoCharging(String chargeTradeOrderId) {
        String sql = "SELECT cpo.* FROM ebike.ebike_log_chargeprepayorder cpo WHERE cpo.chargeTradeOrderId = ? " +
                " AND NOT EXISTS (SELECT 1 FROM ebike.ebike_chargerecord crc WHERE crc.chargeTradeOrderId = cpo.chargeTradeOrderId) " +
                " AND NOT EXISTS (SELECT 1 FROM ebike.ebike_chargerecord_his crh WHERE crh.chargeTradeOrderId = cpo.chargeTradeOrderId) " +
                " LIMIT 1 ";
        ChargeRequest chargeRequest;
        try {
            chargeRequest = ebikeJdbcTemplate.queryForObject(sql, new Object[] {chargeTradeOrderId}, crRowMapper);
        }
        catch(EmptyResultDataAccessException _erdae) {
            chargeRequest = null;
        }
        return chargeRequest;
    }

    @Override
    public boolean checkIsBeChargedByChargeTradeOrderId(String chargeTradeOrderId) {
        String sql1 = "SELECT COUNT(chargeTradeOrderId) AS CNT FROM ebike.ebike_chargerecord WHERE chargeTradeOrderId = ? ";
        int cnt1 = ebikeJdbcTemplate.queryForObject(sql1, new Object[] {chargeTradeOrderId}, Integer.class);
        String sql2 = "SELECT COUNT(chargeTradeOrderId) AS CNT FROM ebike.ebike_chargerecord_his WHERE chargeTradeOrderId = ? ";
        int cnt2 = ebikeJdbcTemplate.queryForObject(sql2, new Object[] {chargeTradeOrderId}, Integer.class);
        return cnt1 > 0 || cnt2 > 0;
    }

    @Override
    public int insertChargeRequestLog(Date requestTime, ChargeRequest chargeRequest, String chargeRecordId) {
        /*if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> start insertChargeRequestLog >>>>>> ");
            logger.debug(" chargeRequest    : " + JsonConvertUtils.convertToString(chargeRequest));
            logger.debug(" chargeRecordId   : " + chargeRecordId);
        }*/
        String sql = "INSERT INTO ebike.ebike_log_chargerequest(requestTime, chargeRecordId, buro, subburo, chargeplugNo, chargeUserNo, chargeUserType, type, money, chargeTradeOrderId, enterUserType, enterUserId) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?) ";
        List<Object> params = new ArrayList<>();
        params.add(requestTime != null ? requestTime : new Date());                                         // requestTime
        params.add(chargeRecordId);                                                                         // chargeRecordId
        params.add(ObjectUtils.isNotNull(chargeRequest) ? chargeRequest.getOrgNo() : null);                 // buro
        params.add(ObjectUtils.isNotNull(chargeRequest) ? chargeRequest.getOrgNo() : null);                 // subburo
        params.add(ObjectUtils.isNotNull(chargeRequest) ? chargeRequest.getChargeplugNo() : null);          // chargeplugNo
        params.add(ObjectUtils.isNotNull(chargeRequest) ? chargeRequest.getChargeUserNo() : null);          // chargeUserNo
        params.add(ObjectUtils.isNotNull(chargeRequest) ? chargeRequest.getChargeUserType() : null);        // chargeUserType
        params.add(ObjectUtils.isNotNull(chargeRequest) ? chargeRequest.getType() : null);                  // type
        params.add(ObjectUtils.isNotNull(chargeRequest) ? chargeRequest.getMoney() : null);                 // money
        params.add(ObjectUtils.isNotNull(chargeRequest) ? chargeRequest.getChargeTradeOrderId() : null);    // chargeTradeOrderId
        params.add(ObjectUtils.isNotNull(chargeRequest) ? chargeRequest.getEnterUserType() : null);         // enterUserType
        params.add(ObjectUtils.isNotNull(chargeRequest) ? chargeRequest.getEnterUserId() : null);           // enterUserId
        /*if(logger.isDebugEnabled()) {
            logger.debug(" sql      : " + sql);
            logger.debug(" params   : " + JsonConvertUtils.convertToString(params));
            logger.debug(" <<<<<<< end insertChargeRequestLog <<<<<<< ");
        }*/
        return ebikeJdbcTemplate.update(sql, params.toArray());
    }

    @Override
    public int updateChargeRequestLog(Date responseTime, ChargeResponse chargeResponse, String chargeRecordId, Boolean gnrtChargeRecord, String gnrtFailReason) {
        /*if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> start updateChargeRequestLog >>>>>> ");
            logger.debug(" chargeResponse   : " + JsonConvertUtils.convertToString(chargeResponse));
            logger.debug(" chargeRecordId   : " + chargeRecordId);
            logger.debug(" gnrtChargeRecord : " + gnrtChargeRecord);
            logger.debug(" gnrtFailReason   : " + gnrtFailReason);
        }*/
        String sql = "UPDATE ebike.ebike_log_chargerequest SET responseTime = ?, responseStatus = ?, responseMsg = ?, gnrtChargeRecord = ?, gnrtFailReason = ? WHERE chargeRecordId = ? ";
        List<Object> params = new ArrayList<>();
        params.add(responseTime != null ? responseTime : new Date());                                       // responseTime
        params.add(ObjectUtils.isNotNull(chargeResponse) ? chargeResponse.getStatus() : null);              // responseStatus
        params.add(ObjectUtils.isNotNull(chargeResponse) ? chargeResponse.getMsg() : null);                 // responseMsg
        params.add(gnrtChargeRecord);                                                                       // gnrtChargeRecord
        params.add(gnrtFailReason);                                                                         // gnrtFailReason
        params.add(chargeRecordId);                                                                         // chargeRecordId
        /*if(logger.isDebugEnabled()) {
            logger.debug(" sql      : " + sql);
            logger.debug(" params   : " + JsonConvertUtils.convertToString(params));
            logger.debug(" <<<<<<< end updateChargeRequestLog <<<<<<< ");
        }*/
        return ebikeJdbcTemplate.update(sql, params.toArray());
    }

    @Override
    public List<Map<String, Object>> getGnrtFailedChargeRequestList(Date startTime, Date endTime) {
        String sql = "SELECT log.chargeRecordId, log.chargeplugNo, log.chargeUserNo, log.chargeUserType, log.type, log.money, log.chargeTradeOrderId, log.responseTime, log.responseStatus, log.responseMsg " +
                "  FROM ebike.ebike_log_chargerequest log " +
                " WHERE (gnrtChargeRecord = 0 OR gnrtChargeRecord IS NULL) AND type = 'start' AND money > 0.0 AND chargeTradeOrderId IS NOT NULL AND (returnFlag = 'N' OR returnFlag IS NULL) " +
                "   AND NOT EXISTS (SELECT 1 FROM ebike.ebike_chargerecord cr WHERE cr.chargeTradeOrderId = log.chargeTradeOrderId AND chargeTime >= ?) " +
                "   AND NOT EXISTS (SELECT 1 FROM ebike.ebike_chargerecord_his crh WHERE crh.chargeTradeOrderId = log.chargeTradeOrderId AND chargeTime >= ?) " +
                "   AND requestTime >= ? AND requestTime <= ? ";
        if(logger.isDebugEnabled()) {
            logger.debug(" sql  : " + sql);
        }
        return ebikeJdbcTemplate.query(sql, new Object[] {startTime, startTime, startTime, endTime}, mapRowMapper);
    }

    @Override
    public int updateChargeRequestLogByReturn(String chargeRecordId, String returnFlag, BigDecimal returnMoney, Date returnTime, String returnDesc, String returnTradeOrderId) {
        String sql = "UPDATE ebike.ebike_log_chargerequest SET returnFlag = ?, returnMoney = ?, returnTime = ?, returnDesc = ?, returnTradeOrderId = ? WHERE chargeRecordId = ? ";
        return ebikeJdbcTemplate.update(sql, returnFlag, returnMoney, returnTime, returnDesc, returnTradeOrderId, chargeRecordId);
    }

    @Override
    public int updateChargeRequestLogByReturnResult(String chargeTradeOrderId, String returnTradeOrderId, String returnFlag, Date returnSuccessTime) {
        String sql = "UPDATE ebike.ebike_log_chargerequest SET returnFlag = ?, returnSuccessTime = ? WHERE chargeTradeOrderId = ? AND returnTradeOrderId = ? ";
        return ebikeJdbcTemplate.update(sql, returnFlag, returnSuccessTime, chargeTradeOrderId, returnTradeOrderId);
    }

    @Override
    public int updateChargePrepayOrderLog(String chargeTradeOrderId, int payFlag, Date paySuccTime, String payFailReason) {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> start updateChargePrepayOrderLog >>>>>> ");
            logger.debug(" chargeTradeOrderId   : " + chargeTradeOrderId);
        }
        if(StringUtils.isNotBlank(chargeTradeOrderId)) {
            String sql = "UPDATE ebike.ebike_log_chargeprepayorder SET payFlag = ?, paySuccTime = ?, payFailReason = ? WHERE chargeTradeOrderId = ? ";
            List<Object> params = new ArrayList<>();
            params.add(payFlag);                                                                // payFlag
            params.add(paySuccTime);                                                            // paySuccTime
            params.add(payFailReason);                                                          // payFailReason
            params.add(chargeTradeOrderId);                                                     // chargeTradeOrderId
            if(logger.isDebugEnabled()) {
                logger.debug(" sql      : " + sql);
                logger.debug(" params   : " + JsonConvertUtils.convertToString(params));
                logger.debug(" <<<<<<< end updateChargePrepayOrderLog <<<<<<< ");
            }
            return ebikeJdbcTemplate.update(sql, params.toArray());
        }
        return 0;
    }

    @Override
    public int insertChargeRecord(Map<String, Object> chargeRecordMap) {
        if(!MapUtils.isEmpty(chargeRecordMap)) {
            String sql = "INSERT INTO ebike.ebike_chargerecord(chargeRecordId, chargeplugId, chargedeviceId, chargestationId, chargeUserId, chargeTime, chargePayMode, chargeMoney, coinMoney, card_id, chargeTradeOrderId, enterUserType, enterUserId, sharingReceivers, sharingMoneyFreezeCycle, serviceRate, schemeId, schemeType, unitPrice1, unitPrice2, unitPrice3, unitPrice4, unitPrice5, stepCount, stepPower1, stepPower2, stepPower3, stepPower4, minCharge, isReturn, closingPeriod, costSchemeId, costUnitPrice, chargeOpenFlag, firstUnitPriceFlag, firstChargeAmount, currUnitPriceFlag, chargeAmount, usageAmount, surplusAmount, usageElecCons, remainElecCons, remainTime, updateTime, schemeDetails) " +
                "VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?) ";
            List<Object> params = new ArrayList<>();
            params.add(chargeRecordMap.get("chargeRecordId"));
            params.add(chargeRecordMap.get("chargeplugId"));
            params.add(chargeRecordMap.get("chargedeviceId"));
            params.add(chargeRecordMap.get("chargestationId"));
            params.add(chargeRecordMap.get("chargeUserId"));
            params.add(chargeRecordMap.get("chargeTime"));
            params.add(chargeRecordMap.get("chargePayMode"));
            params.add(chargeRecordMap.get("chargeMoney"));
            params.add(chargeRecordMap.get("coinMoney"));
            params.add(chargeRecordMap.get("card_id"));
            params.add(chargeRecordMap.get("chargeTradeOrderId"));
            params.add(chargeRecordMap.get("enterUserType"));
            params.add(chargeRecordMap.get("enterUserId"));
            params.add(chargeRecordMap.get("sharingReceivers"));
            params.add(chargeRecordMap.get("sharingMoneyFreezeCycle"));
            params.add(chargeRecordMap.get("serviceRate"));
            params.add(chargeRecordMap.get("schemeId"));
            params.add(chargeRecordMap.get("schemeType"));
            params.add(chargeRecordMap.get("unitPrice1"));
            params.add(chargeRecordMap.get("unitPrice2"));
            params.add(chargeRecordMap.get("unitPrice3"));
            params.add(chargeRecordMap.get("unitPrice4"));
            params.add(chargeRecordMap.get("unitPrice5"));
            params.add(chargeRecordMap.get("stepCount"));
            params.add(chargeRecordMap.get("stepPower1"));
            params.add(chargeRecordMap.get("stepPower2"));
            params.add(chargeRecordMap.get("stepPower3"));
            params.add(chargeRecordMap.get("stepPower4"));
            params.add(chargeRecordMap.get("minCharge"));
            params.add(chargeRecordMap.get("isReturn"));
            params.add(chargeRecordMap.get("closingPeriod"));
            params.add(chargeRecordMap.get("costSchemeId"));
            params.add(chargeRecordMap.get("costUnitPrice"));
            params.add(chargeRecordMap.get("chargeOpenFlag"));
            params.add(chargeRecordMap.get("firstUnitPriceFlag"));
            params.add(chargeRecordMap.get("firstChargeAmount"));
            params.add(chargeRecordMap.get("currUnitPriceFlag"));
            params.add(chargeRecordMap.get("chargeAmount"));
            params.add(chargeRecordMap.get("usageAmount"));
            params.add(chargeRecordMap.get("surplusAmount"));
            params.add(chargeRecordMap.get("usageElecCons"));
            params.add(chargeRecordMap.get("remainElecCons"));
            params.add(chargeRecordMap.get("remainTime"));
            params.add(new Date());                                     // updateTime
            params.add(chargeRecordMap.get("schemeDetails")); // 2024-11-29 新增计费方案信息
            return ebikeJdbcTemplate.update(sql, params.toArray());
        }
        return 0;
    }

    @Override
    public Long getChargeplugIdByChargeplugNo(String orgNo, String chargeplugNo) {
        if(StringUtils.isNotBlank(orgNo) && StringUtils.isNotBlank(chargeplugNo)) {
            StringBuilder sqlBuilder = new StringBuilder();
            List<Object> params = new ArrayList<>();
            sqlBuilder.append("SELECT ecp.id ");
            sqlBuilder.append("  FROM cisp_dev.dev_ebikechargeplug ecp ");
            sqlBuilder.append(" INNER JOIN cisp_dev.dev_powersystemresource pss ON pss.id = ecp.id ");
            sqlBuilder.append(" WHERE ecp.chargeplugNo = ? AND pss.subburo = ? ");
            params.add(chargeplugNo);
            params.add(orgNo);
            sqlBuilder.append(" LIMIT 1 ");
            Long chargeplugId;
            try {
                chargeplugId = ebikeJdbcTemplate.queryForObject(sqlBuilder.toString(), params.toArray(), Long.class);
            }
            catch(EmptyResultDataAccessException _erdae) {
                chargeplugId = null;
            }
            return chargeplugId;
        }
        return null;
    }

    @Override
    public Map<String, Object> getChargePlugValueMap(String orgNo, String chargeplugNo) {
        if(StringUtils.isNotBlank(orgNo) && StringUtils.isNotBlank(chargeplugNo)) {
            StringBuilder sqlBuilder = new StringBuilder();
            List<Object> params = new ArrayList<>();
            sqlBuilder.append("SELECT ecp.`id` AS chargeplugId, ecp.chargedeviceId, ecd.chargestationId ");
            sqlBuilder.append("  FROM cisp_dev.dev_ebikechargeplug ecp ");
            sqlBuilder.append(" INNER JOIN cisp_dev.dev_powersystemresource pss ON pss.`id` = ecp.id AND pss.className = 'EBikeChargePlug' ");
            sqlBuilder.append(" INNER JOIN cisp_dev.dev_ebikechargedevice ecd ON ecd.`id` = ecp.chargedeviceId ");
            sqlBuilder.append(" INNER JOIN cisp_dev.dev_ebikechargestation ecs ON ecs.`id` = ecd.chargestationId ");
            sqlBuilder.append(" WHERE ecp.chargeplugNo = ? AND pss.subburo = ? ");
            params.add(chargeplugNo);
            params.add(orgNo);
            sqlBuilder.append(" LIMIT 1 ");
            Map<String, Object> chargePlugValueMap;
            try {
                chargePlugValueMap = ebikeJdbcTemplate.queryForObject(sqlBuilder.toString(), params.toArray(), mapRowMapper);
            }
            catch(EmptyResultDataAccessException _erdae) {
                chargePlugValueMap = null;
            }
            return chargePlugValueMap;
        }
        return null;
    }

    @Override
    public String getChargeUserIdByChargeUserNo(String chargeUserNo, String chargeUserType) {
        if(StringUtils.isNotBlank(chargeUserNo) && StringUtils.isNotBlank(chargeUserType)) {
            StringBuilder sqlBuilder = new StringBuilder();
            List<Object> params = new ArrayList<>();
            sqlBuilder.append("SELECT ecu.chargeUserId ");
            sqlBuilder.append("  FROM ebike.ebike_chargeuser ecu ");
            sqlBuilder.append(" WHERE ecu.chargeUserNo = ? AND ecu.chargeUserType = ? ");
            params.add(chargeUserNo);
            params.add(chargeUserType);
            String chargeUserId;
            try {
                chargeUserId = ebikeJdbcTemplate.queryForObject(sqlBuilder.toString(), params.toArray(), String.class);
            }
            catch(EmptyResultDataAccessException _erdae) {
                chargeUserId = null;
            }
            return chargeUserId;
        }
        return null;
    }

}
