package cn.yunrui.mqttclient.ebikesrv.handletask;

import cn.yunrui.mqttclient.ebikesrv.common.utils.*;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.cache.ChargeDeviceCache;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.cache.OpsWorkOrderCache;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.entity.ChargeDeviceOps;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.entity.ChargeDeviceProps;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.entity.ChargePlugProps;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.entity.OpsWorkOrder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.annotation.Resource;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.TimeUnit;

/**
 * 充电设备离线处理任务（离线超时判断）
 *
 * @author Nick Zhang
 * @date 2019-04-28
 */
public class ChargeDeviceOfflineHandleTask {

    private final Logger logger = LoggerFactory.getLogger(getClass());
    private final ConcurrentLinkedQueue<ChargeDeviceOps> linkedQueue = new ConcurrentLinkedQueue<>();
    private final static int INTERVAL_WAIT = 60;

    private Thread chargeDeviceOfflineTimeoutScanThread;
    private Thread chargeDeviceOfflineTimeoutHandleThread;

    @Resource(name = "chargeDeviceCache")
    private ChargeDeviceCache chargeDeviceCache;

    @Resource(name = "opsWorkOrderCache")
    private OpsWorkOrderCache opsWorkOrderCache;

    /**
     * 初始化
     */
    @PostConstruct
    public void init() {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> start init ChargeDeviceOfflineHandleTask >>>>>> ");
        }
        chargeDeviceOfflineTimeoutScanThread = new ChargeDeviceOfflineTimeoutScanThread();
        chargeDeviceOfflineTimeoutScanThread.setDaemon(true);
        chargeDeviceOfflineTimeoutScanThread.start();
        chargeDeviceOfflineTimeoutHandleThread = new ChargeDeviceOfflineTimeoutHandleThread();
        chargeDeviceOfflineTimeoutHandleThread.setDaemon(true);
        chargeDeviceOfflineTimeoutHandleThread.start();
        if(logger.isDebugEnabled()) {
            logger.debug(" <<<<<<< end init ChargeDeviceOfflineHandleTask <<<<<<< ");
        }
    }

    /**
     * 销毁
     */
    @PreDestroy
    public void destroy() {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> start destroy ChargeDeviceOfflineHandleTask >>>>>> ");
        }
        if(chargeDeviceOfflineTimeoutScanThread != null) {
            try {
                chargeDeviceOfflineTimeoutScanThread.join();
            }
            catch(InterruptedException _ie) {
                logger.error(_ie.getMessage(), _ie.fillInStackTrace());
            }
        }
        if(chargeDeviceOfflineTimeoutHandleThread != null) {
            try {
                chargeDeviceOfflineTimeoutHandleThread.join();
            }
            catch(InterruptedException _ie) {
                logger.error(_ie.getMessage(), _ie.fillInStackTrace());
            }
        }
        if(logger.isDebugEnabled()) {
            logger.debug(" <<<<<<< end destroy ChargeDeviceOfflineHandleTask <<<<<<< ");
        }
    }

    /**
     * 获取超时充电设备工况信息
     * @return 超时充电设备工况信息
     */
    private List<ChargeDeviceOps> getTimeoutChargeDeviceOpsList() {
        List<ChargeDeviceOps> cdoList = chargeDeviceCache.getTimeoutChargeDeviceOpsList();
        if(CollectionUtils.sizeIsNotEmptyIgnoreNull(cdoList)) {
            chargeDeviceCache.lockTimeoutChargeDeviceOpsList(cdoList);
        }
        return cdoList;
    }

    /**
     * 充电设备离线超时扫描线程
     */
    private class ChargeDeviceOfflineTimeoutScanThread extends Thread {

        @Override
        public void run() {
            while(true) {
                try {
                    List<ChargeDeviceOps> list = getTimeoutChargeDeviceOpsList();
                    if(CollectionUtils.sizeIsEmptyIgnoreNull(list)) {
                        TimeUnit.SECONDS.sleep(INTERVAL_WAIT);
                    }
                    else {
                        synchronized(linkedQueue) {
                            for(ChargeDeviceOps chargeDeviceOps : list) {
                                if(logger.isInfoEnabled()) {
                                    logger.info(" charge device offline timeout scan   : " + JsonConvertUtils.convertToString(chargeDeviceOps));
                                }
                                linkedQueue.offer(chargeDeviceOps);
                            }
                            linkedQueue.notifyAll();
                        }
                    }
                }
                catch(InterruptedException ie) {
                    logger.error(ie.getMessage(), ie.fillInStackTrace());
                    break;
                }
            }
        }
    }

    /**
     * 充电设备离线超时处理线程
     */
    private class ChargeDeviceOfflineTimeoutHandleThread extends Thread {

        @Override
        public void run() {
            while(true) {
                try {
                    ChargeDeviceOps chargeDeviceOps = linkedQueue.poll();
                    if(chargeDeviceOps == null) {
                        synchronized(linkedQueue) {
                            linkedQueue.wait(100L);
                        }
                    }
                    else {
                        if(logger.isInfoEnabled()) {
                            logger.info(" charge device offline timeout handle : " + JsonConvertUtils.convertToString(chargeDeviceOps));
                        }
                        if(StringUtils.isNotBlank(chargeDeviceOps.getDeviceId()) && chargeDeviceOps.getPlugCount() != null && chargeDeviceOps.getPlugCount() > 0) {
                            String[] plugStatus = new String[chargeDeviceOps.getPlugCount()];
                            Arrays.fill(plugStatus, "9");
                            //
                            ChargeDeviceProps cdp = chargeDeviceCache.get(chargeDeviceOps.getDeviceId());
                            if(cdp != null && MapUtils.isNotEmpty(cdp.getPlugPropsMap())) {
                                for(int i = 0; i < cdp.getPlugPropsMap().size(); i++) {
                                    ChargePlugProps cpp = cdp.getPlugPropsMap().get(i + 1);
                                    if(cpp != null && StringUtils.isNotBlank(cpp.getOpStatus())) {
                                        if(StringUtils.equals("1", cpp.getOpStatus())) {
                                            plugStatus[i] = "1";            // 1 - 正在使用
                                        }
                                        else if(StringUtils.equals("2", cpp.getOpStatus())) {
                                            plugStatus[i] = "2";            // 2 - 已被预订
                                        }
                                        else if(StringUtils.equals("8", cpp.getOpStatus())) {
                                            plugStatus[i] = "8";            // 8 - 正在检修
                                        }
                                    }
                                }
                            }
                            chargeDeviceCache.updateOpStatus(chargeDeviceOps.getDeviceId(), "0", plugStatus);
                            //
                            // 设备离线
                            Integer plugSn = 0;
                            // 获取未复归设备停电告警
                            List<OpsWorkOrder> opsWorkOrderList102 = opsWorkOrderCache.getOpsWorkOrderList(chargeDeviceOps.getDeviceId(), String.valueOf(plugSn), "102", "N");      // 102 - 设备停电
                            if(CollectionUtils.sizeIsEmptyIgnoreNull(opsWorkOrderList102)) {
                                // 获取未复归设备离线告警
                                List<OpsWorkOrder> opsWorkOrderList101 = opsWorkOrderCache.getOpsWorkOrderList(chargeDeviceOps.getDeviceId(), String.valueOf(plugSn), "101", "N");  // 101 - 设备离线
                                if(CollectionUtils.sizeIsNotEmptyIgnoreNull(opsWorkOrderList101)) {
                                    int index = 1;
                                    for(OpsWorkOrder opsWorkOrder : opsWorkOrderList101) {
                                        if(index > 1) {
                                            opsWorkOrder.setRecoveryTime(new Date());
                                            opsWorkOrder.setIsRecovery("Y");
                                            opsWorkOrderCache.recoveryOpsWorkOrder(opsWorkOrder);
                                        }
                                        index++;
                                    }
                                }
                                else {
                                    OpsWorkOrder opsWorkOrder = new OpsWorkOrder();
                                    opsWorkOrder.setOpsworkorderId(WorkOrderIdUtils.getNewOpsWorkOrderId(chargeDeviceOps.getChargedeviceNo(), "101"));            // 101 - 设备离线
                                    opsWorkOrder.setOpsworkorderType("101");                                // 101 - 设备离线
                                    opsWorkOrder.setOpsworkorderStatus("10");                               // 10 - 未处理
                                    opsWorkOrder.setOpsObjectType("EBikeChargeDevice");                     //
                                    opsWorkOrder.setOpsObjectId(chargeDeviceOps.getChargedeviceId());
                                    opsWorkOrder.setOpsworkorderDesc("设备离线");
                                    opsWorkOrderCache.createOpsWorkOrder(opsWorkOrder);
                                }
                            }
                        }
                    }
                }
                catch(InterruptedException ie) {
                    logger.error(ie.getMessage(), ie.fillInStackTrace());
                    break;
                }
            }
        }
    }

}
