package cn.yunrui.mqttclient.ebikesrv.handletask;

import cn.yunrui.mqttclient.ebikesrv.common.utils.*;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.constant.ChargeOpenFlag;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.constant.FinishReason;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.constant.TemplateReturnDesc;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.dao.ChargeRecordDao;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.dao.ChargeRecordIncomeDetailDao;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.dao.CsshDistschemeDao;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.entity.ChargeRecord;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.entity.ChargeRecordIncomeDetailInfo;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.entity.CsshDistSchemeInfo;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.entity.SchemeDetail;
import cn.yunrui.mqttclient.ebikesrv.syncevent.publisher.ChargeOrderEventPublisher;
import com.ebikepay.openservices.entity.*;
import com.ebikepay.openservices.request.TradeOrderCreateRequest;
import com.ebikepay.openservices.request.TradeOrderPayRequest;
import com.ebikepay.openservices.request.TradeOrderRefundRequest;
import com.ebikepay.openservices.response.TradeOrderCreateResponse;
import com.ebikepay.openservices.response.TradeOrderPayResponse;
import com.ebikepay.openservices.response.TradeOrderRefundResponse;
import com.ebikepay.openservices.service.TradeOrderService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.TimeUnit;

/**
 * 充电记录结算处理任务（包括退费处理）
 *
 * @author Nick Zhang
 * @date 2019-04-13
 */
public class ChargeRecordClosingHandleTask {

    private final Logger logger = LoggerFactory.getLogger(getClass());

    /**
     * 充电持续时间阀值（已收到结束命令），10分钟
     */
    private final static long CHARGDURTIME_THRVAL1 = 600000L;

    /**
     * 充电持续时间阀值（未收到结束命令），30分钟
     */
    private final static long CHARGDURTIME_THRVAL2 = 1800000L;

    /**
     * 驿吧平台在途资金
     */
    private final static String EB_TRANSIT_USERTYPE = "ebike_transit";

    /**
     * 驿吧平台在途资金
     */
    private final static String EB_TRANSIT_USERID = "10000000";

    /**
     * 驿吧平台服务费
     */
    private final static String EB_SERVICE_USERTYPE = "ebike_service";

    /**
     * 驿吧平台服务费
     */
    private final static String EB_SERVICE_USERID = "10000000";

    /**
     * 单位：秒
     */
    private final static int INTERVAL_WAIT = 1;

    private final ConcurrentLinkedQueue<ChargeRecord> closingLinkedQueue = new ConcurrentLinkedQueue<>();
    private final ConcurrentLinkedQueue<ChargeRecord> returnLinkedQueue = new ConcurrentLinkedQueue<>();
    private Thread closingHandleChargeRecordScanThread;
    private Thread closingChargeRecordHandleThread;
    private Thread returnHandleChargeRecordScanThread;
    private Thread returnChargeRecordHandleThread;

    @Resource(name = "tradeOrderService")
    private TradeOrderService tradeOrderService;

    @Resource(name = "chargeRecordDao")
    private ChargeRecordDao chargeRecordDao;

    @Resource(name = "chargeRecordIncomeDetailDao")
    private ChargeRecordIncomeDetailDao chargeRecordIncomeDetailDao;

    @Resource(name = "csshDistschemeDao")
    private CsshDistschemeDao csshDistschemeDao;

    @Resource(name = "chargeOrderEventPublisher")
    private ChargeOrderEventPublisher chargeOrderEventPublisher;

    /**
     * 初始化
     */
    @PostConstruct
    public void init() {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> start init ChargeRecordClosingHandleTask >>>>>> ");
        }
        closingHandleChargeRecordScanThread = new ClosingHandleChargeRecordScanThread();
        closingHandleChargeRecordScanThread.setDaemon(true);
        closingHandleChargeRecordScanThread.start();
        closingChargeRecordHandleThread = new ClosingChargeRecordHandleThread();
        closingChargeRecordHandleThread.setDaemon(true);
        closingChargeRecordHandleThread.start();
        returnHandleChargeRecordScanThread = new ReturnHandleChargeRecordScanThread();
        returnHandleChargeRecordScanThread.setDaemon(true);
        returnHandleChargeRecordScanThread.start();
        returnChargeRecordHandleThread = new ReturnChargeRecordHandleThread();
        returnChargeRecordHandleThread.setDaemon(true);
        returnChargeRecordHandleThread.start();
        if(logger.isDebugEnabled()) {
            logger.debug(" <<<<<<< end init ChargeRecordClosingHandleTask <<<<<<< ");
        }
    }

    /**
     * 销毁
     */
    @PreDestroy
    public void destroy() {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> start destroy ChargeRecordClosingHandleTask >>>>>> ");
        }
        if(closingHandleChargeRecordScanThread != null) {
            try {
                closingHandleChargeRecordScanThread.join();
            }
            catch(InterruptedException _ie) {
                logger.error(_ie.getMessage(), _ie.fillInStackTrace());
            }
        }
        if(closingChargeRecordHandleThread != null) {
            try {
                closingChargeRecordHandleThread.join();
            }
            catch(InterruptedException _ie) {
                logger.error(_ie.getMessage(), _ie.fillInStackTrace());
            }
        }
        if(returnHandleChargeRecordScanThread != null) {
            try {
                returnHandleChargeRecordScanThread.join();
            }
            catch(InterruptedException _ie) {
                logger.error(_ie.getMessage(), _ie.fillInStackTrace());
            }
        }
        if(returnChargeRecordHandleThread != null) {
            try {
                returnChargeRecordHandleThread.join();
            }
            catch(InterruptedException _ie) {
                logger.error(_ie.getMessage(), _ie.fillInStackTrace());
            }
        }
        if(logger.isDebugEnabled()) {
            logger.debug(" <<<<<<< end destroy ChargeRecordClosingHandleTask <<<<<<< ");
        }
    }

    /**
     * 锁定待结算充电记录
     * @return 待结算充电记录
     */
    private List<ChargeRecord> lockTobeClosingChargeRecordList() throws Exception {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>>>>>> start lockTobeClosingChargeRecordList >>>>>>>>>> ");
        }
        List<ChargeRecord> crList = chargeRecordDao.getNotClosingChargeRecordList();
        if(logger.isDebugEnabled()) {
            logger.debug(" crList size : " + (crList != null ? crList.size() : 0));
            logger.debug(" crList      : " + JsonConvertUtils.convertToString(crList));
        }
        if(CollectionUtils.sizeIsNotEmptyIgnoreNull(crList)) {
            chargeRecordDao.lockNotClosingChargeRecordList(crList);
        }
        return crList;
    }

    /**
     * 锁定待退费充电记录
     * @return 待退费充电记录
     */
    private List<ChargeRecord> lockTobeReturnChargeRecordList() throws Exception {
        List<ChargeRecord> crList = chargeRecordDao.getNotReturnChargeRecordList();
        if(CollectionUtils.sizeIsNotEmptyIgnoreNull(crList)) {
            chargeRecordDao.lockNotReturnChargeRecordList(crList);
        }
        return crList;
    }

    private boolean isFullReturn(ChargeRecord chargeRecord) {
        boolean isFullReturn = false;
        if(ObjectUtils.isNotNull(chargeRecord)) {
            String finishReason = chargeRecord.getFinishReason();
            if(chargeRecord.getChargeOpenFlag() == ChargeOpenFlag.FLAG_SUCCESS) {
                // 充电开启成功
                Date chargeOpenTime = chargeRecord.getChargeOpenTime();
                Date chargeFinishTime = chargeRecord.getChargeFinishTime();
                Double maxPower = chargeRecord.getMaxPower();
                if(StringUtils.equals(FinishReason.REASON_PLUGPULLOUT, finishReason)) {
                    if(chargeOpenTime != null && chargeFinishTime != null && DoubleUtils.defaultIfNull(maxPower, 0.0D).compareTo(0.005D) <= 0 && (chargeFinishTime.getTime() - chargeOpenTime.getTime() < CHARGDURTIME_THRVAL1)) {
                        isFullReturn = true;
                    }
                }
                else if(StringUtils.equals(FinishReason.REASON_PLUGOVERPOWER, finishReason) || StringUtils.equals(FinishReason.REASON_PLUGLOSEPOWER, finishReason)) {
                    // 计费方案为按次时，结束原因是功率超限或端子失电，则全额退款
                    if(StringUtils.equals(chargeRecord.getSchemeType(), "03")) {
                        isFullReturn = true;
                    }
                    else {
                        if(chargeOpenTime != null && chargeFinishTime != null && (chargeFinishTime.getTime() - chargeOpenTime.getTime() < CHARGDURTIME_THRVAL1)) {
                            isFullReturn = true;
                        }
                    }
                }
                else if(StringUtils.equals(FinishReason.REASON_SURPLUS_ZERO, finishReason)) {
                    if(chargeOpenTime != null && chargeFinishTime != null && DoubleUtils.defaultIfNull(maxPower, 0.0D).compareTo(0.005D) <= 0 && (chargeFinishTime.getTime() - chargeOpenTime.getTime() < CHARGDURTIME_THRVAL2)) {
                        isFullReturn = true;
                    }
                }
            }
            else {
                isFullReturn = true;
            }
        }
        return isFullReturn;
    }

    private String getFullReturnDesc(ChargeRecord chargeRecord) {
        String returnDesc = "";
        if(ObjectUtils.isNotNull(chargeRecord)) {
            String finishReason = chargeRecord.getFinishReason();
            if(chargeRecord.getChargeOpenFlag() == ChargeOpenFlag.FLAG_SUCCESS) {
                // 充电开启成功
                Date chargeOpenTime = chargeRecord.getChargeOpenTime();
                Date chargeFinishTime = chargeRecord.getChargeFinishTime();
                Double maxPower = chargeRecord.getMaxPower();
                if(StringUtils.equals(FinishReason.REASON_PLUGPULLOUT, finishReason)) {
                    if(chargeOpenTime != null && chargeFinishTime != null && DoubleUtils.defaultIfNull(maxPower, 0.0D).compareTo(0.005D) <= 0 && (chargeFinishTime.getTime() - chargeOpenTime.getTime() < CHARGDURTIME_THRVAL1)) {
                        returnDesc = TemplateReturnDesc.RETURN_DESC_NOTINTIME;
                    }
                }
                else if(StringUtils.equals(FinishReason.REASON_PLUGOVERPOWER, finishReason)) {
                    // 计费方案为按次时，结束原因是功率超限或端子失电，则全额退款
                    if(StringUtils.equals(chargeRecord.getSchemeType(), "03")) {
                        returnDesc = TemplateReturnDesc.RETURN_DESC_OVERPOWER;
                    }
                    else {
                        if(chargeOpenTime != null && chargeFinishTime != null && (chargeFinishTime.getTime() - chargeOpenTime.getTime() < CHARGDURTIME_THRVAL1)) {
                            returnDesc = TemplateReturnDesc.RETURN_DESC_OVERPOWER;
                        }
                    }
                }
                else if(StringUtils.equals(FinishReason.REASON_PLUGLOSEPOWER, finishReason)) {
                    // 计费方案为按次时，结束原因是功率超限或端子失电，则全额退款
                    if(StringUtils.equals(chargeRecord.getSchemeType(), "03")) {
                        returnDesc = TemplateReturnDesc.RETURN_DESC_LOSEPOWER;
                    }
                    else {
                        if(chargeOpenTime != null && chargeFinishTime != null && (chargeFinishTime.getTime() - chargeOpenTime.getTime() < CHARGDURTIME_THRVAL1)) {
                            returnDesc = TemplateReturnDesc.RETURN_DESC_LOSEPOWER;
                        }
                    }
                }
                else if(StringUtils.equals(FinishReason.REASON_SURPLUS_ZERO, finishReason)) {
                    if(chargeOpenTime != null && chargeFinishTime != null && DoubleUtils.defaultIfNull(maxPower, 0.0D).compareTo(0.005D) <= 0 && (chargeFinishTime.getTime() - chargeOpenTime.getTime() < CHARGDURTIME_THRVAL2)) {
                        returnDesc = TemplateReturnDesc.RETURN_DESC_NOTINTIME;
                    }
                }
            }
            else {
                if(StringUtils.equals(finishReason, FinishReason.REASON_DEVICEOVERPOWER)) {
                    returnDesc = TemplateReturnDesc.RETURN_DESC_DEVICEOVERPOWER;
                }
                else {
                    returnDesc = TemplateReturnDesc.RETURN_DESC_COMMFAILURE;
                }
            }
        }
        return returnDesc;
    }

    private boolean checkToClosingChargeRecord(ChargeRecord chargeRecord) {
        return StringUtils.equals("L", chargeRecordDao.getClosingFlag(chargeRecord.getChargeRecordId(), true));
    }

    /**
     * 充电记录结算处理
     * @param chargeRecord
     *              充电记录
     * @return true / false
     */
    public static void main(String[] args) {
        BigDecimal up = new BigDecimal("0.655");
        if(1 == 1){
            up = up.add(new BigDecimal("0.2")).add(new BigDecimal("0.2"));
        }
        System.out.println(up);
        double closingMoney = new BigDecimal(0.039).multiply(up).setScale(2, BigDecimal.ROUND_HALF_UP).doubleValue();
        System.out.println(closingMoney);
    }

    private boolean closingChargeRecordHandle(ChargeRecord chargeRecord) throws Exception {
        boolean result = false;
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>>>>>> start closingChargeRecordHandle >>>>>>>>>> ");
            logger.debug(" ChargeRecord : " + JsonConvertUtils.convertToString(chargeRecord));
        }
        if(checkToClosingChargeRecord(chargeRecord)) {
            // 计算结算金额
            Integer chargeOpenFlag = IntegerUtils.defaultIfNull(chargeRecord.getChargeOpenFlag(), ChargeOpenFlag.FLAG_SUCCESS);
            if(logger.isDebugEnabled()) {
                logger.debug("chargeOpenFlag : " + chargeOpenFlag);
            }
            String schemeType = chargeRecord.getSchemeType();
            Integer currUnitPriceFlag = IntegerUtils.defaultIfNull(chargeRecord.getCurrUnitPriceFlag(), 1);
            if(currUnitPriceFlag < 1 || currUnitPriceFlag > 5) {
                currUnitPriceFlag = 1;
            }
            Double unitPrice1 = DoubleUtils.defaultIfNegValue(chargeRecord.getUnitPrice1(), 0.0D, 0.0D);
            Double unitPrice2 = DoubleUtils.defaultIfNegValue(chargeRecord.getUnitPrice2(), 0.0D, 0.0D);
            Double unitPrice3 = DoubleUtils.defaultIfNegValue(chargeRecord.getUnitPrice3(), 0.0D, 0.0D);
            Double unitPrice4 = DoubleUtils.defaultIfNegValue(chargeRecord.getUnitPrice4(), 0.0D, 0.0D);
            Double unitPrice5 = DoubleUtils.defaultIfNegValue(chargeRecord.getUnitPrice5(), 0.0D, 0.0D);
            Double chargeAmount = DoubleUtils.defaultIfNegValue(chargeRecord.getChargeAmount(), 0.0D, 0.0D);
            Double usageAmount = DoubleUtils.defaultIfNegValue(chargeRecord.getUsageAmount(), 0.0D, 0.0D);
            Double closingMoney = 0.0D;                             // 结算金额
            if(!isFullReturn(chargeRecord)) {
                if(StringUtils.equals("02", schemeType)) {
                    // 计电量
                    if(unitPrice1.compareTo(0.0D) > 0) {
                        // 计电量退费金额为使用电量/（单价+服务费+损耗费）
                        //closingMoney = new BigDecimal(usageAmount).divide(new BigDecimal(unitPrice1), 2, BigDecimal.ROUND_HALF_UP).doubleValue();
                        SchemeDetail sd = JsonConvertUtils.convertFromString(SchemeDetail.class,chargeRecord.getSchemeDetails());
                        BigDecimal up = new BigDecimal(unitPrice1.toString());
                        if(sd != null){
                            up = up.add(new BigDecimal(sd.getCharge_service_fee())).add(new BigDecimal(sd.getElec_loss_fee()));
                        }
                        closingMoney = new BigDecimal(usageAmount).multiply(up).setScale(2, BigDecimal.ROUND_HALF_UP).doubleValue();
                        if(logger.isDebugEnabled()) {
                            logger.debug(" usageAmount          : " + usageAmount);
                            logger.debug(" unitPrice            : " + unitPrice1);
                            logger.debug(" closingMoney         : " + closingMoney);
                        }
                    }
                }
                else if(StringUtils.equals("03", schemeType)) {
                    // 计次（按时间）
                    closingMoney = new BigDecimal(chargeAmount).setScale(2, BigDecimal.ROUND_HALF_UP).doubleValue();
                }else if(StringUtils.equals("05", schemeType)) {
                    // 计电量+计时(阶梯功率)
                    if(unitPrice1.compareTo(0.0D) > 0) {
                        // 计电量+计时(阶梯功率) 金额为使用电量/（单价+服务费）
                        SchemeDetail sd = JsonConvertUtils.convertFromString(SchemeDetail.class,chargeRecord.getSchemeDetails());
                        //电度电费
                        BigDecimal up = new BigDecimal(unitPrice1.toString());
                        if(sd != null&&sd.getCharge_service_fee()!=null){
                            up = up.add(new BigDecimal(sd.getCharge_service_fee()));
                        }else {
                            up = up.add(new BigDecimal(unitPrice2.toString()));
                        }
                        closingMoney = new BigDecimal(usageAmount).multiply(up).setScale(2, BigDecimal.ROUND_HALF_UP).doubleValue();
                    }
                }
                else {
                    // 计时间
                    Integer closingPeriod = IntegerUtils.defaultIfNegValue(chargeRecord.getClosingPeriod(), 0, 0);
                    Double unitPrice = 0.0D;
                    if(currUnitPriceFlag == 1) {
                        unitPrice = unitPrice1;
                    }
                    else if(currUnitPriceFlag == 2) {
                        unitPrice = unitPrice2;
                    }
                    else if(currUnitPriceFlag == 3) {
                        unitPrice = unitPrice3;
                    }
                    else if(currUnitPriceFlag == 4) {
                        unitPrice = unitPrice4;
                    }
                    else if(currUnitPriceFlag == 5) {
                        unitPrice = unitPrice5;
                    }
                    if(unitPrice.compareTo(0.0D) > 0) {
                        if(closingPeriod > 0) {
                            int closingPeriodCount = usageAmount.intValue() / closingPeriod;
                            int mod = usageAmount.intValue() % closingPeriod;
                            if(mod > 0) {
                                closingPeriodCount = closingPeriodCount + 1;
                            }
                            Double closingAmount = closingPeriodCount * closingPeriod * 1.0D;
                            closingAmount = closingAmount < chargeAmount ? closingAmount : chargeAmount;
                            closingMoney = new BigDecimal(closingAmount).divide(new BigDecimal(unitPrice * 60.0D), 2, BigDecimal.ROUND_HALF_UP).doubleValue();
                            if(logger.isDebugEnabled()) {
                                logger.debug(" mod                      : " + mod);
                                logger.debug(" closingPeriodCount       : " + closingPeriodCount);
                                logger.debug(" closingPeriod            : " + closingPeriod);
                                logger.debug(" closingAmount            : " + closingAmount);
                            }
                        }
                        else {
                            closingMoney = new BigDecimal(usageAmount).divide(new BigDecimal(unitPrice * 60.0D), 2, BigDecimal.ROUND_HALF_UP).doubleValue();
                        }
                        if(logger.isDebugEnabled()) {
                            logger.debug(" usageAmount          : " + usageAmount);
                            logger.debug(" unitPrice            : " + unitPrice);
                            logger.debug(" unitPrice * 60.0D    : " + new BigDecimal(unitPrice * 60.0D).doubleValue());
                            logger.debug(" closingMoney         : " + closingMoney);
                        }
                    }
                }
            }
            chargeRecord.setClosingFlag("Y");
            chargeRecord.setClosingMoney(closingMoney);
            chargeRecord.setClosingTime(new Date());
            chargeRecordDao.finshClosingChargeRecord(chargeRecord);
            result = true;
        }

        if(logger.isDebugEnabled()) {
            logger.debug(" <<<<<<<<<<< end closingChargeRecordHandle <<<<<<<<<<< ");
        }
        return result;
    }

    private boolean checkToReturnChargeRecord(ChargeRecord chargeRecord) {
        return StringUtils.equals("L", chargeRecordDao.getReturnFlag(chargeRecord.getChargeRecordId(), true));
    }

    /**
     * 充电记录退费及分配处理
     * @param chargeRecord
     *              充电记录
     * @return true / false
     */
    private boolean returnChargeRecordHandle(ChargeRecord chargeRecord) throws Exception {
        boolean result = false;
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>>>>>> start returnChargeRecordHandle >>>>>>>>>> ");
            logger.debug(" ChargeRecord         : " + JsonConvertUtils.convertToString(chargeRecord));
        }
        if(checkToReturnChargeRecord(chargeRecord)) {
            // 判断是否需要退费
            boolean needReturn = false;
            if(isFullReturn(chargeRecord)) {
                needReturn = true;
            }
            else {
                if(StringUtils.notEquals(FinishReason.REASON_STOPWAITTIMEOUT, chargeRecord.getFinishReason()) && StringUtils.equals("Y", chargeRecord.getIsReturn())) {
                    needReturn = true;
                }
            }

            String enterUserType = chargeRecord.getEnterUserType();                                                                             // 入账账户类型
            String enterUserId = chargeRecord.getEnterUserId();                                                                                 // 入账账户标识
            Double chargeMoney = DoubleUtils.defaultIfNegValue(chargeRecord.getChargeMoney(), 0.0D, 0.0D);          // 充电金额，单位为元
            Double closingMoney = DoubleUtils.defaultIfNegValue(chargeRecord.getClosingMoney(), 0.0D, 0.0D);        // 结算金额，单位为元
            Double serviceRate = DoubleUtils.defaultIfNegValue(chargeRecord.getServiceRate(), 0.0D, 0.0D);          // 服务费率，单位为%
            if(logger.isDebugEnabled()) {
                logger.debug("enterUserType : " + enterUserType);
                logger.debug("enterUserId   : " + enterUserId);
                logger.debug("chargeMoney   : " + chargeMoney);
                logger.debug("closingMoney  : " + closingMoney);
                logger.debug("serviceRate   : " + serviceRate);
            }

            Double returnMoney = 0.0D;
            if(needReturn) {
                // 计算退费金额
                if(chargeMoney > closingMoney) {
                    returnMoney = new BigDecimal(chargeMoney - closingMoney).setScale(2, BigDecimal.ROUND_HALF_UP).doubleValue();
                    chargeRecord.setReturnFlag("F");
                    chargeRecord.setReturnMoney(returnMoney);
                    chargeRecord.setTotalReturnMoney(returnMoney);
                    chargeRecord.setReturnTime(new Date());
                    chargeRecord.setReturnDesc("退费失败");

                    // 退费处理
                    TradeOrderRefundRequest tradeOrderRefundRequest = new TradeOrderRefundRequest();
                    tradeOrderRefundRequest.setOrderId(chargeRecord.getChargeTradeOrderId());
                    tradeOrderRefundRequest.setRefundNo(UUID.randomUUID().toString().replaceAll("-", "").toLowerCase());
                    tradeOrderRefundRequest.setRefundMoney(returnMoney);
                    TradeOrderRefundResponse tradeOrderRefundResponse = tradeOrderService.refundTradeOrder(tradeOrderRefundRequest);
                    if(tradeOrderRefundResponse != null && ResponseResultEnum.SUCCESS.getResult().equals(tradeOrderRefundResponse.getResult())) {
                        //
                        TradeOrderRefundDo tradeOrderRefund = tradeOrderRefundResponse.getTradeOrderRefund();
                        if(RefundResultEnum.REFUND_NOTIFY_WAITING.getResult().equals(tradeOrderRefund.getRefundResult())) {
                            // 等待通知
                            chargeRecord.setReturnFlag("W");
                            chargeRecord.setReturnDesc(isFullReturn(chargeRecord) ? getFullReturnDesc(chargeRecord) : TemplateReturnDesc.RETURN_DESC_BSISRETUR);
                            chargeRecord.setReturnTradeOrderId(tradeOrderRefund.getRefundNo());
                        }
                        else if(RefundResultEnum.REFUND_SUCCESS.getResult().equals(tradeOrderRefund.getRefundResult())) {
                            // 退款成功
                            chargeRecord.setReturnFlag("Y");
                            chargeRecord.setReturnDesc(isFullReturn(chargeRecord) ? getFullReturnDesc(chargeRecord) : TemplateReturnDesc.RETURN_DESC_BSISRETUR);
                            chargeRecord.setReturnTradeOrderId(tradeOrderRefund.getRefundNo());
                            chargeRecord.setReturnSuccessTime(new Date());
                        }
                        else if(RefundResultEnum.REFUND_FAILURE.getResult().equals(tradeOrderRefund.getRefundResult())) {
                            // 退款失败
                            chargeRecord.setReturnFlag("F");
                            chargeRecord.setReturnDesc(isFullReturn(chargeRecord) ? getFullReturnDesc(chargeRecord) : TemplateReturnDesc.RETURN_DESC_BSISRETUR);
                            chargeRecord.setReturnTradeOrderId(tradeOrderRefund.getRefundNo());
                        }
                    }
                    else if(tradeOrderRefundResponse != null && ResponseResultEnum.FAILURE.getResult().equals(tradeOrderRefundResponse.getResult())) {
                        // 退款失败
                        chargeRecord.setReturnFlag("F");
                        chargeRecord.setReturnDesc((isFullReturn(chargeRecord) ? getFullReturnDesc(chargeRecord) : TemplateReturnDesc.RETURN_DESC_BSISRETUR) + "（" + tradeOrderRefundResponse.getFailureCode() + ":" + tradeOrderRefundResponse.getFailureMsg() + "）");
                    }
                }
                else {
                    chargeRecord.setReturnFlag("D");
                    chargeRecord.setReturnMoney(returnMoney);
                    chargeRecord.setReturnTime(new Date());
                    chargeRecord.setReturnDesc("无需退费" + (chargeMoney.compareTo(0.0D) > 0 ? " : 充电金额小于等于结算金额" : " : 免费充电"));
                }
            }
            else {
                chargeRecord.setReturnFlag("D");
                chargeRecord.setReturnMoney(returnMoney);
                chargeRecord.setReturnTime(new Date());
                chargeRecord.setReturnDesc("无需退费" + (StringUtils.equals("N", chargeRecord.getIsReturn()) ? " : 计费方案为不退费" : ""));
            }

            // 计算收入、成本、收益、服务费、收入（除去服务费）、收益（除去服务费）
            Double usageElecCons = DoubleUtils.defaultIfNegValue(chargeRecord.getUsageElecCons(), 0.0D, 0.0D);
            Double costUnitPrice = DoubleUtils.defaultIfNegValue(chargeRecord.getCostUnitPrice(), 0.0D, 0.0D);
            Double income = new BigDecimal(chargeMoney - returnMoney).setScale(2, BigDecimal.ROUND_HALF_UP).doubleValue();
            Double cost = new BigDecimal(usageElecCons * costUnitPrice).setScale(2, BigDecimal.ROUND_HALF_UP).doubleValue();
            Double profit = new BigDecimal(income - cost).setScale(2, BigDecimal.ROUND_HALF_UP).doubleValue();
            if(serviceRate.compareTo(100.0D) > 0) {
                serviceRate = 100.0D;
            }
            Double serviceMoney = new BigDecimal(income * serviceRate / 100.0D).setScale(2, BigDecimal.ROUND_HALF_UP).doubleValue();
            Double incomeExService = new BigDecimal(income - serviceMoney).setScale(2, BigDecimal.ROUND_HALF_UP).doubleValue();
            Double profitExService = new BigDecimal(profit - serviceMoney).setScale(2, BigDecimal.ROUND_HALF_UP).doubleValue();
            chargeRecord.setIncome(income);
            chargeRecord.setCost(cost);
            chargeRecord.setProfit(profit);
            chargeRecord.setServiceMoney(serviceMoney);
            chargeRecord.setIncomeExService(incomeExService);
            chargeRecord.setProfitExService(profitExService);

            if(serviceMoney.compareTo(0.0D) > 0 && StringUtils.equals(EB_TRANSIT_USERTYPE, enterUserType) && StringUtils.equals(EB_TRANSIT_USERID, enterUserId)) {
                chargeRecord.setServiceTime(new Date());
                // 服务费收取
                // 先创建订单
                TradeOrderCreateRequest tradeOrderCreateRequest = new TradeOrderCreateRequest();
                tradeOrderCreateRequest.setUserId(EB_SERVICE_USERID);
                tradeOrderCreateRequest.setUserType(EB_SERVICE_USERTYPE);
                tradeOrderCreateRequest.setCounterUserId(EB_TRANSIT_USERID);
                tradeOrderCreateRequest.setCounterUserType(EB_TRANSIT_USERTYPE);
                tradeOrderCreateRequest.setOrderMoney(serviceMoney);
                tradeOrderCreateRequest.setOrderType(OrderTypeEnum.TRADE.getCode());
                tradeOrderCreateRequest.setBusinessModule("ebike_service");
                tradeOrderCreateRequest.setBusinessId(chargeRecord.getChargeRecordId());
                tradeOrderCreateRequest.setOrderRemark("电动自行车充电服务费");
                TradeOrderCreateResponse tradeOrderCreateResponse = tradeOrderService.createTradeOrder(tradeOrderCreateRequest);
                if(tradeOrderCreateResponse != null && ResponseResultEnum.SUCCESS.getResult().equals(tradeOrderCreateResponse.getResult())) {   // 创建交易订单成功
                    TradeOrderDo tradeOrder = tradeOrderCreateResponse.getTradeOrder();
                    String serviceTradeOrderId = tradeOrder.getOrderId();
                    if(logger.isDebugEnabled()) {
                        logger.debug(" [ 生成服务费交易订单ID ] : " + serviceTradeOrderId);
                    }
                    TradeOrderPayRequest tradeOrderPayRequest = new TradeOrderPayRequest();
                    tradeOrderPayRequest.setOrderId(serviceTradeOrderId);
                    tradeOrderPayRequest.setPayMoney(tradeOrder.getOrderMoney());
                    tradeOrderPayRequest.setPayType(PayTypeEnum.INNERACCT.getCode());
                    TradeOrderPayResponse tradeOrderPayResponse = tradeOrderService.payTradeOrder(tradeOrderPayRequest);
                    if(ResponseResultEnum.SUCCESS.getResult().equals(tradeOrderPayResponse.getResult()) && PayResultEnum.PAY_SUCCESS.getResult().equals(tradeOrderPayResponse.getTradeOrderPay().getPayResult())) {  //
                        if(logger.isDebugEnabled()) {
                            logger.debug(" [ 电动自行车充电服务费收取成功 ] : " + serviceTradeOrderId);
                        }
                        chargeRecord.setServiceFlag("S");
                        chargeRecord.setServiceDesc("电动自行车充电服务费收取成功");
                        chargeRecord.setServiceTradeOrderId(serviceTradeOrderId);
                        chargeRecord.setServiceSuccessTime(new Date());
                    }
                    else {
                        // 电动自行车充电服务费收取失败
                        if(logger.isDebugEnabled()) {
                            logger.debug(" [ 电动自行车充电服务费收取失败 ] : " + serviceTradeOrderId);
                        }
                        chargeRecord.setServiceFlag("F");
                        chargeRecord.setServiceDesc("电动自行车充电服务费收取失败");
                        chargeRecord.setServiceTradeOrderId(serviceTradeOrderId);
                    }
                }
                else if(tradeOrderCreateResponse != null && ResponseResultEnum.FAILURE.getResult().equals(tradeOrderCreateResponse.getResult())) {
                    chargeRecord.setServiceFlag("F");
                    chargeRecord.setServiceDesc("电动自行车充电服务费收取失败 : 未知原因（" + tradeOrderCreateResponse.getFailureCode() + ":" + tradeOrderCreateResponse.getFailureMsg() + "）");
                }
            }
            else {
                chargeRecord.setServiceFlag("N");           // 无服务费
                chargeRecord.setServiceDesc("无服务费");
            }

            if(incomeExService.compareTo(0.0D) > 0 && StringUtils.equals(EB_TRANSIT_USERTYPE, enterUserType) && StringUtils.equals(EB_TRANSIT_USERID, enterUserId)) {
                // 收入（除去服务费）分配
                Double orgoprIncomeRatio = 100.0D;                      // 运营方收入分配比例，单位：%
                Double allocatedIncomeRatio = 0.0D;                     // 利益相关方已分配比例，单位：%
                List<ChargeRecordIncomeDetailInfo> cridInfoList = new ArrayList<>();
                // 获取利益相关方分配方案信息
                List<CsshDistSchemeInfo> cdsInfoList = csshDistschemeDao.getCsshDistSchemeInfoList(chargeRecord.getChargestationId());
                int distIndex = 0;
                if(CollectionUtils.sizeIsNotEmptyIgnoreNull(cdsInfoList)) {
                    // 利益相关方收入分配
                    for(CsshDistSchemeInfo cdsInfo : cdsInfoList) {
                        if(logger.isDebugEnabled()) {
                            logger.debug(" CsshDistSchemeInfo : " + JsonConvertUtils.convertToString(cdsInfo));
                        }
                        //
                        // 利益相关方收入分配处理
                        //
                        Double distRatio = DoubleUtils.defaultIfNegValue(cdsInfo.getIncomeRatio(), 0.0D, 0.0D);
                        if(distRatio + allocatedIncomeRatio > 100.0D) {
                            distRatio = 100.0D - allocatedIncomeRatio;
                        }
                        Double distMoney = new BigDecimal(incomeExService * distRatio / 100.0D).setScale(2, BigDecimal.ROUND_HALF_UP).doubleValue();
                        allocatedIncomeRatio = distRatio + allocatedIncomeRatio;
                        distIndex++;
                        ChargeRecordIncomeDetailInfo cridInfo = new ChargeRecordIncomeDetailInfo();
                        cridInfo.setChargeRecordId(chargeRecord.getChargeRecordId());
                        cridInfo.setDistSn(distIndex);
                        cridInfo.setDistUserType("platform_org");
                        cridInfo.setDistUserId(cdsInfo.getStakeholderOrgNo());
                        cridInfo.setDistRatio(distRatio);
                        cridInfo.setDistMoney(distMoney);
                        cridInfo.setDistTime(new Date());
                        // 先创建订单
                        TradeOrderCreateRequest tradeOrderCreateRequest = new TradeOrderCreateRequest();
                        tradeOrderCreateRequest.setUserId(cridInfo.getDistUserId());
                        tradeOrderCreateRequest.setUserType(cridInfo.getDistUserType());
                        tradeOrderCreateRequest.setCounterUserId(EB_TRANSIT_USERID);
                        tradeOrderCreateRequest.setCounterUserType(EB_TRANSIT_USERTYPE);
                        tradeOrderCreateRequest.setOrderMoney(cridInfo.getDistMoney());
                        tradeOrderCreateRequest.setOrderType(OrderTypeEnum.TRADE.getCode());
                        tradeOrderCreateRequest.setBusinessModule("ebike_distincome");
                        tradeOrderCreateRequest.setBusinessId(chargeRecord.getChargeRecordId() + "_2_" + cridInfo.getDistSn() + ":" + cdsInfo.getStakeholderOrgNo());
                        tradeOrderCreateRequest.setOrderRemark("电动自行车充电收入分配（利益相关方）");
                        TradeOrderCreateResponse tradeOrderCreateResponse = tradeOrderService.createTradeOrder(tradeOrderCreateRequest);
                        if(tradeOrderCreateResponse != null && ResponseResultEnum.SUCCESS.getResult().equals(tradeOrderCreateResponse.getResult())) {   // 创建交易订单成功
                            TradeOrderDo tradeOrder = tradeOrderCreateResponse.getTradeOrder();
                            String distTradeOrderId = tradeOrder.getOrderId();
                            if(logger.isDebugEnabled()) {
                                logger.debug(" [ 生成收入分配交易订单ID ] : " + distTradeOrderId);
                            }
                            TradeOrderPayRequest tradeOrderPayRequest = new TradeOrderPayRequest();
                            tradeOrderPayRequest.setOrderId(distTradeOrderId);
                            tradeOrderPayRequest.setPayMoney(tradeOrder.getOrderMoney());
                            tradeOrderPayRequest.setPayType(PayTypeEnum.INNERACCT.getCode());
                            TradeOrderPayResponse tradeOrderPayResponse = tradeOrderService.payTradeOrder(tradeOrderPayRequest);
                            if(ResponseResultEnum.SUCCESS.getResult().equals(tradeOrderPayResponse.getResult()) && PayResultEnum.PAY_SUCCESS.getResult().equals(tradeOrderPayResponse.getTradeOrderPay().getPayResult())) {  //
                                // 电动自行车充电收入分配（利益相关方）成功
                                if(logger.isDebugEnabled()) {
                                    logger.debug(" [ 电动自行车充电收入分配（利益相关方）成功 ] : " + distTradeOrderId);
                                }
                                cridInfo.setDistFlag("S");
                                cridInfo.setDistDesc("电动自行车充电收入分配（利益相关方）成功");
                                cridInfo.setDistTradeOrderId(distTradeOrderId);
                                cridInfo.setDistSuccessTime(new Date());
                            }
                            else {
                                // 电动自行车充电收入分配（利益相关方）失败
                                if(logger.isDebugEnabled()) {
                                    logger.debug(" [ 电动自行车充电收入分配（利益相关方）失败 ] : " + distTradeOrderId);
                                }
                                cridInfo.setDistFlag("F");
                                cridInfo.setDistDesc("电动自行车充电收入分配（利益相关方）失败");
                                cridInfo.setDistTradeOrderId(distTradeOrderId);
                            }
                        }
                        else if(tradeOrderCreateResponse != null && ResponseResultEnum.FAILURE.getResult().equals(tradeOrderCreateResponse.getResult())) {
                            cridInfo.setDistFlag("F");
                            cridInfo.setDistDesc("电动自行车充电收入分配（利益相关方）失败 : 原因（" + tradeOrderCreateResponse.getFailureCode() + ":" + tradeOrderCreateResponse.getFailureMsg() + "）");
                        }
                        cridInfoList.add(cridInfo);
                    }
                }
                // 运营方收入分配
                orgoprIncomeRatio = orgoprIncomeRatio - allocatedIncomeRatio;
                Double orgoprIncomeMoney = new BigDecimal(incomeExService * orgoprIncomeRatio / 100.0D).setScale(2, BigDecimal.ROUND_HALF_UP).doubleValue();
                distIndex++;
                ChargeRecordIncomeDetailInfo orgoprCridInfo = new ChargeRecordIncomeDetailInfo();
                orgoprCridInfo.setChargeRecordId(chargeRecord.getChargeRecordId());
                orgoprCridInfo.setDistSn(distIndex);
                orgoprCridInfo.setDistUserType("platform_org");
                orgoprCridInfo.setDistUserId(chargeRecord.getOrgNo());
                orgoprCridInfo.setDistRatio(orgoprIncomeRatio);
                orgoprCridInfo.setDistMoney(orgoprIncomeMoney);
                orgoprCridInfo.setDistTime(new Date());
                // 先创建订单
                TradeOrderCreateRequest tradeOrderCreateRequest = new TradeOrderCreateRequest();
                tradeOrderCreateRequest.setUserId(orgoprCridInfo.getDistUserId());
                tradeOrderCreateRequest.setUserType(orgoprCridInfo.getDistUserType());
                tradeOrderCreateRequest.setCounterUserId(EB_TRANSIT_USERID);
                tradeOrderCreateRequest.setCounterUserType(EB_TRANSIT_USERTYPE);
                tradeOrderCreateRequest.setOrderMoney(orgoprCridInfo.getDistMoney());
                tradeOrderCreateRequest.setOrderType(OrderTypeEnum.TRADE.getCode());
                tradeOrderCreateRequest.setBusinessModule("ebike_distincome");
                tradeOrderCreateRequest.setBusinessId(chargeRecord.getChargeRecordId() + "_2_" + orgoprCridInfo.getDistSn() + ":" + chargeRecord.getOrgNo());
                tradeOrderCreateRequest.setOrderRemark("电动自行车充电收入分配（运营方）");
                TradeOrderCreateResponse tradeOrderCreateResponse = tradeOrderService.createTradeOrder(tradeOrderCreateRequest);
                if(tradeOrderCreateResponse != null && ResponseResultEnum.SUCCESS.getResult().equals(tradeOrderCreateResponse.getResult())) {   // 创建交易订单成功
                    TradeOrderDo tradeOrder = tradeOrderCreateResponse.getTradeOrder();
                    String orgoprDistTradeOrderId = tradeOrder.getOrderId();
                    if(logger.isDebugEnabled()) {
                        logger.debug(" [ 生成收入分配交易订单ID ] : " + orgoprDistTradeOrderId);
                    }
                    TradeOrderPayRequest tradeOrderPayRequest = new TradeOrderPayRequest();
                    tradeOrderPayRequest.setOrderId(orgoprDistTradeOrderId);
                    tradeOrderPayRequest.setPayMoney(tradeOrder.getOrderMoney());
                    tradeOrderPayRequest.setPayType(PayTypeEnum.INNERACCT.getCode());
                    TradeOrderPayResponse tradeOrderPayResponse = tradeOrderService.payTradeOrder(tradeOrderPayRequest);
                    if(ResponseResultEnum.SUCCESS.getResult().equals(tradeOrderPayResponse.getResult()) && PayResultEnum.PAY_SUCCESS.getResult().equals(tradeOrderPayResponse.getTradeOrderPay().getPayResult())) {  //
                        // 电动自行车充电收入分配（运营方）成功
                        if(logger.isDebugEnabled()) {
                            logger.debug(" [ 电动自行车充电收入分配（运营方）成功 ] : " + orgoprDistTradeOrderId);
                        }
                        orgoprCridInfo.setDistFlag("S");
                        orgoprCridInfo.setDistDesc("电动自行车充电收入分配（运营方）成功");
                        orgoprCridInfo.setDistTradeOrderId(orgoprDistTradeOrderId);
                        orgoprCridInfo.setDistSuccessTime(new Date());
                    }
                    else {
                        // 电动自行车充电收入分配（运营方）失败
                        if(logger.isDebugEnabled()) {
                            logger.debug(" [ 电动自行车充电收入分配（运营方）失败 ] : " + orgoprDistTradeOrderId);
                        }
                        orgoprCridInfo.setDistFlag("F");
                        orgoprCridInfo.setDistDesc("电动自行车充电收入分配（运营方）失败");
                        orgoprCridInfo.setDistTradeOrderId(orgoprDistTradeOrderId);
                    }
                }
                else if(tradeOrderCreateResponse != null && ResponseResultEnum.FAILURE.getResult().equals(tradeOrderCreateResponse.getResult())) {
                    orgoprCridInfo.setDistFlag("F");
                    orgoprCridInfo.setDistDesc("电动自行车充电收入分配（运营方）失败 : 原因（" + tradeOrderCreateResponse.getFailureCode() + ":" + tradeOrderCreateResponse.getFailureMsg() + "）");
                }

                cridInfoList.add(orgoprCridInfo);

                // 插入历史充电记录收入分配明细表
                chargeRecordIncomeDetailDao.createChargeRecordIncomeDetailInfoList(cridInfoList);
            }

            chargeRecordDao.finshReturnChargeRecord(chargeRecord);
            if(StringUtils.equals(chargeRecord.getReturnFlag(), "Y")) {
                chargeOrderEventPublisher.publishEvent("return", chargeRecord);
            }
            result = true;
        }

        if(logger.isDebugEnabled()) {
            logger.debug(" <<<<<<<<<<< end returnChargeRecordHandle <<<<<<<<<<< ");
        }
        return result;
    }

    /**
     * 待结算充电记录扫描线程
     */
    private class ClosingHandleChargeRecordScanThread extends Thread {

        @Override
        public void run() {
            while(true) {
                try {
                    List<ChargeRecord> list = null;
                    try {
                        list = lockTobeClosingChargeRecordList();
                    }
                    catch(Exception e) {
                        logger.error(e.getMessage(), e.fillInStackTrace());
                    }
                    if(CollectionUtils.sizeIsEmptyIgnoreNull(list)) {
                        TimeUnit.SECONDS.sleep(INTERVAL_WAIT);
                    }
                    else {
                        synchronized(closingLinkedQueue) {
                            for(ChargeRecord chargeRecord : list) {
                                if(logger.isInfoEnabled()) {
                                    logger.info(" closing charge record scan   : " + chargeRecord.toString());
                                }
                                closingLinkedQueue.offer(chargeRecord);
                            }
                            closingLinkedQueue.notifyAll();
                        }
                    }
                }
                catch(InterruptedException _ie) {
                    logger.error(_ie.getMessage(), _ie.fillInStackTrace());
                    break;
                }
            }
        }

    }

    /**
     * 充电记录结算处理线程
     */
    private class ClosingChargeRecordHandleThread extends Thread {

        @Override
        public void run() {
            while(true) {
                try {
                    ChargeRecord chargeRecord = closingLinkedQueue.poll();
                    if(chargeRecord == null) {
                        synchronized(closingLinkedQueue) {
                            closingLinkedQueue.wait(100L);
                        }
                    }
                    else {
                        if(logger.isInfoEnabled()) {
                            logger.info(" closing charge record handle : " + chargeRecord.toString());
                        }
                        boolean b = false;
                        try {
                            b = closingChargeRecordHandle(chargeRecord);
                        }
                        catch(Exception e) {
                            logger.error(e.getMessage(), e.fillInStackTrace());
                            TimeUnit.SECONDS.sleep(INTERVAL_WAIT);
                        }
                        if(b) {
                            if(logger.isInfoEnabled()) {
                                logger.info(" closing charge record finish : " + chargeRecord.toString());
                            }
                            synchronized(returnLinkedQueue) {
                                List<ChargeRecord> crList = new ArrayList<>();
                                crList.add(chargeRecord);
                                chargeRecordDao.lockNotReturnChargeRecordList(crList);
                                if(logger.isInfoEnabled()) {
                                    logger.info(" return charge record scan   : " + chargeRecord.toString());
                                }
                                returnLinkedQueue.offer(chargeRecord);
                                returnLinkedQueue.notify();
                            }
                        }
                    }
                }
                catch(InterruptedException _ie) {
                    logger.error(_ie.getMessage(), _ie.fillInStackTrace());
                    break;
                }
            }
        }

    }

    /**
     * 待退费充电记录扫描线程
     */
    private class ReturnHandleChargeRecordScanThread extends Thread {

        @Override
        public void run() {
            while(true) {
                try {
                    List<ChargeRecord> list = null;
                    try {
                        list = lockTobeReturnChargeRecordList();
                    }
                    catch(Exception e) {
                        logger.error(e.getMessage(), e.fillInStackTrace());
                    }
                    if(CollectionUtils.sizeIsEmptyIgnoreNull(list)) {
                        TimeUnit.SECONDS.sleep(INTERVAL_WAIT);
                    }
                    else {
                        synchronized(returnLinkedQueue) {
                            for(ChargeRecord chargeRecord : list) {
                                if(logger.isInfoEnabled()) {
                                    logger.info(" return charge record scan   : " + chargeRecord.toString());
                                }
                                returnLinkedQueue.offer(chargeRecord);
                            }
                            returnLinkedQueue.notifyAll();
                        }
                    }
                }
                catch(InterruptedException _ie) {
                    logger.error(_ie.getMessage(), _ie.fillInStackTrace());
                    break;
                }
            }
        }

    }

    /**
     * 充电记录退费处理线程
     */
    private class ReturnChargeRecordHandleThread extends Thread {

        @Override
        public void run() {
            while(true) {
                try {
                    ChargeRecord chargeRecord = returnLinkedQueue.poll();
                    if(chargeRecord == null) {
                        synchronized(returnLinkedQueue) {
                            returnLinkedQueue.wait(500L);
                        }
                    }
                    else {
                        if(logger.isInfoEnabled()) {
                            logger.info(" return charge record handle   : " + chargeRecord.toString());
                            logger.info(" tradeOrderService             : " + tradeOrderService);
                        }
                        boolean b = false;
                        try {
                            b = returnChargeRecordHandle(chargeRecord);
                        }
                        catch(Exception e) {
                            logger.error(e.getMessage(), e.fillInStackTrace());
                            TimeUnit.SECONDS.sleep(INTERVAL_WAIT);
                        }
                        if(b) {
                            if(logger.isInfoEnabled()) {
                                logger.info(" return charge record finish : " + chargeRecord.toString());
                            }
                        }
                    }
                }
                catch(InterruptedException e) {
                    logger.error(e.getMessage(), e.fillInStackTrace());
                    break;
                }
            }
        }

    }

}
