package cn.yunrui.mqttclient.ebikesrv.handletask;

import cn.yunrui.mqttclient.ebikesrv.common.utils.CollectionUtils;
import cn.yunrui.mqttclient.ebikesrv.common.utils.DoubleUtils;
import cn.yunrui.mqttclient.ebikesrv.common.utils.JsonConvertUtils;
import cn.yunrui.mqttclient.ebikesrv.common.utils.StringUtils;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.dao.ChargeRecordDao;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.dao.ChargeRecordIncomeDetailDao;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.entity.ChargeRecord;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.entity.ChargeRecordIncomeDetailInfo;
import com.ebikepay.openservices.entity.RefundResultEnum;
import com.ebikepay.openservices.entity.TradeOrderRefundDo;
import com.ebikepay.openservices.request.TradeOrderRefundRequest;
import com.ebikepay.openservices.response.TradeOrderRefundResponse;
import com.ebikepay.openservices.service.TradeOrderService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.TimeUnit;

/**
 * 客户服务工单退款处理任务
 */
public class CscWorkorderRefundHandleTask {

    private final static String EB_TRANSIT_USERTYPE = "ebike_transit";          // 驿吧平台在途资金
    private final static String EB_TRANSIT_USERID = "10000000";                 // 驿吧平台在途资金

    private final Logger logger = LoggerFactory.getLogger(getClass());
    private final ConcurrentLinkedQueue<ChargeRecord> csrefundLinkedQueue = new ConcurrentLinkedQueue<>();
    private final static int INTERVAL_WAIT = 60;                                                             // 单位：秒

    private Thread csrefundHandleChargeRecordScanThread;
    private Thread csrefundChargeRecordHandleThread;

    @Resource(name = "tradeOrderService")
    private TradeOrderService tradeOrderService;

    @Resource(name = "chargeRecordDao")
    private ChargeRecordDao chargeRecordDao;

    @Resource(name = "chargeRecordIncomeDetailDao")
    private ChargeRecordIncomeDetailDao chargeRecordIncomeDetailDao;

    /**
     * 初始化
     */
    @PostConstruct
    public void init() {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> start init CscWorkorderRefundHandleTask >>>>>> ");
        }
        csrefundHandleChargeRecordScanThread = new CsrefundHandleChargeRecordScanThread();
        csrefundHandleChargeRecordScanThread.setDaemon(true);
        csrefundHandleChargeRecordScanThread.start();
        csrefundChargeRecordHandleThread = new CsrefundChargeRecordHandleThread();
        csrefundChargeRecordHandleThread.setDaemon(true);
        csrefundChargeRecordHandleThread.start();
        if(logger.isDebugEnabled()) {
            logger.debug(" <<<<<<< end init CscWorkorderRefundHandleTask <<<<<<< ");
        }
    }

    /**
     * 销毁
     */
    @PreDestroy
    public void destroy() {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> start destroy CscWorkorderRefundHandleTask >>>>>> ");
        }
        if(csrefundHandleChargeRecordScanThread != null) {
            try {
                csrefundHandleChargeRecordScanThread.join();
            }
            catch(InterruptedException _ie) {
                logger.error(_ie.getMessage(), _ie.fillInStackTrace());
            }
        }
        if(csrefundChargeRecordHandleThread != null) {
            try {
                csrefundChargeRecordHandleThread.join();
            }
            catch(InterruptedException _ie) {
                logger.error(_ie.getMessage(), _ie.fillInStackTrace());
            }
        }
        if(logger.isDebugEnabled()) {
            logger.debug(" <<<<<<< end destroy CscWorkorderRefundHandleTask <<<<<<< ");
        }
    }

    /**
     * 锁定等待退款充电记录
     * @return 等待退款充电记录
     */
    private List<ChargeRecord> lockTobeCsrefundChargeRecordList() throws Exception {
        List<ChargeRecord> crList = chargeRecordDao.getWaitingCsrefundChargeRecordList();
        if(CollectionUtils.sizeIsNotEmptyIgnoreNull(crList)) {
            chargeRecordDao.lockWaitingCsrefundChargeRecordList(crList);
        }
        return crList;
    }

    private boolean checkToCsrefundChargeRecord(ChargeRecord chargeRecord) {
        return StringUtils.equals("L", chargeRecordDao.getCsrefundResult(chargeRecord.getChargeRecordId(), true));
    }

    private boolean checkCscReturnHandleResult(ChargeRecord chargeRecord) {
        return StringUtils.equals("Y", chargeRecordDao.getCscReturnHandleResult(chargeRecord));
    }

    /**
     * 客户服务工单退款处理
     * @param chargeRecord
     *              充电记录
     * @return true / false
     */
    private boolean csrefundChargeRecordHandle(ChargeRecord chargeRecord) throws Exception {
        boolean result = false;
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>>>>>> start csrefundChargeRecordHandle >>>>>>>>>> ");
            logger.debug(" ChargeRecord : " + JsonConvertUtils.convertToString(chargeRecord));
        }
        if(checkToCsrefundChargeRecord(chargeRecord)) {
            // 判断是否需要退款
            boolean needCsrefund = false;
            if(checkCscReturnHandleResult(chargeRecord)) {
                needCsrefund = true;
            }

            String enterUserType = chargeRecord.getEnterUserType();                                                                             // 入账账户类型
            String enterUserId = chargeRecord.getEnterUserId();
            Double chargeMoney = DoubleUtils.defaultIfNegValue(chargeRecord.getChargeMoney(), 0.0D, 0.0D);
            Double totalReturnMoney = DoubleUtils.defaultIfNegValue(chargeRecord.getReturnMoney(), 0.0D, 0.0D);
            Double income = DoubleUtils.defaultIfNegValue(chargeRecord.getIncome(), 0.0D, 0.0D);
            Double serviceMoney = DoubleUtils.defaultIfNegValue(chargeRecord.getServiceMoney(), 0.0D, 0.0D);
            Double incomeExService = DoubleUtils.defaultIfNegValue(chargeRecord.getIncomeExService(), 0.0D, 0.0D);
            Double csrefundMoney = 0.0D;
            if(needCsrefund) {
                // 计算退款申请处理金额
                if(chargeMoney >= totalReturnMoney) {
                    csrefundMoney = new BigDecimal(chargeMoney - totalReturnMoney).setScale(2, BigDecimal.ROUND_HALF_UP).doubleValue();
                    totalReturnMoney = chargeMoney;
                    // 处理运营方和利益相关方分配收入退款
                    if(incomeExService.compareTo(0.0D) > 0 && StringUtils.equals(EB_TRANSIT_USERTYPE, enterUserType) && StringUtils.equals(EB_TRANSIT_USERID, enterUserId)) {
                        List<ChargeRecordIncomeDetailInfo> cridInfoList = chargeRecordIncomeDetailDao.getChargeRecordIncomeDetailInfoList(chargeRecord.getChargeRecordId());
                        for(ChargeRecordIncomeDetailInfo cridInfo : cridInfoList) {
                            TradeOrderRefundRequest distRefundRequest = new TradeOrderRefundRequest();
                            distRefundRequest.setOrderId(cridInfo.getDistTradeOrderId());
                            distRefundRequest.setRefundNo(UUID.randomUUID().toString().replaceAll("-", "").toLowerCase());
                            distRefundRequest.setRefundMoney(cridInfo.getDistMoney());
                            TradeOrderRefundResponse distRefundResponse = tradeOrderService.refundTradeOrder(distRefundRequest);
                            if(distRefundResponse.isSuccess()) {
                                TradeOrderRefundDo distTradeOrderRefund = distRefundResponse.getTradeOrderRefund();
                                String csrefundDistRefundNo = distRefundResponse.getTradeOrderRefund().getRefundNo();
                                cridInfo.setCsrefundResult("S");
                                cridInfo.setCsrefundMoney(distRefundRequest.getRefundMoney());
                                cridInfo.setCsrefundTime(new Date());
                                cridInfo.setCsrefundTradeOrderId(csrefundDistRefundNo);
                                if(RefundResultEnum.REFUND_SUCCESS.getResult().equals(distTradeOrderRefund.getRefundResult())) {
                                    // 退款成功
                                    cridInfo.setCsrefundResult("S");
                                    cridInfo.setCsrefundSuccessTime(new Date());
                                }
                                else if(RefundResultEnum.REFUND_FAILURE.getResult().equals(distTradeOrderRefund.getRefundResult())) {
                                    // 退款失败
                                    cridInfo.setCsrefundResult("F");
                                }
                            }
                            else {
                                cridInfo.setCsrefundResult("F");
                                cridInfo.setCsrefundMoney(distRefundRequest.getRefundMoney());
                                cridInfo.setCsrefundTime(new Date());
                                cridInfo.setCsrefundTradeOrderId(null);
                                cridInfo.setCsrefundSuccessTime(null);
                            }
                            chargeRecordIncomeDetailDao.updateChargeRecordIncomeDetailInfo(cridInfo);
                        }
                    }

                    // 处理服务费退款
                    if(serviceMoney.compareTo(0.0D) > 0 && StringUtils.equals(EB_TRANSIT_USERTYPE, enterUserType) && StringUtils.equals(EB_TRANSIT_USERID, enterUserId)) {
                        TradeOrderRefundRequest serviceRefundRequest = new TradeOrderRefundRequest();
                        serviceRefundRequest.setOrderId(chargeRecord.getServiceTradeOrderId());
                        serviceRefundRequest.setRefundNo(UUID.randomUUID().toString().replaceAll("-", "").toLowerCase());
                        serviceRefundRequest.setRefundMoney(serviceMoney);
                        TradeOrderRefundResponse serviceRefundResponse = tradeOrderService.refundTradeOrder(serviceRefundRequest);
                        if(serviceRefundResponse.isSuccess()) {
                            String csrefundServiceRefundNo = serviceRefundResponse.getTradeOrderRefund().getRefundNo();
                            chargeRecord.setCsrefundTradeOrderId(csrefundServiceRefundNo);
                        }
                    }

                    TradeOrderRefundRequest chargeRefundRequest = new TradeOrderRefundRequest();
                    chargeRefundRequest.setOrderId(chargeRecord.getChargeTradeOrderId());
                    chargeRefundRequest.setRefundNo(UUID.randomUUID().toString().replaceAll("-", "").toLowerCase());
                    chargeRefundRequest.setRefundMoney(income);
                    TradeOrderRefundResponse chargeRefundResponse = tradeOrderService.refundTradeOrder(chargeRefundRequest);
                    if(chargeRefundResponse.isSuccess()) {
                        TradeOrderRefundDo csrefundRefund = chargeRefundResponse.getTradeOrderRefund();
                        String csrefundRefundNo = csrefundRefund.getRefundNo();
                        chargeRecord.setCsrefundResult("S");
                        chargeRecord.setCsrefundMoney(csrefundMoney);
                        chargeRecord.setTotalReturnMoney(totalReturnMoney);
                        chargeRecord.setCsrefundTime(new Date());
                        chargeRecord.setCsrefundTradeOrderId(csrefundRefundNo);
                        if(RefundResultEnum.REFUND_NOTIFY_WAITING.getResult().equals(csrefundRefund.getRefundResult())) {
                            // T 退款处理，等待通知
                            chargeRecord.setCsrefundResult("T");
                        }
                        else if(RefundResultEnum.REFUND_SUCCESS.getResult().equals(csrefundRefund.getRefundResult())) {
                            // 退款成功
                            chargeRecord.setCsrefundResult("S");
                        }
                        else if(RefundResultEnum.REFUND_FAILURE.getResult().equals(csrefundRefund.getRefundResult())) {
                            // 退款失败
                            chargeRecord.setCsrefundResult("F");
                        }
                    }
                    else {
                        chargeRecord.setCsrefundResult("F");
                        chargeRecord.setCsrefundMoney(csrefundMoney);
                        chargeRecord.setTotalReturnMoney(totalReturnMoney);
                        chargeRecord.setCsrefundTime(new Date());
                    }

                }
                else {
                    chargeRecord.setCsrefundResult("F");
                    chargeRecord.setCsrefundMoney(csrefundMoney);
                    chargeRecord.setCsrefundTime(new Date());
                    chargeRecord.setRemark(StringUtils.isNotBlank(chargeRecord.getRemark()) ? chargeRecord.getRemark() + "\n" + "已经退款" : chargeRecord.getRemark());
                }
            }
            else {
                chargeRecord.setCsrefundResult("F");
                chargeRecord.setCsrefundMoney(csrefundMoney);
                chargeRecord.setCsrefundTime(new Date());
                chargeRecord.setRemark(StringUtils.isNotBlank(chargeRecord.getRemark()) ? chargeRecord.getRemark() + "\n" + "不予退款" : chargeRecord.getRemark());
            }

            chargeRecordDao.finshCsrefundChargeRecord(chargeRecord);
            result = true;
        }

        if(logger.isDebugEnabled()) {
            logger.debug(" <<<<<<<<<<< end csrefundChargeRecordHandle <<<<<<<<<<< ");
        }
        return result;
    }

    /**
     * 待客户服务工单退款充电记录扫描线程
     */
    private class CsrefundHandleChargeRecordScanThread extends Thread {

        @Override
        public void run() {
            while(true) {
                try {
                    List<ChargeRecord> list = null;
                    try {
                        list = lockTobeCsrefundChargeRecordList();
                    }
                    catch(Exception e) {
                        logger.error(e.getMessage(), e.fillInStackTrace());
                    }
                    if(CollectionUtils.sizeIsEmptyIgnoreNull(list)) {
                        TimeUnit.SECONDS.sleep(INTERVAL_WAIT);
                    }
                    else {
                        synchronized(csrefundLinkedQueue) {
                            for(ChargeRecord chargeRecord : list) {
                                if(logger.isInfoEnabled()) {
                                    logger.info(" csrefund charge record scan   : " + chargeRecord.toString());
                                }
                                csrefundLinkedQueue.offer(chargeRecord);
                            }
                            csrefundLinkedQueue.notifyAll();
                        }
                    }
                }
                catch(InterruptedException _ie) {
                    logger.error(_ie.getMessage(), _ie.fillInStackTrace());
                    break;
                }
            }
        }

    }

    /**
     * 充电记录客户服务工单退款处理线程
     */
    private class CsrefundChargeRecordHandleThread extends Thread {

        @Override
        public void run() {
            while(true) {
                try {
                    ChargeRecord chargeRecord = csrefundLinkedQueue.poll();
                    if(chargeRecord == null) {
                        synchronized(csrefundLinkedQueue) {
                            csrefundLinkedQueue.wait(100L);
                        }
                    }
                    else {
                        if(logger.isInfoEnabled()) {
                            logger.info(" csrefund charge record handle : " + chargeRecord.toString());
                        }
                        boolean b = false;
                        try {
                            b = csrefundChargeRecordHandle(chargeRecord);
                        }
                        catch(Exception e) {
                            logger.error(e.getMessage(), e.fillInStackTrace());
                            TimeUnit.SECONDS.sleep(INTERVAL_WAIT);
                        }
                        if(b) {
                            if(logger.isInfoEnabled()) {
                                logger.info(" csrefund charge record finish : " + chargeRecord.toString());
                            }
                        }
                    }
                }
                catch(InterruptedException _ie) {
                    logger.error(_ie.getMessage(), _ie.fillInStackTrace());
                    break;
                }
            }
        }

    }

}
