package cn.yunrui.mqttclient.ebikesrv.handletask;

import cn.yunrui.mqttclient.ebikesrv.mqttclient.cache.ChargeRecordCache;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.entity.ChargeRecord;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.service.EBikeChargeService;
import cn.yunrui.mqttclient.ebikesrv.common.utils.CollectionUtils;
import cn.yunrui.mqttclient.ebikesrv.common.utils.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.TimeUnit;

/**
 * 开启充电重试处理任务
 */
public class StartChargeRetryHandleTask {

    private final Logger logger = LoggerFactory.getLogger(getClass());
    private final ConcurrentLinkedQueue<ChargeRecord> linkedQueue = new ConcurrentLinkedQueue<>();
    private final static int INTERVAL_WAIT = 7;                                                             // 单位：秒

    private Thread startChargeRetryScanThread;
    private Thread startChargeRetryHandleThread;

    @Resource(name = "chargeRecordCache")
    private ChargeRecordCache chargeRecordCache;

    @Resource(name = "ebikeChargeService")
    private EBikeChargeService ebikeChargeService;

    /**
     * 初始化
     */
    @PostConstruct
    public void init() {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> start init StartChargeRetryHandleTask >>>>>> ");
        }
        startChargeRetryScanThread = new StartChargeRetryScanThread();
        startChargeRetryScanThread.setDaemon(true);
        startChargeRetryScanThread.start();
        startChargeRetryHandleThread = new StartChargeRetryHandleThread();
        startChargeRetryHandleThread.setDaemon(true);
        startChargeRetryHandleThread.start();
        if(logger.isDebugEnabled()) {
            logger.debug(" <<<<<<< end init StartChargeRetryHandleTask <<<<<<< ");
        }
    }

    /**
     * 销毁
     */
    @PreDestroy
    public void destroy() {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> start destroy StartChargeRetryHandleTask >>>>>> ");
        }
        if(startChargeRetryScanThread != null) {
            try {
                startChargeRetryScanThread.join();
            }
            catch(InterruptedException _ie) {
                logger.error(_ie.getMessage(), _ie.fillInStackTrace());
            }
        }
        if(startChargeRetryHandleThread != null) {
            try {
                startChargeRetryHandleThread.join();
            }
            catch(InterruptedException _ie) {
                logger.error(_ie.getMessage(), _ie.fillInStackTrace());
            }
        }
        if(logger.isDebugEnabled()) {
            logger.debug(" <<<<<<< end destroy StartChargeRetryHandleTask <<<<<<< ");
        }
    }

    /**
     * 获取需要重试开启充电充电记录信息
     * @return 需要重试开启充电充电记录信息
     */
    private List<ChargeRecord> getBeReopeningChargeRecordList() {
        List<ChargeRecord> chargeRecordList = chargeRecordCache.getBeReopeningChargeRecordList();
        if(CollectionUtils.sizeIsNotEmptyIgnoreNull(chargeRecordList)) {
            chargeRecordCache.lockBeReopeningChargeRecordList(chargeRecordList);
        }
        return chargeRecordList;
    }

    /**
     * 开始充电重试扫描线程
     */
    private class StartChargeRetryScanThread extends Thread {

        @Override
        public void run() {
            while(true) {
                try {
                    List<ChargeRecord> list = getBeReopeningChargeRecordList();
                    if(CollectionUtils.sizeIsEmptyIgnoreNull(list)) {
                        TimeUnit.SECONDS.sleep(INTERVAL_WAIT);
                    }
                    else {
                        synchronized(linkedQueue) {
                            for(ChargeRecord chargeRecord : list) {
                                if(logger.isInfoEnabled()) {
                                    logger.info(" start charge retry scan   : " + chargeRecord.toString());
                                }
                                linkedQueue.offer(chargeRecord);
                            }
                            linkedQueue.notifyAll();
                        }
                    }
                }
                catch(InterruptedException _ie) {
                    logger.error(_ie.getMessage(), _ie.fillInStackTrace());
                    break;
                }
            }
        }

    }

    /**
     * 开始充电重试处理线程
     */
    private class StartChargeRetryHandleThread extends Thread {

        @Override
        public void run() {
            while(true) {
                try {
                    ChargeRecord chargeRecord = linkedQueue.poll();
                    if(chargeRecord == null) {
                        synchronized(linkedQueue) {
                            linkedQueue.wait(100L);
                        }
                    }
                    else {
                        if(logger.isInfoEnabled()) {
                            logger.info(" start charge retry handle : " + chargeRecord.toString());
                        }
                        if(StringUtils.isNotBlank(chargeRecord.getDeviceId()) && StringUtils.isNotBlank(chargeRecord.getPlugId())) {
                            Map<String, Object> valueMap = chargeRecordCache.getChargeRecordMap(chargeRecord.getDeviceId(), chargeRecord.getPlugId());
                            Double remainElecCons = (Double) valueMap.get("remainElecCons");
                            //Double remainTime = (Double) valueMap.get("remainTime");
                            // 适配佳和新充电桩---下发超过999分钟，下发的值就是999
                            long remainTime = Math.round((Double) valueMap.get("remainTime"));
                            if(remainTime > 999){
                                remainTime = 999;
                            }
                            String msg2Device = "start" + "#" + chargeRecord.getPlugId() + "#" + Math.round(remainElecCons * 1000.0D) + "#" + remainTime;
                            logger.info("=======重试任务-----------------开始充电报文========= : " + msg2Device);
                            valueMap.put("reopeningFlag", "N");
                            valueMap.put("reopens", ((Integer) valueMap.get("reopens") + 1));
                            chargeRecordCache.updateChargeRecord(chargeRecord.getDeviceId(), chargeRecord.getPlugId(), valueMap);
                            ebikeChargeService.publish2device(chargeRecord.getDeviceId(), msg2Device, "start");
                        }
                    }
                }
                catch(InterruptedException _ie) {
                    logger.error(_ie.getMessage(), _ie.fillInStackTrace());
                    break;
                }
            }
        }

    }

}
