package cn.yunrui.mqttclient.ebikesrv.handletask;

import cn.yunrui.mqttclient.ebikesrv.mqttclient.cache.ChargeRecordCache;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.constant.FinishReason;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.entity.ChargeRecord;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.service.EBikeChargeService;
import cn.yunrui.mqttclient.ebikesrv.common.utils.CollectionUtils;
import cn.yunrui.mqttclient.ebikesrv.common.utils.IntegerUtils;
import cn.yunrui.mqttclient.ebikesrv.common.utils.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.annotation.Resource;
import java.util.List;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.TimeUnit;

/**
 * 结束充电超时处理任务
 */
public class StopChargeTimeoutHandleTask {

    private final Logger logger = LoggerFactory.getLogger(getClass());
    private final ConcurrentLinkedQueue<ChargeRecord> linkedQueue = new ConcurrentLinkedQueue<>();
    private final static int INTERVAL_WAIT = 30;                                                             // 单位：秒

    private Thread stopChargeTimeoutScanThread;
    private Thread stopChargeTimeoutHandleThread;

    @Resource(name = "chargeRecordCache")
    private ChargeRecordCache chargeRecordCache;

    @Resource(name = "ebikeChargeService")
    private EBikeChargeService ebikeChargeService;

    /**
     * 初始化
     */
    @PostConstruct
    public void init() {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> start init StopChargeTimeoutHandleTask >>>>>> ");
        }
        stopChargeTimeoutScanThread = new StopChargeTimeoutScanThread();
        stopChargeTimeoutScanThread.setDaemon(true);
        stopChargeTimeoutScanThread.start();
        stopChargeTimeoutHandleThread = new StopChargeTimeoutHandleThread();
        stopChargeTimeoutHandleThread.setDaemon(true);
        stopChargeTimeoutHandleThread.start();
        if(logger.isDebugEnabled()) {
            logger.debug(" <<<<<<< end init StopChargeTimeoutHandleTask <<<<<<< ");
        }
    }

    /**
     * 销毁
     */
    @PreDestroy
    public void destroy() {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> start destroy StopChargeTimeoutHandleTask >>>>>> ");
        }
        if(stopChargeTimeoutScanThread != null) {
            try {
                stopChargeTimeoutScanThread.join();
            }
            catch(InterruptedException _ie) {
                logger.error(_ie.getMessage(), _ie.fillInStackTrace());
            }
        }
        if(stopChargeTimeoutHandleThread != null) {
            try {
                stopChargeTimeoutHandleThread.join();
            }
            catch(InterruptedException _ie) {
                logger.error(_ie.getMessage(), _ie.fillInStackTrace());
            }
        }
        if(logger.isDebugEnabled()) {
            logger.debug(" <<<<<<< end destroy StopChargeTimeoutHandleTask <<<<<<< ");
        }
    }

    /**
     * 获取结束超时充电记录信息
     * @return 结束超时充电记录信息
     */
    private List<ChargeRecord> getWaitingStopTimeoutChargeRecordList() {
        List<ChargeRecord> chargeRecordList = chargeRecordCache.getWaitingStopTimeoutChargeRecordList();
        if(CollectionUtils.sizeIsNotEmptyIgnoreNull(chargeRecordList)) {
            chargeRecordCache.lockWaitingTimeoutChargeRecordList(chargeRecordList);
        }
        return chargeRecordList;
    }

    /**
     * 结束充电等待超时扫描线程
     */
    private class StopChargeTimeoutScanThread extends Thread {

        @Override
        public void run() {
            while(true) {
                try {
                    List<ChargeRecord> list = getWaitingStopTimeoutChargeRecordList();
                    if(CollectionUtils.sizeIsEmptyIgnoreNull(list)) {
                        TimeUnit.SECONDS.sleep(INTERVAL_WAIT);
                    }
                    else {
                        synchronized(linkedQueue) {
                            for(ChargeRecord chargeRecord : list) {
                                if(logger.isInfoEnabled()) {
                                    logger.info(" stop charge timeout scan   : " + chargeRecord.toString());
                                }
                                linkedQueue.offer(chargeRecord);
                            }
                            linkedQueue.notifyAll();
                        }
                    }
                }
                catch(InterruptedException _ie) {
                    logger.error(_ie.getMessage(), _ie.fillInStackTrace());
                    break;
                }
            }
        }

    }

    /**
     * 结束充电等待超时处理线程
     */
    private class StopChargeTimeoutHandleThread extends Thread {

        @Override
        public void run() {
            while(true) {
                try {
                    ChargeRecord chargeRecord = linkedQueue.poll();
                    if(chargeRecord == null) {
                        synchronized(linkedQueue) {
                            linkedQueue.wait(100L);
                        }
                    }
                    else {
                        if(logger.isInfoEnabled()) {
                            logger.info(" stop charge timeout handle : " + chargeRecord.toString());
                        }
                        if(StringUtils.isNotBlank(chargeRecord.getDeviceId()) && StringUtils.isNotBlank(chargeRecord.getPlugId())) {
                            ebikeChargeService.finishChargeRecord(chargeRecord.getDeviceId(), chargeRecord.getPlugId(), chargeRecord.getChargeOpenFlag(), IntegerUtils.parseIntIgnoreIllegal(FinishReason.REASON_STOPWAITTIMEOUT));
                        }
                    }
                }
                catch(InterruptedException _ie) {
                    logger.error(_ie.getMessage(), _ie.fillInStackTrace());
                    break;
                }
            }
        }

    }

}
