package cn.yunrui.mqttclient.ebikesrv.mqttclient.cache;

import cn.yunrui.mqttclient.ebikesrv.common.utils.*;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.dao.BillingDao;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.dao.ChargeDeviceDao;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.entity.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.annotation.Resource;
import java.util.*;

/**
 * 充电设备缓存（运行设备）
 */
public class ChargeDeviceCache {

    private final Logger logger = LoggerFactory.getLogger(getClass());

    private Map<String, ChargeDeviceProps> devicePropsMap = new HashMap<>();

    @Resource(name = "chargeDeviceDao")
    private ChargeDeviceDao chargeDeviceDao;

    @Resource(name = "billingDao")
    private BillingDao billingDao;

    /**
     * 初始化
     */
    @PostConstruct
    public void init() {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> start init ChargeDeviceCache >>>>>> ");
        }

        devicePropsMap.clear();

        // 加载全部
        List<ChargeDeviceProps> cdpList = chargeDeviceDao.getAllChargeDevicePropsList();
        if(CollectionUtils.isNotEmpty(cdpList)) {
            for(ChargeDeviceProps cdp : cdpList) {
                if(logger.isDebugEnabled()) {
                    logger.debug(" put [ " + cdp.getDeviceId() + " ] to cache : " + cdp.toString());
                }
                devicePropsMap.put(cdp.getDeviceId(), cdp);
            }
        }

        if(logger.isDebugEnabled()) {
            logger.debug(" <<<<<<< end init ChargeDeviceCache <<<<<<< ");
        }
    }

    /**
     * 销毁
     */
    @PreDestroy
    public void destroy() {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> start destroy ChargeDeviceCache >>>>>> ");
        }

        devicePropsMap.clear();

        if(logger.isDebugEnabled()) {
            logger.debug(" <<<<<<< end destroy ChargeDeviceCache <<<<<<< ");
        }
    }

    /**
     * 通过设备通讯标识从充电设备缓存中获取充电设备属性
     * @param deviceId
     *              设备通讯标识（MQTT规约中定义的deviceId）
     * @return 充电设备属性
     */
    public ChargeDeviceProps get(String deviceId) {
        ChargeDeviceProps cdp = devicePropsMap.get(deviceId);
        if(ObjectUtils.isNull(cdp) && StringUtils.isNotBlank(deviceId)) {
            cdp = reload(deviceId);
        }
        return cdp;
    }

    /**
     * 重新获取充电设备属性，并存入缓存
     * @param deviceId
     *              设备通讯标识（MQTT规约中定义的deviceId）
     * @return 充电设备属性
     */
    public ChargeDeviceProps reload(String deviceId) {
        if(StringUtils.isNotBlank(deviceId)) {
            put(deviceId, chargeDeviceDao.getChargeDeviceProps(deviceId));
        }
        return devicePropsMap.get(deviceId);
    }

    /**
     * 插入或更新充电设备缓存
     * @param deviceId
     *              设备通讯标识（MQTT规约中定义的deviceId）
     * @param deviceProps
     *              充电设备属性
     */
    public void put(String deviceId, ChargeDeviceProps deviceProps) {
        devicePropsMap.put(deviceId, deviceProps);
    }

    /**
     * 删除充电设备缓存
     * @param deviceId
     *              设备通讯标识（MQTT规约中定义的deviceId）
     */
    public void remove(String deviceId) {
        if(logger.isDebugEnabled()) {
            logger.debug(" remove [ " + deviceId + " ] from cache ");
        }
        devicePropsMap.remove(deviceId);
    }

    /**
     * 更新运行状态（包括充电设备及充电插座）
     * @param deviceId
     *              设备通讯标识（MQTT规约中定义的deviceId）
     * @param status
     *              设备运行状态
     * @param plugStatuss
     *              插座运行状态
     */
    public void updateOpStatus(String deviceId, String status, String[] plugStatuss) {
        chargeDeviceDao.updateOpStatus(deviceId, status, plugStatuss);
        // 更新缓存
        ChargeDeviceProps cdp = get(deviceId);
        if(ObjectUtils.isNotNull(cdp)) {
            cdp.setOpStatus(status);
            Map<Integer, ChargePlugProps> plugPropsMap = cdp.getPlugPropsMap();
            if(MapUtils.isNotEmpty(plugPropsMap) && ArrayUtils.isNotEmpty(plugStatuss) && plugStatuss.length > 0) {
                for(int i = 1; i <= plugStatuss.length; i++) {
                    ChargePlugProps cpp = plugPropsMap.get(i);
                    String plugStatus = plugStatuss[i - 1];
                    cpp.setOpStatus(plugStatus);
                    cpp.setStatus(convertPlugStatus(plugStatus));
                    plugPropsMap.put(i, cpp);
                }
                cdp.setPlugPropsMap(plugPropsMap);
            }
        }
    }

    /**
     * 更新插座运行状态
     * @param deviceId
     *              设备通讯标识（MQTT规约中定义的deviceId）
     * @param plugId
     *              插座通讯标识（MQTT规约中定义的plugId）
     * @param status
     *              插座运行状态
     */
    public void updatePlugStatus(String deviceId, String plugId, String status) {
        chargeDeviceDao.updatePlugStatus(deviceId, plugId, status);
        // 更新缓存
        ChargeDeviceProps cdp = get(deviceId);
        if(ObjectUtils.isNotNull(cdp) && StringUtils.isNotBlank(plugId) && StringUtils.isNotBlank(status)) {
            Map<Integer, ChargePlugProps> plugPropsMap = cdp.getPlugPropsMap();
            if(MapUtils.isNotEmpty(plugPropsMap)) {
                try {
                    Integer plugSn = Integer.parseInt(plugId);
                    ChargePlugProps cpp = plugPropsMap.get(plugSn);
                    cpp.setOpStatus(status);
                    cpp.setStatus(convertPlugStatus(status));
                    plugPropsMap.put(plugSn, cpp);
                    cdp.setPlugPropsMap(plugPropsMap);
                }
                catch(NumberFormatException _nfe) {
                    logger.error(_nfe.getMessage(), _nfe.fillInStackTrace());
                }
            }
        }
    }

    /**
     * 更新设备资产状态
     * @param deviceId
     *              设备通讯标识（MQTT规约中定义的deviceId）
     * @param status
     *              设备资产状态：00 - 库存；02 - 已装（待运行）；03 - 运行；07 - 拆除；
     */
    public void updateDeviceAssertStatus(String deviceId, String status) {
        chargeDeviceDao.updateDeviceAssertStatus(deviceId, status);
        // 更新缓存
        put(deviceId, chargeDeviceDao.getChargeDeviceProps(deviceId));
    }

    /**
     * 获取超时充电设备工况信息
     * @return 超时充电设备工况信息
     */
    public List<ChargeDeviceOps> getTimeoutChargeDeviceOpsList() {
        return chargeDeviceDao.getTimeoutChargeDeviceOpsList();
    }

    public void lockTimeoutChargeDeviceOpsList(List<ChargeDeviceOps> cdoList) {
        chargeDeviceDao.lockTimeoutChargeDeviceOpsList(cdoList);
    }

    public BillingInfo getDefaultBillingInfoByDeviceId(String deviceId) {
        return billingDao.getDefaultBillingInfoByDeviceId(deviceId);
    }

    public BillingInfo getBillingInfoByDeviceId(String deviceId, String chargeUserId, String cardId) {
        return chargeDeviceDao.getBillingInfoByDeviceId(deviceId, chargeUserId, cardId);
    }

    public Integer getCountByDeviceIdAndCardId(String deviceId, String cardId) {
        return chargeDeviceDao.getCountByDeviceIdAndCardId(deviceId, cardId);
    }

    public Double getServiceRateByDeviceId(String deviceId) {
        return chargeDeviceDao.getServiceRateByDeviceId(deviceId);
    }

    public CostInfo getCostInfoByDeviceId(String deviceId) {
        return chargeDeviceDao.getCostInfoByDeviceId(deviceId);
    }

    private String convertPlugStatus(String plugStatus) {
        String result;
        if(StringUtils.equals("1", plugStatus)) {
            result = "U";
        }
        else if(StringUtils.equals("2", plugStatus)) {
            result = "R";
        }
        else {
            result = "F";
        }
        return result;
    }

}
