package cn.yunrui.mqttclient.ebikesrv.mqttclient.cache;

import cn.yunrui.mqttclient.ebikesrv.common.utils.*;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.dao.ChargeCardDao;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.dao.ChargeLoadDao;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.dao.ChargeRecordDao;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.entity.ChargeRecord;
import com.ebikepay.openservices.request.InnerAccountCreateRequest;
import com.ebikepay.openservices.request.InnerAccountLoadRequest;
import com.ebikepay.openservices.request.TradeOrderCreateRequest;
import com.ebikepay.openservices.request.TradeOrderPayRequest;
import com.ebikepay.openservices.response.InnerAccountCreateResponse;
import com.ebikepay.openservices.response.InnerAccountLoadResponse;
import com.ebikepay.openservices.response.TradeOrderCreateResponse;
import com.ebikepay.openservices.response.TradeOrderPayResponse;
import com.ebikepay.openservices.service.InnerAccountService;
import com.ebikepay.openservices.service.TradeOrderService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.annotation.Resource;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 充电订单缓存（当前充电订单）
 *
 * @author Nick Zhang
 * @date 2019-04-15
 *
 */
public class ChargeRecordCache {

    private final Logger logger = LoggerFactory.getLogger(getClass());

    /**
     * <deviceId>_<plugId>  , ChargeRecord
     */
    private Map<String, ChargeRecord> chargeRecordMap = new HashMap<>();

    @Resource(name = "chargeCardDao")
    private ChargeCardDao chargeCardDao;

    @Resource(name = "chargeRecordDao")
    private ChargeRecordDao chargeRecordDao;

    @Resource(name = "chargeLoadDao")
    private ChargeLoadDao chargeLoadDao;

    @Resource(name = "innerAccountService")
    private InnerAccountService innerAccountService;

    @Resource(name = "tradeOrderService")
    private TradeOrderService tradeOrderService;

    /**
     * 初始化
     */
    @PostConstruct
    public void init() {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> start init ChargeRecordCache >>>>>> ");
        }

        chargeRecordMap.clear();

        // 加载当前充电订单列表
        List<ChargeRecord> chargeRecordList = chargeRecordDao.getBeChargingRecordList();
        if(CollectionUtils.isNotEmpty(chargeRecordList)) {
            for(ChargeRecord chargeRecord : chargeRecordList) {
                if(logger.isDebugEnabled()) {
                    logger.debug(" put [ " + chargeRecord.getDeviceId() + "_" + chargeRecord.getPlugId() + " ] to cache : " + chargeRecord.toString());
                }
                chargeRecordMap.put(chargeRecord.getDeviceId() + "_" + chargeRecord.getPlugId(), chargeRecord);
            }
        }

        if(logger.isDebugEnabled()) {
            logger.debug(" <<<<<<< end init ChargeRecordCache <<<<<<< ");
        }
    }

    /**
     * 销毁
     */
    @PreDestroy
    public void destroy() {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> start destroy ChargeRecordCache >>>>>> ");
        }

        chargeRecordMap.clear();

        if(logger.isDebugEnabled()) {
            logger.debug(" <<<<<<< end destroy ChargeRecordCache <<<<<<< ");
        }
    }

    /**
     * 通过设备通讯标识和插座标识从当前充电订单缓存中获取充电订单
     * @param deviceId
     *              设备通讯标识（MQTT规约中定义的deviceId）
     * @param plugId
     *              插座通讯标识（MQTT规约中定义的plugId）
     * @return 充电订单
     */
    public ChargeRecord get(String deviceId, String plugId) {
        ChargeRecord chargeRecord = chargeRecordMap.get(deviceId + "_" + plugId);
        if(ObjectUtils.isNull(chargeRecord) && StringUtils.isNotBlank(deviceId)) {
            chargeRecord = reload(deviceId, plugId);
        }
        return chargeRecord;
    }

    /**
     * 重新获取充电订单，并存入缓存
     * @param deviceId
     *              设备通讯标识（MQTT规约中定义的deviceId）
     * @param plugId
     *              插座通讯标识（MQTT规约中定义的plugId）
     * @return 充电订单
     */
    public ChargeRecord reload(String deviceId, String plugId) {
        if(StringUtils.isNotBlank(deviceId) && StringUtils.isNotBlank(plugId)) {
            put(deviceId, plugId, chargeRecordDao.getBeChargingRecordByDeviceIdAndPlugId(deviceId, plugId));
        }
        return chargeRecordMap.get(deviceId + "_" + plugId);
    }

    /**
     * 插入或更新充电设备缓存
     * @param deviceId
     *              设备通讯标识（MQTT规约中定义的deviceId）
     * @param plugId
     *              插座通讯标识（MQTT规约中定义的plugId）
     * @param chargeRecord
     *              充电订单
     */
    public void put(String deviceId, String plugId, ChargeRecord chargeRecord) {
        chargeRecordMap.put(deviceId + "_" + plugId, chargeRecord);
    }

    public int insertChargeRecordByCoinsin(Map<String, Object> chargeRecordMap, String chargeplugNo) {
        return chargeRecordDao.insertChargeRecordByCoinsin(chargeRecordMap, chargeplugNo);
    }

    public int insertChargeRecordByCard(Map<String, Object> chargeRecordMap, String chargeplugNo, String cardId) {
        return chargeRecordDao.insertChargeRecordByCard(chargeRecordMap, chargeplugNo, cardId);
    }

    public int insertChargeRecordByCustCard(Map<String, Object> chargeRecordMap, String chargeplugNo, String cardId) {
        return chargeRecordDao.insertChargeRecordByCustCard(chargeRecordMap, chargeplugNo, cardId);
    }

    /**
     * 更新充电记录
     * @param deviceId
     *              设备通讯标识（MQTT规约中定义的deviceId）
     * @param plugId
     *              插座通讯标识（MQTT规约中定义的plugId）
     * @param valueMap
     *              更新充电记录值集合
     */
    public void updateChargeRecord(String deviceId, String plugId, Map<String, Object> valueMap) {
        chargeRecordDao.updateChargeRecord(deviceId, plugId, valueMap);
    }

    /**
     * 结束充电
     * @param deviceId
     *              设备通讯标识（MQTT规约中定义的deviceId）
     * @param plugId
     *              插座通讯标识（MQTT规约中定义的plugId）
     * @param valueMap
     *              结束充电记录值集合
     */
    public void finishChargeRecord(String deviceId, String plugId, Map<String, Object> valueMap) {
        chargeRecordDao.finishChargeRecord(deviceId, plugId, valueMap);
    }

    /**
     * 检查是否充电中
     * @param deviceId
     *              设备通讯标识（MQTT规约中定义的deviceId）
     * @param plugId
     *              插座通讯标识（MQTT规约中定义的plugId）
     * @return true : 正在充电中；false : 不在充电中
     */
    public boolean checkIsCharging(String deviceId, String plugId) {
        return chargeRecordDao.checkIsCharging(deviceId, plugId);
    }

    /**
     * 获取充电记录
     * @param deviceId
     *              设备通讯标识（MQTT规约中定义的deviceId）
     * @param plugId
     *              插座通讯标识（MQTT规约中定义的plugId）
     * @return
     *              获取充电记录
     */
    public Map<String, Object> getChargeRecordMap(String deviceId, String plugId) {
        return chargeRecordDao.getChargeRecordMap(deviceId, plugId);
    }

    /**
     * 获取等待超时充电记录列表
     * @return 充电记录列表
     */
    public List<ChargeRecord> getWaitingStartTimeoutChargeRecordList() {
        return chargeRecordDao.getWaitingStartTimeoutChargeRecordList();
    }

    /**
     * 获取结束超时充电记录列表
     * @return 充电记录列表
     */
    public List<ChargeRecord> getWaitingStopTimeoutChargeRecordList() {
        return chargeRecordDao.getWaitingStopTimeoutChargeRecordList();
    }

    public void lockWaitingTimeoutChargeRecordList(List<ChargeRecord> chargeRecordList) {
        chargeRecordDao.lockWaitingTimeoutChargeRecordList(chargeRecordList);
    }

    public List<ChargeRecord> getBeReopeningChargeRecordList() {
        return chargeRecordDao.getBeReopeningChargeRecordList();
    }

    public void lockBeReopeningChargeRecordList(List<ChargeRecord> chargeRecordList) {
        chargeRecordDao.lockBeReopeningChargeRecordList(chargeRecordList);
    }

    public ChargeRecord getBeChargingRecord(String chargeRecordId) {
        return chargeRecordDao.getBeChargingRecord(chargeRecordId);
    }

    public ChargeRecord getHistroyChargeRecord(String chargeRecordId) {
        return chargeRecordDao.getHistroyChargeRecord(chargeRecordId);
    }

    public void saveChargedeviceLoadMap(Map<String, Object> chargedeviceLoadMap) {
        chargeLoadDao.saveChargedeviceLoadMap(chargedeviceLoadMap);
    }

    public void saveChargeplugLoadMap(Map<String, Object> chargeplugLoadMap) {
        chargeLoadDao.saveChargeplugLoadMap(chargeplugLoadMap);
    }

    public void saveChargeorderLoadMap(Map<String, Object> chargeorderLoadMap) {
        chargeLoadDao.saveChargeorderLoadMap(chargeorderLoadMap);
    }

    public String getStatusByCardId(String cardId) {
        return chargeCardDao.getStatusByCardId(cardId);
    }

    public Integer getCardTypeByCardId(String cardId) {
        return chargeCardDao.getCardTypeByCardId(cardId);
    }

    public String getOrgNoByCustCardId(String cardId) {
        return chargeCardDao.getOrgNoByCustCardId(cardId);
    }

    public String getChargeUserIdByCardId(String cardId) {
        return chargeCardDao.getChargeUserIdByCardId(cardId);
    }

    public InnerAccountLoadResponse loadInnerAccount(InnerAccountLoadRequest innerAccountLoadRequest) {
        return innerAccountService.loadInnerAccount(innerAccountLoadRequest);
    }

    public InnerAccountCreateResponse createInnerAccount(InnerAccountCreateRequest innerAccountCreateRequest) {
        return innerAccountService.createInnerAccount(innerAccountCreateRequest);
    }

    public TradeOrderCreateResponse createTradeOrder(TradeOrderCreateRequest tradeOrderCreateRequest) {
        return tradeOrderService.createTradeOrder(tradeOrderCreateRequest);
    }

    public TradeOrderPayResponse payTradeOrder(TradeOrderPayRequest tradeOrderPayRequest) {
        return tradeOrderService.payTradeOrder(tradeOrderPayRequest);
    }

}
