package cn.yunrui.mqttclient.ebikesrv.mqttclient.dao.impl;

import cn.yunrui.mqttclient.ebikesrv.common.utils.*;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.dao.ChargeDeviceDao;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.entity.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.core.BatchPreparedStatementSetter;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;

import java.sql.*;
import java.util.*;
import java.util.Date;

/**
 *
 */
public class ChargeDeviceDaoImpl implements ChargeDeviceDao {

    private final Logger logger = LoggerFactory.getLogger(getClass());

    private final RowMapper<String> strRowMapper = (rs, rowNum) -> rs.getString(1);

    private final RowMapper<ChargeDeviceOps> cdoRowMapper = (rs, rowNum) -> {
        ChargeDeviceOps cpo = new ChargeDeviceOps();
        cpo.setOrgNo(rs.getString("subburo"));
        cpo.setChargedeviceNo(rs.getString("chargedeviceNo"));
        cpo.setDeviceId(rs.getString("deviceId"));
        cpo.setPlugCount(rs.getInt("plugCount"));
        cpo.setChargedeviceId(rs.getLong("chargedeviceId"));
        cpo.setOpStatus(rs.getString("opStatus"));
        cpo.setLatestCommTime(new Date(rs.getTimestamp("latestCommTime").getTime()));
        return cpo;
    };

    @SuppressWarnings("unchecked")
    private final RowMapper<ChargeDeviceProps> cdpRowMapper = (rs, rowNum) -> {
        ChargeDeviceProps cdp = new ChargeDeviceProps();
        cdp.setDeviceId(rs.getString("deviceId"));
        cdp.setOrgNo(rs.getString("subburo"));
        cdp.setChargedeviceNo(rs.getString("chargedeviceNo"));
        if(StringUtils.equals("mqtt.yjm2m", rs.getString("protocolType"))) {
            if(StringUtils.equals("201710", rs.getString("protocolVersion"))) {
                cdp.setChargeDeviceProtocol(ChargeDeviceProtocolEnum.MQTT_YJM2M_201710);
            }
            else if(StringUtils.equals("201712", rs.getString("protocolVersion"))) {
                cdp.setChargeDeviceProtocol(ChargeDeviceProtocolEnum.MQTT_YJM2M_201712);
            }
            else if(StringUtils.equals("201801", rs.getString("protocolVersion"))) {
                cdp.setChargeDeviceProtocol(ChargeDeviceProtocolEnum.MQTT_YJM2M_201801);
            }
            else if(StringUtils.equals("201804", rs.getString("protocolVersion"))) {
                cdp.setChargeDeviceProtocol(ChargeDeviceProtocolEnum.MQTT_YJM2M_201804);
            }
            else {
                cdp.setChargeDeviceProtocol(ChargeDeviceProtocolEnum.UNKNOWN);
            }
        }
        else if(StringUtils.equals("mqtt.ebike", rs.getString("protocolType"))) {
            if(StringUtils.equals("201804", rs.getString("protocolVersion"))) {
                cdp.setChargeDeviceProtocol(ChargeDeviceProtocolEnum.MQTT_EBIKE_201804);
            }
            else if(StringUtils.equals("201805", rs.getString("protocolVersion"))) {
                cdp.setChargeDeviceProtocol(ChargeDeviceProtocolEnum.MQTT_EBIKE_201805);
            }
            else {
                cdp.setChargeDeviceProtocol(ChargeDeviceProtocolEnum.UNKNOWN);
            }
        }
        else {
            cdp.setChargeDeviceProtocol(ChargeDeviceProtocolEnum.UNKNOWN);
        }
        cdp.setFactory(rs.getString("factory"));
        cdp.setModelNo(rs.getString("modelNo"));
        cdp.setPlugCount(rs.getInt("plugCount"));
        cdp.setOpStatus(rs.getString("opStatus"));
        if(rs.getTimestamp("latestCommTime") != null) {
            cdp.setLatestCommTime(new Date(rs.getTimestamp("latestCommTime").getTime()));
        }
        cdp.setStatus(rs.getString("status"));
        cdp.setChargedeviceId(rs.getLong("chargedeviceId"));
        if(rs.getTimestamp("importTime") != null) {
            cdp.setImportTime(new Date(rs.getTimestamp("importTime").getTime()));
        }
        if(rs.getTimestamp("installTime") != null) {
            cdp.setInstallTime(new Date(rs.getTimestamp("installTime").getTime()));
        }
        if(rs.getTimestamp("runTime") != null) {
            cdp.setRunTime(new Date(rs.getTimestamp("runTime").getTime()));
        }
        if(rs.getTimestamp("dismantleTime") != null) {
            cdp.setDismantleTime(new Date(rs.getTimestamp("dismantleTime").getTime()));
        }
        //
        cdp.setMqttBrokerHostIp(rs.getString("mqttBrokerHostIp"));
        cdp.setMqttBrokerHostPort(rs.getString("mqttBrokerHostPort"));
        cdp.setServerSubTopic(rs.getString("serverSubTopic"));
        // paramsMap & attachParamsMap
        String deviceParams = rs.getString("deviceParams");
        Map deviceParamsMap;
        Map<String, String> paramsMap;
        Map<String, String> attachParamsMap;
        if(StringUtils.isNotBlank(deviceParams)) {
            deviceParamsMap = JsonConvertUtils.convertFromString(Map.class, deviceParams);
            if(deviceParamsMap != null && deviceParamsMap.get("params") != null) {
                paramsMap = (Map<String, String>) deviceParamsMap.get("params");
                if(paramsMap != null) {
                    cdp.setParamsMap(paramsMap);
                }
            }
            if(deviceParamsMap != null && deviceParamsMap.get("attachParams") != null) {
                attachParamsMap = (Map<String, String>) deviceParamsMap.get("attachParams");
                if(attachParamsMap != null) {
                    cdp.setAttachParamsMap(attachParamsMap);
                }
            }
        }
        cdp.setChargestationId(rs.getLong("chargestationId"));
        cdp.setBillingSchemeId(rs.getString("schemeId"));
        cdp.setBillingSchemeType(rs.getString("schemeType"));
        cdp.setBillingUnitPrice1(rs.getDouble("unitPrice1"));
        cdp.setBillingUnitPrice2(rs.getDouble("unitPrice2"));
        cdp.setBillingUnitPrice3(rs.getDouble("unitPrice3"));
        cdp.setMinChargeMoney(rs.getDouble("minCharge"));
        cdp.setIsReturn(rs.getString("isReturn"));
        // plugPropsMap
        Map<Integer, ChargePlugProps> plugPropsMap = new HashMap<>();
        List<ChargePlugProps> cppList = getChargePlugPropsList(cdp.getDeviceId());
        for(ChargePlugProps cpp : cppList) {
            plugPropsMap.put(cpp.getPlugSn(), cpp);
        }
        if(MapUtils.isNotEmpty(plugPropsMap)) {
            cdp.setPlugPropsMap(plugPropsMap);
        }
        return cdp;
    };

    private final RowMapper<ChargePlugProps> cppRowMapper = (rs, rowNum) -> {
        ChargePlugProps cpp = new ChargePlugProps();
        cpp.setDeviceId(rs.getString("deviceId"));
        cpp.setPlugId(rs.getString("plugId"));
        cpp.setChargeplugId(rs.getLong("chargeplugId"));
        cpp.setPlugSn(rs.getInt("plugSn"));
        cpp.setOpStatus(rs.getString("opStatus"));
        if(rs.getTimestamp("opStatusTime") != null) {
            cpp.setOpStatusTime(new Date(rs.getTimestamp("opStatusTime").getTime()));
        }
        cpp.setStatus(rs.getString("status"));
        return cpp;
    };

    private final RowMapper<BillingInfo> biRowMapper = (rs, rowNum) -> {
        BillingInfo bi = new BillingInfo();
        bi.setOrgNo(rs.getString(1));
        bi.setChargestationId(rs.getLong(2));
        bi.setCsbsType(rs.getString(3));
        bi.setSchemeId(rs.getString(4));
        bi.setSchemeType(rs.getString(5));
        bi.setUnitPrice1(rs.getDouble(6));
        bi.setUnitPrice2(rs.getDouble(7));
        bi.setUnitPrice3(rs.getDouble(8));
        bi.setUnitPrice4(rs.getDouble(9));
        bi.setUnitPrice5(rs.getDouble(10));
        bi.setStepCount(rs.getInt(11));
        bi.setStepPower1(rs.getInt(12));
        bi.setStepPower2(rs.getInt(13));
        bi.setStepPower3(rs.getInt(14));
        bi.setStepPower4(rs.getInt(15));
        bi.setMinCharge(rs.getDouble(16));
        bi.setIsReturn(rs.getString(17));
        bi.setClosingPeriod(rs.getInt(18));
        bi.setMaxChargeTime(rs.getInt(19));
        return bi;
    };

    private final RowMapper<CostInfo> ciRowMapper = (rs, rowNum) -> {
        CostInfo ci = new CostInfo();
        ci.setOrgNo(rs.getString(1));
        ci.setCostSchemeId(rs.getString(2));
        ci.setCostSchemeNo(rs.getString(3));
        ci.setCostSchemeName(rs.getString(4));
        ci.setCostUnitPrice(rs.getDouble(5));
        return ci;
    };

    private StringBuilder builderChargeDeviceOpsPreSQL() {
        StringBuilder sqlBuilder = new StringBuilder();
        sqlBuilder.append("SELECT eac.buro, eac.subburo, eac.chargedeviceNo, eac.deviceId, eac.plugCount, eco.chargedeviceId, eco.opStatus, eco.latestCommTime ");
        sqlBuilder.append("  FROM ebike.ebike_chargedevice_ops eco ");
        sqlBuilder.append(" INNER JOIN ebike.ebike_assets_chargedevice eac ON eac.chargedeviceId = eco.chargedeviceId ");
        return sqlBuilder;
    }

    private StringBuilder builderChargeDevicePropsPreSQL() {
        StringBuilder sqlBuilder = new StringBuilder();
        sqlBuilder.append("SELECT eac.buro, eac.subburo, eac.chargedeviceNo, eac.deviceId, eac.protocolType, eac.protocolVersion, eac.plugCount, eac.factory, eac.modelNo ");
        sqlBuilder.append("     , eac.`status`, eac.chargedeviceId, eac.importTime, eac.installTime, eac.runTime, eac.dismantleTime, eac.mqttBrokerHostIp, eac.mqttBrokerHostPort, eac.serverSubTopic ");
        sqlBuilder.append("     , ops.opStatus, ops.latestCommTime, ecd.deviceParams ");
        sqlBuilder.append("     , ecd.chargestationId, ebs.schemeId, ebs.schemeType, ebs.unitPrice1, ebs.unitPrice2, ebs.unitPrice3, ebs.minCharge, ebs.isReturn ");
        sqlBuilder.append("  FROM ebike.ebike_assets_chargedevice eac ");
        sqlBuilder.append("  LEFT JOIN ebike.ebike_chargedevice_ops ops ON ops.chargedeviceId = eac.chargedeviceId ");
        sqlBuilder.append("  LEFT JOIN cisp_dev.dev_ebikechargedevice ecd ON ecd.id = eac.chargedeviceId ");
        sqlBuilder.append("  LEFT JOIN ebike.ebike_cs_bs_rela cbr ON cbr.chargestationId = ecd.chargestationId ");
        sqlBuilder.append("  LEFT JOIN ebike.ebike_billingscheme ebs ON ebs.schemeId = cbr.schemeId ");
        return sqlBuilder;
    }

    private StringBuilder builderChargePlugPropsPreSQL() {
        StringBuilder sqlBuilder = new StringBuilder();
        sqlBuilder.append("SELECT ecd.deviceId, ecp.plugId, ecd.chargedeviceNo, ecp.plugSn, CASE ops.opStatus WHEN '1' THEN 'U' WHEN '2' THEN 'R' ELSE 'F' END AS `status`, ecd.`id` AS chargedeviceId, ecp.`id` AS chargeplugId, ops.opStatus, ops.opStatusTime ");
        sqlBuilder.append("  FROM cisp_dev.dev_ebikechargeplug ecp ");
        sqlBuilder.append(" INNER JOIN cisp_dev.dev_ebikechargedevice ecd ON ecd.`id` = ecp.chargedeviceId ");
        sqlBuilder.append("  LEFT JOIN ebike.ebike_chargeplug_ops ops ON ops.chargeplugId = ecp.`id` ");
        return sqlBuilder;
    }

    private final JdbcTemplate ebikeJdbcTemplate;

    public ChargeDeviceDaoImpl(JdbcTemplate ebikeJdbcTemplate) {
        this.ebikeJdbcTemplate = ebikeJdbcTemplate;
    }

    @Override
    public List<ChargeDeviceOps> getTimeoutChargeDeviceOpsList() {
        StringBuilder sqlBuilder = builderChargeDeviceOpsPreSQL();
        List<Object> params = new ArrayList<>();
        sqlBuilder.append(" WHERE eco.opStatus = ? ");
        params.add("1");
        sqlBuilder.append("   AND eco.latestCommTime < ? ");
        params.add(DateUtils.addMinutes(new Date(), -30));
        return ebikeJdbcTemplate.query(sqlBuilder.toString(), params.toArray(), cdoRowMapper);
    }

    @Override
    public void lockTimeoutChargeDeviceOpsList(final List<ChargeDeviceOps> cdoList) {
        if(CollectionUtils.sizeIsNotEmptyIgnoreNull(cdoList)) {
            String sql = "UPDATE ebike.ebike_chargedevice_ops cdo SET cdo.opStatus = '2' WHERE cdo.opStatus = '1' AND cdo.chargedeviceId = ? ";
            ebikeJdbcTemplate.batchUpdate(sql, new BatchPreparedStatementSetter() {

                @Override
                public int getBatchSize() {
                    return cdoList.size();
                }

                @Override
                public void setValues(PreparedStatement ps, int romNum) throws SQLException {
                    ChargeDeviceOps cdo = cdoList.get(romNum);
                    ps.setLong(1, cdo.getChargedeviceId());
                }

            });
        }
    }

    @Override
    public List<ChargeDeviceProps> getAllChargeDevicePropsList() {
        return getChargeDevicePropsList(null, null);
    }

    @Override
    public List<String> getNewUpdatedDeviceIds(Date lastLoadTime) {
        StringBuilder sqlBuilder = new StringBuilder();
        List<Object> params = new ArrayList<>();
        sqlBuilder.append("SELECT eac.deviceId ");
        sqlBuilder.append("  FROM ebike.ebike_assets_chargedevice eac ");
        sqlBuilder.append("  LEFT JOIN cisp_dev.dev_ebikechargedevice ecd ON ecd.id = eac.chargedeviceId ");
        sqlBuilder.append("  LEFT JOIN ebike.ebike_cs_bs_rela cbr ON cbr.chargestationId = ecd.chargestationId ");
        sqlBuilder.append("  LEFT JOIN ebike.ebike_billingscheme ebs ON ebs.schemeId = cbr.schemeId ");
        sqlBuilder.append(" WHERE eac.importTime >= ? ");
        params.add(lastLoadTime);
        sqlBuilder.append("    OR eac.installTime >= ? ");
        params.add(lastLoadTime);
        sqlBuilder.append("    OR eac.runTime >= ? ");
        params.add(lastLoadTime);
        sqlBuilder.append("    OR eac.dismantleTime >= ? ");
        params.add(lastLoadTime);
        sqlBuilder.append("    OR cbr.effTime >= ? ");
        params.add(lastLoadTime);
        sqlBuilder.append("    OR ebs.updateTime >= ? ");
        params.add(lastLoadTime);
        return ebikeJdbcTemplate.query(sqlBuilder.toString(), params.toArray(), strRowMapper);
    }

    @Override
    public List<ChargeDeviceProps> getChargeDevicePropsList(Map<String, Object> paramsMap, List<String> ordersList) {
        StringBuilder sqlBuilder = builderChargeDevicePropsPreSQL();
        List<Object> params = new ArrayList<>();
        sqlBuilder.append(" WHERE 1 = 1 ");
        if(MapUtils.isNotEmpty(paramsMap)) {
            for(String key : paramsMap.keySet()) {
                sqlBuilder.append("   AND ").append(key).append(" = ? ");
                params.add(paramsMap.get(key));
            }
        }
        if(ordersList != null && ordersList.size() > 0) {
            sqlBuilder.append(" ORDER BY ");
            int idx = 0;
            for(String orders : ordersList) {
                if(idx > 0) {
                    sqlBuilder.append(", ");
                }
                sqlBuilder.append(orders);
                idx++;
            }
        }
        return ebikeJdbcTemplate.query(sqlBuilder.toString(), params.toArray(), cdpRowMapper);
    }

    @Override
    public ChargeDeviceProps getChargeDeviceProps(String deviceId) {
        if(StringUtils.isNotBlank(deviceId)) {
            StringBuilder sqlBuilder = builderChargeDevicePropsPreSQL();
            List<Object> params = new ArrayList<>();
            sqlBuilder.append(" WHERE eac.deviceId = ? ");
            params.add(deviceId);
            ChargeDeviceProps cdp;
            try {
                if(logger.isDebugEnabled()) {
                    logger.debug("sql       : " + sqlBuilder.toString());
                    logger.debug("params    : " + Arrays.toString(params.toArray()));
                }
                cdp = ebikeJdbcTemplate.queryForObject(sqlBuilder.toString(), params.toArray(), cdpRowMapper);
            }
            catch(EmptyResultDataAccessException _erdae) {
                cdp = null;
            }
            return cdp;
        }
        return null;
    }

    @Override
    public ChargeDeviceProps getChargeDevicePropsByChargedeviceNo(String orgNo, String chargedeviceNo) {
        if(StringUtils.isNotBlank(orgNo) && StringUtils.isNotBlank(chargedeviceNo)) {
            StringBuilder sqlBuilder = builderChargeDevicePropsPreSQL();
            List<Object> params = new ArrayList<>();
            sqlBuilder.append(" WHERE eac.buro = ? AND eac.subburo = ? AND eac.chargedeviceNo = ? ");
            params.add(orgNo);
            params.add(orgNo);
            params.add(chargedeviceNo);
            ChargeDeviceProps cdp;
            try {
                cdp = ebikeJdbcTemplate.queryForObject(sqlBuilder.toString(), params.toArray(), cdpRowMapper);
            }
            catch(EmptyResultDataAccessException _erdae) {
                cdp = null;
            }
            return cdp;
        }
        return null;
    }

    @Override
    public List<ChargePlugProps> getChargePlugPropsList(String deviceId) {
        if(StringUtils.isNotBlank(deviceId)) {
            StringBuilder sqlBuilder = builderChargePlugPropsPreSQL();
            List<Object> params = new ArrayList<>();
            sqlBuilder.append(" WHERE ecd.deviceId = ? ");
            params.add(deviceId);
            return ebikeJdbcTemplate.query(sqlBuilder.toString(), params.toArray(), cppRowMapper);
        }
        return null;
    }

    @Override
    public ChargePlugProps getChargePlugProps(String deviceId, String plugId) {
        if(StringUtils.isNotBlank(deviceId) && StringUtils.isNotBlank(plugId)) {
            StringBuilder sqlBuilder = builderChargePlugPropsPreSQL();
            List<Object> params = new ArrayList<>();
            sqlBuilder.append(" WHERE ecd.deviceId = ? AND ecp.plugSn = ? ");
            params.add(deviceId);
            try {
                params.add(Integer.parseInt(plugId));
            }
            catch(NumberFormatException _nfe) {
                logger.error(_nfe.getMessage(), _nfe.fillInStackTrace());
                params.add(0);
            }
            ChargePlugProps cpp;
            try {
                cpp = ebikeJdbcTemplate.queryForObject(sqlBuilder.toString(), params.toArray(), cppRowMapper);
            }
            catch(EmptyResultDataAccessException _erdae) {
                cpp = null;
            }
            return cpp;
        }
        return null;
    }

    @Override
    public ChargePlugProps getChargePlugPropsByChargedeviceNo(String orgNo, String chargedeviceNo, Integer plugSn) {
        return null;
    }

    @Override
    public void updateOpStatus(final String deviceId, final String status, final String[] plugStatus) {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> updateOpStatus >>>>>> ");
            logger.debug(" deviceId   : " + deviceId);
            logger.debug(" status     : " + status);
            logger.debug(" plugStatus : " + ArrayUtils.toString(plugStatus));
        }
        if(StringUtils.equals("1", status)) {
            String sql = "INSERT INTO ebike.ebike_chargedevice_ops(chargedeviceId, opStatus, firstCommTime, latestCommTime) " +
                    "SELECT ecd.id, ?, IFNULL(eco.firstCommTime, ?), ? FROM cisp_dev.dev_ebikechargedevice ecd " +
                    "  LEFT JOIN ebike.ebike_chargedevice_ops eco ON eco.chargedeviceId = ecd.id " +
                    " WHERE ecd.deviceId = ? " +
                    "ON DUPLICATE KEY UPDATE opStatus = VALUES(opStatus), firstCommTime = VALUES(firstCommTime), latestCommTime = VALUES(latestCommTime) ";
            ebikeJdbcTemplate.update(sql, status, new Date(), new Date(), deviceId);
        }
        else {
            String sql = "INSERT INTO ebike.ebike_chargedevice_ops(chargedeviceId, opStatus) " +
                    "SELECT ecd.id, ? FROM cisp_dev.dev_ebikechargedevice ecd " +
                    "  LEFT JOIN ebike.ebike_chargedevice_ops eco ON eco.chargedeviceId = ecd.id " +
                    " WHERE ecd.deviceId = ? " +
                    "ON DUPLICATE KEY UPDATE opStatus = VALUES(opStatus) ";
            ebikeJdbcTemplate.update(sql, status, deviceId);
        }

        if(plugStatus != null && plugStatus.length > 0) {
            final Date d = new Date();
            String sql = "INSERT INTO ebike.ebike_chargeplug_ops(chargeplugId, opStatus, opStatusTime) " +
                    "SELECT ecp.id, ?, ? FROM cisp_dev.dev_ebikechargeplug ecp " +
                    " INNER JOIN cisp_dev.dev_ebikechargedevice ecd ON ecd.id = ecp.chargedeviceId " +
                    "  LEFT JOIN ebike.ebike_chargeplug_ops eco ON eco.chargeplugId = ecp.id " +
                    " WHERE ecd.deviceId = ? AND ecp.plugSn = ? " +
                    "ON DUPLICATE KEY UPDATE opStatus = VALUES(opStatus), opStatusTime = VALUES(opStatusTime) ";
            ebikeJdbcTemplate.batchUpdate(sql, new BatchPreparedStatementSetter() {

                @Override
                public int getBatchSize() {
                    return plugStatus.length;
                }

                @Override
                public void setValues(PreparedStatement ps, int i) throws SQLException {
                    ps.setString(1, plugStatus[i]);
                    ps.setTimestamp(2, new Timestamp(d.getTime()));
                    ps.setString(3, deviceId);
                    ps.setInt(4, (i + 1));
                }

            });
        }
    }

    @Override
    public void updatePlugStatus(String deviceId, String plugId, String status) {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> updatePlugStatus >>>>>> ");
            logger.debug(" deviceId : " + deviceId);
            logger.debug(" plugId   : " + plugId);
            logger.debug(" status   : " + status);
        }
        String sql = "INSERT INTO ebike.ebike_chargeplug_ops(chargeplugId, opStatus, opStatusTime) " +
                "SELECT ecp.id, ?, ? FROM cisp_dev.dev_ebikechargeplug ecp " +
                " INNER JOIN cisp_dev.dev_ebikechargedevice ecd ON ecd.id = ecp.chargedeviceId " +
                "  LEFT JOIN ebike.ebike_chargeplug_ops eco ON eco.chargeplugId = ecp.id " +
                " WHERE ecd.deviceId = ? AND ecp.plugSn = ? " +
                "ON DUPLICATE KEY UPDATE opStatus = VALUES(opStatus), opStatusTime = VALUES(opStatusTime) ";
        List<Object> params = new ArrayList<>();
        params.add(status);
        params.add(new Date());
        params.add(deviceId);
        try {
            params.add(Integer.parseInt(plugId));
        }
        catch(NumberFormatException _nfe) {
            logger.error(_nfe.getMessage(), _nfe.fillInStackTrace());
            params.add(0);
        }
        ebikeJdbcTemplate.update(sql, params.toArray());
    }

    @Override
    public void updateDeviceAssertStatus(String deviceId, String status) {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> updateDeviceAssertStatus >>>>>> ");
            logger.debug(" deviceId : " + deviceId);
            logger.debug(" status   : " + status);
        }
        if(StringUtils.isNotBlank(deviceId) && StringUtils.isNotBlank(status)) {
            StringBuilder sqlBuilder = new StringBuilder();
            List<Object> params = new ArrayList<>();
            sqlBuilder.append("UPDATE ebike.ebike_assets_chargedevice eac ");
            sqlBuilder.append("   SET eac.`status` = ? ");
            params.add(status);
            if(StringUtils.equals("03", status)) {
                // 03 - 运行
                sqlBuilder.append("     , eac.runTime = ? ");
                params.add(new Date());
                sqlBuilder.append("     , eac.dismantleTime = NULL ");
            }
            sqlBuilder.append(" WHERE eac.deviceId = ? ");
            params.add(deviceId);
            if(StringUtils.equals("03", status)) {
                sqlBuilder.append("   AND eac.`status` = '02' ");
            }
            ebikeJdbcTemplate.update(sqlBuilder.toString(), params.toArray());
        }
    }

    @Override
    public BillingInfo getBillingInfoByDeviceId(String deviceId, String chargeUserId, String cardId) {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>>> start getBillingInfoByDeviceId >>>>>>> ");
            logger.debug(" deviceId     : " + deviceId);
            logger.debug(" chargeUserId : " + chargeUserId);
            logger.debug(" cardId       : " + cardId);
        }
        BillingInfo bi;
        StringBuilder sqlBuilder1 = new StringBuilder();
        List<Object> params1 = new ArrayList<>();
        sqlBuilder1.append("SELECT acd.subburo, ecd.chargestationId, '00' AS csbsType, bs.schemeId, bs.schemeType, bs.unitPrice1, bs.unitPrice2, bs.unitPrice3, bs.unitPrice4, bs.unitPrice5, bs.stepCount, bs.stepPower1, bs.stepPower2, bs.stepPower3, bs.stepPower4, bs.minCharge, bs.isReturn, bs.closingPeriod, bs.maxChargeTime ");
        sqlBuilder1.append("  FROM ebike.ebike_billingscheme bs ");
        sqlBuilder1.append(" INNER JOIN ebike.ebike_csbs_info cbi ON cbi.schemeId = bs.schemeId ");
        sqlBuilder1.append(" INNER JOIN ebike.ebike_csbs_info_userlist cbiul ON cbiul.csbsUsersListId = cbi.csbsUsersListId ");
        sqlBuilder1.append(" INNER JOIN ebike.ebike_chargeuser cu ON cu.chargeUserMobile = cbiul.csbsUserMobile ");
        sqlBuilder1.append(" INNER JOIN cisp_dev.dev_ebikechargedevice ecd ON ecd.chargestationId = cbi.chargestationId AND cbi.csbsType = '01' AND cbi.csbsStatus IN ('00', '02', '03') ");
        sqlBuilder1.append(" INNER JOIN ebike.ebike_assets_chargedevice acd ON acd.chargedeviceId = ecd.id ");
        sqlBuilder1.append(" WHERE acd.deviceId = ? AND cu.chargeUserId = ? ");
        sqlBuilder1.append(" LIMIT 1 ");
        params1.add(deviceId);
        params1.add(chargeUserId);
        try {
            bi = ebikeJdbcTemplate.queryForObject(sqlBuilder1.toString(), params1.toArray(), biRowMapper);
        }
        catch(EmptyResultDataAccessException erdae) {
            bi = null;
        }
        if(bi == null) {
            StringBuilder sqlBuilder2 = new StringBuilder();
            List<Object> params2 = new ArrayList<>();
            sqlBuilder2.append("SELECT acd.subburo, ecd.chargestationId, '00' AS csbsType, bs.schemeId, bs.schemeType, bs.unitPrice1, bs.unitPrice2, bs.unitPrice3, bs.unitPrice4, bs.unitPrice5, bs.stepCount, bs.stepPower1, bs.stepPower2, bs.stepPower3, bs.stepPower4, bs.minCharge, bs.isReturn, bs.closingPeriod, bs.maxChargeTime ");
            sqlBuilder2.append("  FROM ebike.ebike_billingscheme bs ");
            sqlBuilder2.append(" INNER JOIN ebike.ebike_cs_bs_rela cbr ON cbr.schemeId = bs.schemeId ");
            sqlBuilder2.append(" INNER JOIN cisp_dev.dev_ebikechargedevice ecd ON ecd.chargestationId = cbr.chargestationId ");
            sqlBuilder2.append(" INNER JOIN ebike.ebike_assets_chargedevice acd ON acd.chargedeviceId = ecd.id ");
            sqlBuilder2.append(" WHERE acd.deviceId = ? AND cbr.disabled = 'F' ");
            sqlBuilder2.append(" LIMIT 1 ");
            params2.add(deviceId);
            try {
                bi = ebikeJdbcTemplate.queryForObject(sqlBuilder2.toString(), params2.toArray(), biRowMapper);
            }
            catch(EmptyResultDataAccessException erdae) {
                bi = null;
            }
        }
        return bi;
    }

    @Override
    public int getCountByDeviceIdAndCardId(String deviceId, String cardId) {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>>> start getCountByDeviceIdAndCardId >>>>>>> ");
            logger.debug(" deviceId     : " + deviceId);
            logger.debug(" cardId       : " + cardId);
        }
        int count = 0;
        StringBuilder sqlBuilder1 = new StringBuilder();
        List<Object> params1 = new ArrayList<>();
        sqlBuilder1.append("SELECT COUNT(*) AS CNT ");
        sqlBuilder1.append("  FROM ebike.card_info ci ");
        sqlBuilder1.append(" INNER JOIN ebike.card_custom_purchase_apply_relacs ccpar ON ccpar.purchase_apply_id = ci.purchase_apply_id ");
        sqlBuilder1.append(" INNER JOIN cisp_dev.dev_ebikechargedevice ecd ON ecd.chargestationId = ccpar.chargestation_id ");
        sqlBuilder1.append(" INNER JOIN ebike.ebike_assets_chargedevice acd ON acd.chargedeviceId = ecd.id ");
        sqlBuilder1.append(" WHERE ci.card_id = ? AND acd.deviceId = ? ");
        params1.add(deviceId);
        params1.add(cardId);
        try {
            count = ebikeJdbcTemplate.queryForObject(sqlBuilder1.toString(), params1.toArray(), Integer.class);
        }
        catch(EmptyResultDataAccessException erdae) {
            count = 0;
        }
        return count;
    }

    @Override
    public Double getServiceRateByDeviceId(String deviceId) {
        if(logger.isDebugEnabled()) {
            logger.debug("deviceId  : " + deviceId);
        }
        String sql = "SELECT IFNULL(ecc.serviceRate, IFNULL(oo.serviceRate, 0.0)) AS serviceRate " +
                "  FROM cisp_dev.dev_ebikechargestation ecs " +
                " INNER JOIN cisp_dev.dev_powersystemresource pss ON pss.id = ecs.id AND pss.className = 'EBikeChargeStation' " +
                "  LEFT JOIN ebike.ebike_orgopr oo ON oo.subburo = pss.subburo " +
                "  LEFT JOIN ebike.ebike_cssr_config ecc ON ecc.chargestationId = ecs.id " +
                " INNER JOIN cisp_dev.dev_ebikechargedevice ecd ON ecd.chargestationId = ecs.id " +
                " WHERE ecd.deviceId = ? " +
                " LIMIT 1 ";
        Double serviceRate;
        try {
            serviceRate = ebikeJdbcTemplate.queryForObject(sql, new Object[] {deviceId}, Double.class);
        }
        catch(EmptyResultDataAccessException erdae) {
            serviceRate = 0.0D;
        }
        return serviceRate;
    }

    @Override
    public CostInfo getCostInfoByDeviceId(String deviceId) {
        StringBuilder sqlBuilder = new StringBuilder();
        List<Object> params = new ArrayList<>();
        sqlBuilder.append("SELECT ecs.subburo, ecs.costSchemeId, ecs.costSchemeNo, ecs.costSchemeName, ecs.costUnitPrice ");
        sqlBuilder.append("  FROM ebike.ebike_costscheme ecs ");
        sqlBuilder.append(" INNER JOIN ebike.ebike_cs_cs_rela ccr ON ccr.costSchemeId = ecs.costSchemeId ");
        sqlBuilder.append(" INNER JOIN cisp_dev.dev_ebikechargedevice ecd ON ecd.chargestationId = ccr.chargestationId ");
        sqlBuilder.append(" INNER JOIN cisp_dev.dev_powersystemresource pss ON pss.`id` = ecd.`id` AND pss.className = 'EBikeChargeDevice' ");
        sqlBuilder.append(" WHERE ecd.deviceId = ? ");
        params.add(deviceId);
        sqlBuilder.append(" LIMIT 1 ");
        CostInfo ci;
        try {
            ci = ebikeJdbcTemplate.queryForObject(sqlBuilder.toString(), params.toArray(), ciRowMapper);
        }
        catch(EmptyResultDataAccessException erdae) {
            ci = null;
        }
        return ci;

    }
}
