package cn.yunrui.mqttclient.ebikesrv.mqttclient.handler.impl;

import cn.yunrui.mqttclient.ebikesrv.common.utils.*;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.cache.ChargeDeviceCache;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.cache.ChargeRecordCache;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.cache.FitconnDebuggingCache;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.cache.OpsWorkOrderCache;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.constant.ChargeOpenFlag;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.dao.MqttMsgLogDao;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.entity.*;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.handler.MqttMessageHandler;
import cn.yunrui.mqttclient.ebikesrv.syncevent.publisher.ChargeOrderEventPublisher;
import com.google.common.base.Joiner;
import net.sf.json.JSONObject;
import org.apache.commons.collections.MapUtils;
import org.eclipse.paho.client.mqttv3.*;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.math.BigDecimal;
import java.util.*;

/**
 *
 * @author Nick Zhang
 * @date 2019-03-26
 *
 */
public abstract class MqttMessageHandlerImpl implements MqttMessageHandler {

    private final static int QOS = 1;
    private final static boolean RETAINED = false;
    private final static long WAITTIMEOUT = 3000L;
    /**
     * 计时间模式，最大充电电量，9000W
     */
    private final static Double MAX_CHARGE_EC = 9000.0D;
    /**
     * 计时间模式，免费最大充电时长，600分钟
     */
    private final static Integer MAX_CHARGE_TIME_FOR_FREE = 600;
    /**
     * 计时间模式，免费最大充电时长，480分钟----适配国标
     */
    //private final static Integer MAX_CHARGE_TIME_FOR_FREE = 480;
    /**
     * 计电量模式，最大充电时间，900分钟
     */
    private final static Double MAX_CHARGE_TIME = 900.0D;
    /**
     * 计电量模式，免费最大充电电量，2000W
     */
    private final static Double MAX_CHARGE_EC_FOR_FREE = 2000.0D;

    final Logger logger = LoggerFactory.getLogger(getClass());
    final String opType;
    final boolean needRespond;

    protected final MqttClient mqttClient;
    protected final MqttMsgLogDao mqttMsgLogDao;

    ChargeDeviceCache chargeDeviceCache;
    ChargeRecordCache chargeRecordCache;
    FitconnDebuggingCache fitconnDebuggingCache;
    OpsWorkOrderCache opsWorkOrderCache;
    ChargeOrderEventPublisher chargeOrderEventPublisher;

    MqttMessageHandlerImpl(String opType, MqttClient mqttClient, MqttMsgLogDao mqttMsgLogDao, boolean needRespond) {
        this.opType = opType;
        this.mqttClient = mqttClient;
        this.mqttMsgLogDao = mqttMsgLogDao;
        this.needRespond = needRespond;
    }

    String getNewChargeRecordId(String chargeplugNo, Date chargeTime) {
        return Joiner.on("-").skipNulls().join("EB" + chargeplugNo, new DateTime(chargeTime).toString("yyyyMMddHHmmssSSS"));
    }

    Map<String, Object> getChargeRecordMapByCoinsin(BillingInfo billingInfo, Double coinMoney) {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>>> start getChargeRecordMapByCoinsin >>>>>>> ");
            logger.debug(" billingInfo  : " + (billingInfo != null ? billingInfo.toString() : null));
            logger.debug(" coinMoney    : " + coinMoney);
        }
        Double chargeMoney = 0.0D;
        coinMoney = (coinMoney != null ? coinMoney : 0.0D);
        Map<String, Object> chargeRecordMap = new HashMap<>();
        chargeRecordMap.put("chargeMoney", chargeMoney);
        chargeRecordMap.put("coinMoney", coinMoney);
        if(billingInfo != null) {
            chargeRecordMap.put("schemeId", billingInfo.getSchemeId());
            chargeRecordMap.put("schemeType", billingInfo.getSchemeType());
            chargeRecordMap.put("unitPrice1", billingInfo.getUnitPrice1());
            chargeRecordMap.put("unitPrice2", billingInfo.getUnitPrice2());
            chargeRecordMap.put("unitPrice3", billingInfo.getUnitPrice3());
            chargeRecordMap.put("unitPrice4", billingInfo.getUnitPrice4());
            chargeRecordMap.put("unitPrice5", billingInfo.getUnitPrice5());
            chargeRecordMap.put("stepCount", billingInfo.getStepCount());
            chargeRecordMap.put("stepPower1", billingInfo.getStepPower1());
            chargeRecordMap.put("stepPower2", billingInfo.getStepPower2());
            chargeRecordMap.put("stepPower3", billingInfo.getStepPower3());
            chargeRecordMap.put("stepPower4", billingInfo.getStepPower4());
            chargeRecordMap.put("minCharge", billingInfo.getMinCharge());
            chargeRecordMap.put("isReturn", billingInfo.getIsReturn());
            chargeRecordMap.put("closingPeriod", billingInfo.getClosingPeriod());
        }
        // 充电开启标志：为1表示充电等待中（等待主站回应）
        chargeRecordMap.put("chargeOpenFlag", ChargeOpenFlag.FLAG_WAITING);
        Integer firstUnitPriceFlag = 1;
        Double firstChargeAmount = 0.0D;
        Double usageAmount = 0.0D;
        Double surplusAmount = 0.0D;
        Double usageElecCons = 0.0D;
        if(billingInfo != null && StringUtils.equals("01", billingInfo.getSchemeType())) {
            // 计时间
            firstUnitPriceFlag = 1;
            if(billingInfo.getUnitPrice1() != null && billingInfo.getUnitPrice1().compareTo(0.0D) > 0) {
                // 单位为分钟，需转换
                firstChargeAmount = coinMoney * billingInfo.getUnitPrice1() * 60;
                BigDecimal bd = new BigDecimal(firstChargeAmount);
                bd = bd.setScale(4, BigDecimal.ROUND_HALF_UP);
                firstChargeAmount = bd.doubleValue();
            }
            else if(billingInfo.getUnitPrice1() != null && billingInfo.getUnitPrice1().compareTo(0.0D) == 0) {
                // 免费充电
                firstChargeAmount = (billingInfo.getMaxChargeTime() != null ? billingInfo.getMaxChargeTime() : MAX_CHARGE_TIME_FOR_FREE) * 1.0D;
            }
            surplusAmount = firstChargeAmount - usageAmount;
        }
        else {
            firstUnitPriceFlag = 1;
            firstChargeAmount = MAX_CHARGE_TIME_FOR_FREE * 1.0D;
            surplusAmount = firstChargeAmount - usageAmount;
        }
        chargeRecordMap.put("firstUnitPriceFlag", firstUnitPriceFlag);
        chargeRecordMap.put("firstChargeAmount", firstChargeAmount);
        chargeRecordMap.put("currUnitPriceFlag", firstUnitPriceFlag);
        chargeRecordMap.put("chargeAmount", firstChargeAmount);
        chargeRecordMap.put("usageAmount", usageAmount);
        chargeRecordMap.put("surplusAmount", surplusAmount);
        chargeRecordMap.put("usageElecCons", usageElecCons);
        // 计时间（投币只有计时间）
        // 剩余充电电量：单位为kWh，需转换
        chargeRecordMap.put("remainElecCons", (MAX_CHARGE_EC / 1000.0D));
        // 剩余充电时间：单位为分钟
        chargeRecordMap.put("remainTime", firstChargeAmount);
        return chargeRecordMap;
    }

    Map<String, Object> getChargeRecordMapByCard(BillingInfo billingInfo, Double chargeMoney) {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>>> start getChargeRecordMapByCard >>>>>>> ");
            logger.debug(" billingInfo  : " + (billingInfo != null ? billingInfo.toString() : null));
            logger.debug(" chargeMoney  : " + chargeMoney);
        }
        if(billingInfo == null) {
            return null;
        }
        chargeMoney = (chargeMoney != null ? chargeMoney : 0.0D);
        Double coinMoney = 0.0D;
        Map<String, Object> chargeRecordMap = new HashMap<>();
        chargeRecordMap.put("chargeMoney", chargeMoney);
        chargeRecordMap.put("coinMoney", coinMoney);
        if(billingInfo != null) {
            chargeRecordMap.put("schemeId", billingInfo.getSchemeId());
            chargeRecordMap.put("schemeType", billingInfo.getSchemeType());
            chargeRecordMap.put("unitPrice1", billingInfo.getUnitPrice1());
            chargeRecordMap.put("unitPrice2", billingInfo.getUnitPrice2());
            chargeRecordMap.put("unitPrice3", billingInfo.getUnitPrice3());
            chargeRecordMap.put("unitPrice4", billingInfo.getUnitPrice4());
            chargeRecordMap.put("unitPrice5", billingInfo.getUnitPrice5());
            chargeRecordMap.put("stepCount", billingInfo.getStepCount());
            chargeRecordMap.put("stepPower1", billingInfo.getStepPower1());
            chargeRecordMap.put("stepPower2", billingInfo.getStepPower2());
            chargeRecordMap.put("stepPower3", billingInfo.getStepPower3());
            chargeRecordMap.put("stepPower4", billingInfo.getStepPower4());
            chargeRecordMap.put("minCharge", billingInfo.getMinCharge());
            chargeRecordMap.put("isReturn", billingInfo.getIsReturn());
            chargeRecordMap.put("closingPeriod", billingInfo.getClosingPeriod());
        }
        // 充电开启标志：为1表示充电等待中（等待主站回应）
        chargeRecordMap.put("chargeOpenFlag", ChargeOpenFlag.FLAG_WAITING);
        Integer firstUnitPriceFlag = 1;
        Double firstChargeAmount = 0.0D;
        Double usageAmount = 0.0D;
        Double surplusAmount = 0.0D;
        Double usageElecCons = 0.0D;
        if(StringUtils.equals("01", billingInfo.getSchemeType())) {
            // 计时间
            firstUnitPriceFlag = 1;
            if(billingInfo.getUnitPrice1() != null && billingInfo.getUnitPrice1().compareTo(0.0D) > 0) {
                firstChargeAmount = chargeMoney * billingInfo.getUnitPrice1() * 60;
                BigDecimal bd = new BigDecimal(firstChargeAmount);
                bd = bd.setScale(4, BigDecimal.ROUND_HALF_UP);
                firstChargeAmount = bd.doubleValue();
            }
            else if(billingInfo.getUnitPrice1() != null && billingInfo.getUnitPrice1().compareTo(0.0D) == 0) {
                // 免费充电
                firstChargeAmount = (billingInfo.getMaxChargeTime() != null ? billingInfo.getMaxChargeTime() : MAX_CHARGE_TIME_FOR_FREE) * 1.0D;
            }
            surplusAmount = firstChargeAmount - usageAmount;
        }
        else if(StringUtils.equals("02", billingInfo.getSchemeType())) {
            // 计电量
            firstUnitPriceFlag = 1;
            if(billingInfo.getUnitPrice1() != null && billingInfo.getUnitPrice1().compareTo(0.0D) > 0) {
                firstChargeAmount = chargeMoney * billingInfo.getUnitPrice1();             // 单位为kWh
                BigDecimal bd = new BigDecimal(firstChargeAmount);
                bd = bd.setScale(4, BigDecimal.ROUND_HALF_UP);
                firstChargeAmount = bd.doubleValue();
            }
            else if(billingInfo.getUnitPrice1() != null && billingInfo.getUnitPrice1().compareTo(0.0D) == 0) {
                // 免费充电
                firstChargeAmount = MAX_CHARGE_EC_FOR_FREE / 1000.0;
            }
            surplusAmount = firstChargeAmount - usageAmount;
        }
        else {
            firstUnitPriceFlag = 1;
            firstChargeAmount = MAX_CHARGE_TIME_FOR_FREE * 1.0D;
            surplusAmount = firstChargeAmount - usageAmount;
        }
        chargeRecordMap.put("firstUnitPriceFlag", firstUnitPriceFlag);
        chargeRecordMap.put("firstChargeAmount", firstChargeAmount);
        chargeRecordMap.put("currUnitPriceFlag", firstUnitPriceFlag);
        chargeRecordMap.put("chargeAmount", firstChargeAmount);
        chargeRecordMap.put("usageAmount", usageAmount);
        chargeRecordMap.put("surplusAmount", surplusAmount);
        chargeRecordMap.put("usageElecCons", usageElecCons);
        if(StringUtils.equals("02", billingInfo.getSchemeType())) {
            // 计电量
            chargeRecordMap.put("remainElecCons", firstChargeAmount);               // 剩余充电电量：单位为kWh
            chargeRecordMap.put("remainTime", MAX_CHARGE_TIME);                     // 剩余充电时间：单位为分钟
        }
        else {
            // 计时间
            chargeRecordMap.put("remainElecCons", (MAX_CHARGE_EC / 1000.0D));       // 剩余充电电量：单位为kWh，需转换
            chargeRecordMap.put("remainTime", firstChargeAmount);                   // 剩余充电时间：单位为分钟
        }
        return chargeRecordMap;
    }

    /**
     * 通过设备通讯标识发布消息
     * @param deviceId
     *              设备通讯标识（MQTT规约中定义的deviceId）
     * @param message
     *              消息内容
     * @param type
     *              消息类型：start - 开始充电命令；stop - 结束充电命令；status - 设备状态；param - 设备参数；cardBalance - 卡片余额；cardCharge - 卡充电命令；attachParam - 设备附加参数；
     * @return 发送标志 true 表示发送成功；false 标识发送失败
     */
    boolean response2device(String deviceId, String message, String type) {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> start response2device >>>>>> ");
            logger.debug(" [ deviceId : " + deviceId + ", message : " + message + ", type : " + type + " ] ");
        }

        if(chargeDeviceCache.get(deviceId) != null && StringUtils.equals("hzchaoxiang", chargeDeviceCache.get(deviceId).getFactory())) {
            // 超翔设备结尾需加#
            message += "#";
        }
        else if(chargeDeviceCache.get(deviceId) != null && StringUtils.equals("silan", chargeDeviceCache.get(deviceId).getFactory())) {
            // 士兰微设备结尾需加#<time_stamp>
            message += "#" + String.format("%010d", new DateTime(DateTimeZone.forID("Etc/GMT-8")).getMillis() / 1000);
        }
        MqttTopic mqttTopicDevice = mqttClient.getTopic((chargeDeviceCache.get(deviceId) != null && StringUtils.equals("hzchaoxiang", chargeDeviceCache.get(deviceId).getFactory())) ? "$client/" + deviceId : deviceId);
        boolean result = false;
        try {
            MqttMessage mm = new MqttMessage(message.getBytes());
            mm.setQos(QOS);
            mm.setRetained(RETAINED);
            MqttDeliveryToken token = mqttTopicDevice.publish(mm);
            token.waitForCompletion(WAITTIMEOUT);
            if(token.isComplete()) {
                if(logger.isDebugEnabled()) {
                    logger.debug(" [ response2device MqttMessage ] : " + JsonConvertUtils.convertToString(mm));
                    logger.debug(" >>>>>> completion : " + token.isComplete() + " >>>>>> ");
                }
                // 插入MQTT消息日志
                mqttMsgLogDao.insertMqttMsgLog(mqttTopicDevice.getName(), mm, deviceId, type, "down", "response");
                if(StringUtils.equalsAny(type,"start", "cardCharge", "cardCharge2", "coinCharge")) {
                    String plugId = null;
                    if(StringUtils.isNotBlank(message)) {
                        String[] msgArray = message.split("#");
                        if(msgArray.length >= 2 && StringUtils.equalsAny(type,"start", "coinCharge")) {
                            plugId = msgArray[1];
                        }
                        if(msgArray.length >= 4 && StringUtils.equalsAny(type,"cardCharge", "cardCharge2")) {
                            plugId = msgArray[3];
                        }
                    }
                    if(StringUtils.isNotBlank(deviceId) && StringUtils.isNotBlank(plugId)) {
                        // 更新充电插座运行状态
                        chargeDeviceCache.updatePlugStatus(deviceId, plugId, "1");
                    }
                    // 重新加载缓存
                    ChargeDeviceProps cdp = chargeDeviceCache.reload(deviceId);
                    if(logger.isDebugEnabled()) {
                        logger.debug(" >>> 重新加载缓存 >>> ");
                        logger.debug(" ChargeDeviceProps : " + (cdp != null ? cdp.toString() : null));
                    }
                }
            }
            result = token.isComplete();
        }
        catch(MqttException _me) {
            logger.error(_me.getMessage(), _me.fillInStackTrace());
        }

        if(logger.isDebugEnabled()) {
            logger.debug(" <<<<<<< end response2device <<<<<<< ");
        }
        return result;
    }

    RefreshChargeRecordResult refreshChargeRecord(String deviceId, String plugId, Double remain, Double power, Double remainTime, Date currTime) {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> start refreshChargeRecord >>>>>> ");
            logger.debug(" deviceId     : " + deviceId);
            logger.debug(" plugId       : " + plugId);
            logger.debug(" remain       : " + remain);
            logger.debug(" power        : " + power);
            logger.debug(" remainTime   : " + remainTime);
            logger.debug(" currTime     : " + currTime);
        }
        if(remain != null) {
            // 单位转换：Wh >> kWh
            remain = DoubleUtils.defaultIfNegValue(remain, 0.0D, 0.0D) / 1000.0D;
        }
        if(power != null) {
            // 单位转换：W >> kW
            power = DoubleUtils.defaultIfNegValue(power, 0.0D, 0.0D) / 1000.0D;
        }
        if(remainTime != null) {
            // 单位为分钟
            remainTime = DoubleUtils.defaultIfNegValue(remainTime, 0.0D, 0.0D);
        }
        boolean resendParamFlag = false;
        Map<String, Object> valueMap = chargeRecordCache.getChargeRecordMap(deviceId, plugId);
        if(MapUtils.isNotEmpty(valueMap)) {
            if(logger.isDebugEnabled()) {
                logger.debug(" before handle [ valueMap ] : " + JsonConvertUtils.convertToString(valueMap));
            }
            // 处理充电等待中的充电记录
            handleWaitingChargeOpenFlag(remain, remainTime, valueMap);

            // 计算负荷特性（当前功率、最大功率、最小功率、平均功率等）
            calcPowerChar(power, currTime, valueMap);

            // 计算当前充电使用情况（使用量、剩余量、使用电量、当前剩余充电电量、当前剩余充电时间）
            if(calcChargeUsage(remain, remainTime, valueMap)) {
                resendParamFlag = true;
            }

            if(logger.isDebugEnabled()) {
                logger.debug(" after handle [ valueMap ]  : " + JsonConvertUtils.convertToString(valueMap));
            }
        }

        if(logger.isDebugEnabled()) {
            logger.debug(" <<<<<<< end refreshChargeRecord <<<<<<< ");
        }
        return new RefreshChargeRecordResult(resendParamFlag, valueMap);
    }

    String makeupParamDownMsg(ChargeDeviceProps chargeDeviceProps) {
        // 单个插座最大功率限额，单位W
        String maxPlugPower = "600";
        // 充电桩设备最大功率限额，单位W
        String maxDevicePower = "2500";
        //
        String cardPassword = "000000000000";
        // 充电桩设备中计量模块的校准系数，1000表示单位校准系数
        String factor = "1000";
        // 定时报告时间，单位分钟
        String reporttime = "5";
        String remain = "0";
        String time = "0";
        StringBuilder msg2Device = new StringBuilder();
        if(ObjectUtils.isNotNull(chargeDeviceProps)) {
            if(ObjectUtils.isNotNull(chargeDeviceProps.getParamsMap())) {
                maxPlugPower = StringUtils.defaultIfBlank(chargeDeviceProps.getParamsMap().get("maxPlugPower"), maxPlugPower);
                maxDevicePower = StringUtils.defaultIfBlank(chargeDeviceProps.getParamsMap().get("maxDevicePower"), maxDevicePower);
                cardPassword = StringUtils.defaultIfBlank(chargeDeviceProps.getParamsMap().get("cardPassword"), cardPassword);
                factor = StringUtils.defaultIfBlank(chargeDeviceProps.getParamsMap().get("factor"), factor);
                reporttime = StringUtils.defaultIfBlank(chargeDeviceProps.getParamsMap().get("reporttime"), reporttime);
            }

            msg2Device.append("param");
            if(chargeDeviceProps.getChargeDeviceProtocol().equals(ChargeDeviceProtocolEnum.MQTT_YJM2M_201710)
                    || chargeDeviceProps.getChargeDeviceProtocol().equals(ChargeDeviceProtocolEnum.MQTT_YJM2M_201712)
                    || chargeDeviceProps.getChargeDeviceProtocol().equals(ChargeDeviceProtocolEnum.MQTT_YJM2M_201801)
                    || chargeDeviceProps.getChargeDeviceProtocol().equals(ChargeDeviceProtocolEnum.MQTT_YJM2M_201804)) {
                msg2Device.append("#").append(maxPlugPower);
                msg2Device.append("#").append(maxDevicePower);
                msg2Device.append("#").append(cardPassword);
                msg2Device.append("#").append(factor);
            }
            else if(chargeDeviceProps.getChargeDeviceProtocol().equals(ChargeDeviceProtocolEnum.MQTT_EBIKE_201804)) {
                msg2Device.append("#").append(maxPlugPower);
                msg2Device.append("#").append(maxDevicePower);
                msg2Device.append("#").append(factor);
            }
            else if(chargeDeviceProps.getChargeDeviceProtocol().equals(ChargeDeviceProtocolEnum.MQTT_EBIKE_201805)) {
                /*if(StringUtils.equals("silan", chargeDeviceProps.getFactory())) {
                    maxDevicePower = NumberUtils.isParsable(maxDevicePower) && NumberUtils.isParsable(maxPlugPower) ? String.valueOf(Integer.parseInt(maxDevicePower) + Integer.parseInt(maxPlugPower) * 2) : maxDevicePower;
                }*/
                msg2Device.append("#").append(maxPlugPower);
                msg2Device.append("#").append(maxDevicePower);
                msg2Device.append("#").append(factor);
                msg2Device.append("#").append(reporttime);
            }
            else {
                msg2Device.append("#").append(maxPlugPower);
                msg2Device.append("#").append(maxDevicePower);
                msg2Device.append("#").append(cardPassword);
                msg2Device.append("#").append(factor);
            }
            String deviceId = chargeDeviceProps.getDeviceId();
            Integer plugCount = IntegerUtils.defaultIfNull(chargeDeviceProps.getPlugCount(), 0);
            if(StringUtils.isNotBlank(deviceId)) {
                for(int i = 1; i <= plugCount; i++) {
                    String plugId = String.valueOf(i);
                    String r = remain;
                    String t = time;
                    Map<String, Object> chargeRecordMap = chargeRecordCache.getChargeRecordMap(deviceId, plugId);
                    if(cn.yunrui.mqttclient.ebikesrv.common.utils.MapUtils.isNotEmpty(chargeRecordMap)) {
                        // 该充电插座有当前充电记录，返回上次正在充电记录中的剩余充电电量和剩余充电时间
                        r = String.valueOf(Math.round((Double) ObjectUtils.defaultIfNull(chargeRecordMap.get("remainElecCons"), 0.0D) * 1000.0D));
                        t = String.valueOf(Math.round((Double) ObjectUtils.defaultIfNull(chargeRecordMap.get("remainTime"), 0.0D)));
                    }

                    switch(chargeDeviceProps.getChargeDeviceProtocol()) {
                        case MQTT_YJM2M_201710:
                            // <remaini>
                            msg2Device.append("#").append(r);
                            break;
                        case MQTT_YJM2M_201712:
                            // <remaini>,<timei>
                            msg2Device.append("#").append(r).append(",").append(t);
                            break;
                        case MQTT_YJM2M_201801:
                            // <remaini>,<timei>
                            msg2Device.append("#").append(r).append(",").append(t);
                            break;
                        case MQTT_YJM2M_201804:
                            // <remaini>,<timei>
                            msg2Device.append("#").append(r).append(",").append(t);
                            break;
                        case MQTT_EBIKE_201804:
                            // <remaini>,<timei>
                            msg2Device.append("#").append(r).append(",").append(t);
                            break;
                        case MQTT_EBIKE_201805:
                            // <remaini>,<timei>
                            msg2Device.append("#").append(r).append(",").append(t);
                            break;
                        default:
                            // <remaini>,<timei>
                            msg2Device.append("#").append(r).append(",").append(t);
                            break;

                    }
                }
            }
        }
        return msg2Device.toString();
    }

    String makeupAttachParamDownMsg(ChargeDeviceProps chargeDeviceProps) {
        // 浮充充电时间，120分钟
        String floatChargeTime = "120";
        // 空载（功率为0）持续供电时间，3分钟
        String noloadTime = "180";
        // 单元充电量，默认值1000Wh/元
        String unitLimit = "1000";
        // 单元充电时间，默认值180分钟/元
        String unitLimitTime = "180";
        StringBuilder msg2Device = new StringBuilder();
        if(ObjectUtils.isNotNull(chargeDeviceProps)) {
            BillingInfo bi = chargeDeviceCache.getDefaultBillingInfoByDeviceId(chargeDeviceProps.getDeviceId());
            if(bi != null && StringUtils.equals("01", bi.getSchemeType()) && bi.getUnitPrice1() != null) {
                if(bi.getUnitPrice1() > 0) {
                    unitLimitTime = "" + Math.round(bi.getUnitPrice1() * 60.0D);
                }
                else {
                    unitLimitTime = "" + IntegerUtils.defaultIfNull(bi.getMaxChargeTime(), MAX_CHARGE_TIME_FOR_FREE);
                }
            }
            if(ObjectUtils.isNotNull(chargeDeviceProps.getAttachParamsMap()) && StringUtils.isNotBlank(chargeDeviceProps.getAttachParamsMap().get("floatChargeTime"))) {
                floatChargeTime = chargeDeviceProps.getAttachParamsMap().get("floatChargeTime");
            }

            if(ObjectUtils.isNotNull(chargeDeviceProps.getAttachParamsMap())) {
                noloadTime = StringUtils.defaultIfBlank(chargeDeviceProps.getAttachParamsMap().get("noloadTime"), noloadTime);
            }

            msg2Device.append("attachParam");
            if(chargeDeviceProps.getChargeDeviceProtocol().equals(ChargeDeviceProtocolEnum.MQTT_YJM2M_201710)
                    || chargeDeviceProps.getChargeDeviceProtocol().equals(ChargeDeviceProtocolEnum.MQTT_YJM2M_201712)
                    || chargeDeviceProps.getChargeDeviceProtocol().equals(ChargeDeviceProtocolEnum.MQTT_YJM2M_201801)
                    || chargeDeviceProps.getChargeDeviceProtocol().equals(ChargeDeviceProtocolEnum.MQTT_YJM2M_201804)) {
                // <floatChargeTime>
                msg2Device.append("#").append(floatChargeTime);
            }
            else if(chargeDeviceProps.getChargeDeviceProtocol().equals(ChargeDeviceProtocolEnum.MQTT_EBIKE_201804)) {
                if(StringUtils.equals("hzchaoxiang", chargeDeviceProps.getFactory())) {
                    try {
                        // <floatChargeTime> 转换为秒
                        msg2Device.append("#").append(String.valueOf(Integer.parseInt(floatChargeTime) * 60));
                    }
                    catch(NumberFormatException nfe) {
                        // <floatChargeTime> 转换为秒
                        msg2Device.append("#").append(String.valueOf(120 * 60));
                    }
                }
                else {
                    // <floatChargeTime>
                    msg2Device.append("#").append(floatChargeTime);
                }

                if(StringUtils.equals("hzchaoxiang", chargeDeviceProps.getFactory())) {
                    // <noloadTime> 超限设备空载供电时间除以2
                    msg2Device.append("#").append(Math.round(NumberUtils.toDouble(noloadTime) / 2.0D));
                }
                else {
                    // <noloadTime>
                    msg2Device.append("#").append(noloadTime);
                }

                // <unitLimit>
                msg2Device.append("#").append(unitLimit);
                // <unitLimitTime>
                msg2Device.append("#").append(unitLimitTime);
            }
            else if(chargeDeviceProps.getChargeDeviceProtocol().equals(ChargeDeviceProtocolEnum.MQTT_EBIKE_201805)) {
                // <floatChargeTime>
                msg2Device.append("#").append(floatChargeTime);
                // <noloadTime>
                msg2Device.append("#").append(noloadTime);
                if(StringUtils.equals("gaea", chargeDeviceProps.getFactory())) {
                    // <unitLimit>
                    msg2Device.append("#").append(unitLimit);
                    // <unitLimitTime>
                    msg2Device.append("#").append(unitLimitTime);
                }
            }
            else {
                // <floatChargeTime>
                msg2Device.append("#").append(floatChargeTime);
            }
        }
        return msg2Device.toString();
    }

    String makeupSynctimeDownMsg(ChargeDeviceProps chargeDeviceProps) {
        StringBuilder msg2Device = new StringBuilder();
        if(ObjectUtils.isNotNull(chargeDeviceProps)) {
            msg2Device.append("synctime");
            msg2Device.append("#").append(String.format("%010d", new DateTime(DateTimeZone.forID("Etc/GMT-8")).getMillis() / 1000));
        }
        return msg2Device.toString();
    }

    /**
     * 处理充电等待中的充电记录
     * @param valueMap
     *              当前充电记录值集合
     */
    private void handleWaitingChargeOpenFlag(Double remain, Double remainTime, Map<String, Object> valueMap) {
        Integer chargeOpenFlag = (Integer) ObjectUtils.defaultIfNull(valueMap.get("chargeOpenFlag"), ChargeOpenFlag.FLAG_SUCCESS);
        if(chargeOpenFlag == ChargeOpenFlag.FLAG_WAITING) {
            if((remain != null && remainTime != null && remain.compareTo(0.0D) > 0 && remainTime.compareTo(0.0D) > 0) || (remain != null && remainTime == null && remain.compareTo(0.0D) > 0) || (remain == null && remainTime != null && remainTime.compareTo(0.0D) > 0)) {
                valueMap.put("chargeOpenFlag", ChargeOpenFlag.FLAG_SUCCESS);
                valueMap.put("chargeOpenTime", new Date());
            }
            else {
                // 剩余充电电量或剩余充电时间小于等于0
                valueMap.put("chargeOpenFlag", ChargeOpenFlag.FLAG_FAILURE_STARTWAITTIMEOUT);
                /// valueMap.remove("chargeOpenTime");
                valueMap.put("chargeOpenTime", new Date());
            }
        }
    }

    /**
     * 计算负荷特性（当前功率、最大功率、最小功率、平均功率等）
     * @param power
     *              功率
     * @param valueMap
     *              当前充电记录值集合
     */
    private void calcPowerChar(Double power, Date currTime, Map<String, Object> valueMap) {
        // 最大功率
        Double maxPower = (Double) ObjectUtils.defaultIfNull(valueMap.get("maxPower"), -1.0D);
        // 最小功率
        Double minPower = (Double) ObjectUtils.defaultIfNull(valueMap.get("minPower"), -1.0D);
        // 平均功率
        Double avgPower = (Double) ObjectUtils.defaultIfNull(valueMap.get("avgPower"), 0.0D);
        // 统计功率点数
        Integer statPowerCount = (Integer) ObjectUtils.defaultIfNull(valueMap.get("statPowerCount"), 0);
        if(power != null) {
            valueMap.put("curPower", power);
            if(power.compareTo(maxPower) > 0 || maxPower.compareTo(0.0D) < 0) {
                valueMap.put("maxPower", power);
                valueMap.put("maxPowerTime", currTime != null ? currTime : new Date());
            }
            if(power.compareTo(minPower) < 0 || minPower.compareTo(0.0D) < 0) {
                valueMap.put("minPower", power);
                valueMap.put("minPowerTime", currTime != null ? currTime : new Date());
            }
            avgPower = (power + avgPower * statPowerCount) / (statPowerCount + 1);
            statPowerCount = statPowerCount + 1;
            valueMap.put("avgPower", avgPower);
            valueMap.put("statPowerCount", statPowerCount);
        }
    }

    /**
     * 计算当前充电使用情况（使用量、剩余量、使用电量、当前剩余充电电量、当前剩余充电时间）
     * @param remain
     *              剩余充电电量：单位为Wh
     * @param remainTime
     *              剩余充电时间：单位为分钟
     * @param valueMap
     *              当前充电记录值集合
     * @return true：表示当前计费单价标记值改变了
     */
    private boolean calcChargeUsage(Double remain, Double remainTime, Map<String, Object> valueMap) {
        // 是否改变充电量，默认为false
        boolean changeChargeAmount = false;
        // 01 - 计时间；02 - 计电量；
        String schemeType = (String) ObjectUtils.defaultIfNull(valueMap.get("schemeType"), "");
        // 充电量
        Double chargeAmount = (Double) ObjectUtils.defaultIfNull(valueMap.get("chargeAmount"), 0.0D);
        // 使用量
        Double usageAmount = (Double) ObjectUtils.defaultIfNull(valueMap.get("usageAmount"), 0.0D);
        // 剩余量
        Double surplusAmount = (Double) ObjectUtils.defaultIfNull(valueMap.get("surplusAmount"), 0.0D);
        // 使用电量
        Double usageElecCons = (Double) ObjectUtils.defaultIfNull(valueMap.get("usageElecCons"), 0.0D);
        // 上次剩余充电电量，单位为Wh
        Double lastRemainElecCons = (Double) ObjectUtils.defaultIfNull(valueMap.get("remainElecCons"), 0.0D);
        // 上次剩余充电时间，单位为分钟
        Double lastRemainTime = (Double) ObjectUtils.defaultIfNull(valueMap.get("remainTime"), 0.0D);
        // 本次使用电量，单位为Wh
        Double ttUsageElecCons = 0.0D;
        // 本次使用时间，单位为分钟
        Double ttUsageTime = 0.0D;

        // 计算本次使用电量、本次使用时间
        ttUsageElecCons = (remain != null && remain.compareTo(0.0D) > 0 && (lastRemainElecCons - remain) > 0.0D) ? (lastRemainElecCons - remain) : 0.0D;
        ttUsageTime = (remainTime != null && remainTime.compareTo(0.0D) > 0 && (lastRemainTime - remainTime) > 0.0D) ? (lastRemainTime - remainTime) : 0.0D;

        if(StringUtils.equals("02", schemeType)) {
            // 计电量
            surplusAmount = (remain != null ? remain : surplusAmount);
            if(chargeAmount.compareTo(0.0D) >= 0 && surplusAmount.compareTo(0.0D) >= 0 && chargeAmount.compareTo(surplusAmount) >= 0) {
                usageAmount = chargeAmount - surplusAmount;
                usageElecCons = usageAmount;
            }



        } else if (StringUtils.equals("05",schemeType)) {
            surplusAmount = (remain != null ? remain : surplusAmount);
            if(chargeAmount.compareTo(0.0D) >= 0 && surplusAmount.compareTo(0.0D) >= 0 && chargeAmount.compareTo(surplusAmount) >= 0) {
                usageAmount = chargeAmount - surplusAmount;
                usageElecCons = usageAmount;
            }
            // 判断是否改变当前充电计费方案标记
            Double chargeMoney = (Double) valueMap.get("chargeMoney");
            Double maxPower = (Double) valueMap.get("maxPower");
            // 功率阶梯数，最大为5个阶梯
            Integer stepCount = (Integer) valueMap.get("stepCount");
            // 阶梯功率1，单位为W
            Integer stepPower1 = (Integer) valueMap.get("stepPower1");
            // 阶梯功率2，单位为W
            Integer stepPower2 = (Integer) valueMap.get("stepPower2");
            // 阶梯功率3，单位为W
            Integer stepPower3 = (Integer) valueMap.get("stepPower3");
            // 阶梯功率4，单位为W
            Integer stepPower4 = (Integer) valueMap.get("stepPower4");
            // 计费单价1 --单价
            Double unitPrice1 = (Double) valueMap.get("unitPrice1");
            // 计费单价2
            Double unitPrice2 = (Double) valueMap.get("unitPrice2");
            // 计费单价3
            Double unitPrice3 = (Double) valueMap.get("unitPrice3");
            // 计费单价4
            Double unitPrice4 = (Double) valueMap.get("unitPrice4");
            // 计费单价5
            Double unitPrice5 = (Double) valueMap.get("unitPrice5");

            //判断最大功率在阶梯功率多少区间
            List<Integer> validSteps = new LinkedList<>();
            if (stepCount >= 1 && stepPower1 != null) validSteps.add(stepPower1);
            if (stepCount >= 2 && stepPower2 != null && stepPower2 != 0) validSteps.add(stepPower2);
            if (stepCount >= 3 && stepPower3 != null && stepPower3 != 0) validSteps.add(stepPower3);
            if (stepCount >= 4 && stepPower4 != null && stepPower4 != 0) validSteps.add(stepPower4);

            List<Double> unitPriceList = new LinkedList<>();
            if (stepCount >= 1 && stepPower1 != null) unitPriceList.add(unitPrice2);
            if (stepCount >= 2 && stepPower2 != null && unitPrice3 != 0 ) unitPriceList.add(unitPrice3);
            if (stepCount >= 3 && stepPower3 != null && unitPrice4 != 0 ) unitPriceList.add(unitPrice4);
            if (stepCount >= 4 && stepPower4 != null && unitPrice5 != 0) unitPriceList.add(unitPrice5);

            // 2. 判断maxPower所在区间
            int targetInterval = 0; // 默认值，可根据业务调整
            for (int i = 0; i < validSteps.size(); i++) {
                if (maxPower*1000 <= validSteps.get(i).doubleValue()) {
                    targetInterval = i+1; // 第i+1个阶梯（索引从0开始）
                    break;
                }
            }
            // 3. 若超过所有阶梯，区间为"阶梯数+1"
            if (targetInterval == 0) {
                targetInterval = validSteps.size()+1;
            }

            // 当前计费单价标记
            Integer currUnitPriceFlag = (Integer) valueMap.get("currUnitPriceFlag");
            if (currUnitPriceFlag!=targetInterval){
                //如超过所有阶梯 最大阶梯值+200
                Integer powerResult =targetInterval>validSteps.size()?validSteps.get(targetInterval-2)+200:validSteps.get(targetInterval-1);
                Double addPrice = unitPriceList.get(targetInterval-1)/(Double.valueOf(powerResult)*0.001);
                Double resultPrice = unitPrice1 + addPrice;
                changeChargeAmount = true;
                Double resultAmount = chargeMoney / resultPrice;
                surplusAmount = resultAmount.compareTo(usageAmount) > 0 ? (resultAmount - usageAmount) : resultAmount;
                valueMap.put("currUnitPriceFlag", targetInterval);
                valueMap.put("chargeAmount", resultAmount);
                if (valueMap.get("schemeDetails")!=null){
                    JSONObject schemeDetails = JSONObject.fromObject(valueMap.get("schemeDetails").toString());
                    schemeDetails.put("charge_service_fee",addPrice);
                    valueMap.put("schemeDetails", schemeDetails.toString());
                }
            }

        } else {
            // 计时间
            surplusAmount = (remainTime != null ? remainTime : surplusAmount);
            // 计时间针对性处理总时间，由于佳和新充电桩最大只支持999分钟的充电时间，故当前数据库如果充电时间超过999，默认当做999处理
            if(chargeAmount > 999){
                chargeAmount = 999D;
            }

            if(chargeAmount.compareTo(0.0D) >= 0 && surplusAmount.compareTo(0.0D) >= 0 && chargeAmount.compareTo(surplusAmount) >= 0) {
                usageAmount = chargeAmount - surplusAmount;
                usageElecCons = usageElecCons + ttUsageElecCons;
            }

            // 判断是否改变当前充电计费方案标记
            Double chargeMoney = (Double) valueMap.get("chargeMoney");
            Double maxPower = (Double) valueMap.get("maxPower");
            Double avgPower = (Double) valueMap.get("avgPower");
            // 功率阶梯数，最大为5个阶梯
            Integer stepCount = (Integer) valueMap.get("stepCount");
            // 阶梯功率1，单位为W
            Integer stepPower1 = (Integer) valueMap.get("stepPower1");
            // 阶梯功率2，单位为W
            Integer stepPower2 = (Integer) valueMap.get("stepPower2");
            // 阶梯功率3，单位为W
            Integer stepPower3 = (Integer) valueMap.get("stepPower3");
            // 阶梯功率4，单位为W
            Integer stepPower4 = (Integer) valueMap.get("stepPower4");
            // 计费单价1
            Double unitPrice1 = (Double) valueMap.get("unitPrice1");
            // 计费单价2
            Double unitPrice2 = (Double) valueMap.get("unitPrice2");
            // 计费单价3
            Double unitPrice3 = (Double) valueMap.get("unitPrice3");
            // 计费单价4
            Double unitPrice4 = (Double) valueMap.get("unitPrice4");
            // 计费单价5
            Double unitPrice5 = (Double) valueMap.get("unitPrice5");

            // 当前计费单价标记
            Integer currUnitPriceFlag = (Integer) valueMap.get("currUnitPriceFlag");
            Integer unitPriceFlag = 0;
            Double unitPrice = 0.0D;

            //
            if(stepCount < 1) {
                stepCount = 1;
            }
            else if(stepCount > 5) {
                stepCount = 5;
            }

            if(stepCount == 1) {
                unitPriceFlag = 1;
                unitPrice = unitPrice1;
            }
            else if(stepCount == 2) {
                if(Double.compare(NumberUtils.min(maxPower, avgPower), stepPower1 / 1000.0D) <= 0) {
                    unitPriceFlag = 1;
                    unitPrice = unitPrice1;
                }
                else {
                    unitPriceFlag = 2;
                    unitPrice = unitPrice2;
                }
            }
            else if(stepCount == 3) {
                if(Double.compare(NumberUtils.min(maxPower, avgPower), stepPower1 / 1000.0D) <= 0) {
                    unitPriceFlag = 1;
                    unitPrice = unitPrice1;
                }
                else if(Double.compare(NumberUtils.min(maxPower, avgPower), stepPower1 / 1000.0D) > 0 && Double.compare(NumberUtils.min(maxPower, avgPower), stepPower2 / 1000.0D) <= 0) {
                    unitPriceFlag = 2;
                    unitPrice = unitPrice2;
                }
                else {
                    unitPriceFlag = 3;
                    unitPrice = unitPrice3;
                }
            }
            else if(stepCount == 4) {
                if(Double.compare(NumberUtils.min(maxPower, avgPower), stepPower1 / 1000.0D) <= 0) {
                    unitPriceFlag = 1;
                    unitPrice = unitPrice1;
                }
                else if(Double.compare(NumberUtils.min(maxPower, avgPower), stepPower1 / 1000.0D) > 0 && Double.compare(NumberUtils.min(maxPower, avgPower), stepPower2 / 1000.0D) <= 0) {
                    unitPriceFlag = 2;
                    unitPrice = unitPrice2;
                }
                else if(Double.compare(NumberUtils.min(maxPower, avgPower), stepPower2 / 1000.0D) > 0 && Double.compare(NumberUtils.min(maxPower, avgPower), stepPower3 / 1000.0D) <= 0) {
                    unitPriceFlag = 3;
                    unitPrice = unitPrice3;
                }
                else {
                    unitPriceFlag = 4;
                    unitPrice = unitPrice4;
                }
            }
            else {
                if(Double.compare(NumberUtils.min(maxPower, avgPower), stepPower1 / 1000.0D) <= 0) {
                    unitPriceFlag = 1;
                    unitPrice = unitPrice1;
                }
                else if(Double.compare(NumberUtils.min(maxPower, avgPower), stepPower1 / 1000.0D) > 0 && Double.compare(NumberUtils.min(maxPower, avgPower), stepPower2 / 1000.0D) <= 0) {
                    unitPriceFlag = 2;
                    unitPrice = unitPrice2;
                }
                else if(Double.compare(NumberUtils.min(maxPower, avgPower), stepPower2 / 1000.0D) > 0 && Double.compare(NumberUtils.min(maxPower, avgPower), stepPower3 / 1000.0D) <= 0) {
                    unitPriceFlag = 3;
                    unitPrice = unitPrice3;
                }
                else if(Double.compare(NumberUtils.min(maxPower, avgPower), stepPower3 / 1000.0D) > 0 && Double.compare(NumberUtils.min(maxPower, avgPower), stepPower4 / 1000.0D) <= 0) {
                    unitPriceFlag = 4;
                    unitPrice = unitPrice4;
                }
                else {
                    unitPriceFlag = 5;
                    unitPrice = unitPrice5;
                }
            }

            if(unitPriceFlag.compareTo(currUnitPriceFlag) > 0 && unitPrice.compareTo(0.0D) > 0) {
                // 单位为分钟，需转换
                Double currChargeAmount = chargeMoney * unitPrice * 60;
                BigDecimal bd = new BigDecimal(currChargeAmount);
                bd = bd.setScale(4, BigDecimal.ROUND_HALF_UP);
                currChargeAmount = bd.doubleValue();
                if(currChargeAmount.compareTo(chargeAmount) < 0) {
                    // 改变充电量
                    changeChargeAmount = true;
                    chargeAmount = currChargeAmount;
                    surplusAmount = chargeAmount.compareTo(usageAmount) > 0 ? (chargeAmount - usageAmount) : 0.0D;
                    remainTime = surplusAmount;
                    valueMap.put("currUnitPriceFlag", unitPriceFlag);
                    valueMap.put("chargeAmount", chargeAmount);
                }
            }
        }
        valueMap.put("usageAmount", usageAmount);
        valueMap.put("surplusAmount", surplusAmount);
        valueMap.put("usageElecCons", usageElecCons);
        if(remain != null) {
            valueMap.put("remainElecCons", remain);
        }
        if(remainTime != null) {
            valueMap.put("remainTime", remainTime);
        }
        return changeChargeAmount;
    }

    @Override
    public void setChargeDeviceCache(ChargeDeviceCache chargeDeviceCache) {
        this.chargeDeviceCache = chargeDeviceCache;
    }

    @Override
    public void setChargeRecordCache(ChargeRecordCache chargeRecordCache) {
        this.chargeRecordCache = chargeRecordCache;
    }

    @Override
    public void setFitconnDebuggingCache(FitconnDebuggingCache fitconnDebuggingCache) {
        this.fitconnDebuggingCache = fitconnDebuggingCache;
    }

    @Override
    public void setOpsWorkOrderCache(OpsWorkOrderCache opsWorkOrderCache) {
        this.opsWorkOrderCache = opsWorkOrderCache;
    }

    @Override
    public void setChargeOrderEventPublisher(ChargeOrderEventPublisher chargeOrderEventPublisher) {
        this.chargeOrderEventPublisher = chargeOrderEventPublisher;
    }

    /**
     * 处理上行（设备到服务器）消息
     * @param chargeDeviceProps
     *              充电设备属性
     * @param mqttMessage
     *              MQTT消息
     * @return 消息处理回应
     */
    @Override
    public MsgHandleResponse handleUpMessage(ChargeDeviceProps chargeDeviceProps, MqttMessage mqttMessage) {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> start handleUpMessage >>>>>> ");
            logger.debug(" chargeDeviceProps : " + JsonConvertUtils.convertToString(chargeDeviceProps));
        }
        MsgHandleResponse res = new MsgHandleResponse();
        res.setStatus(0);
        res.setMsg("处理上行（设备到服务器）消息成功");
        if(needRespond && res.getStatus() == 0) {
            res = handleDownMessage(chargeDeviceProps);
        }
        if(logger.isDebugEnabled()) {
            logger.debug(" <<<<<<< end handleUpMessage <<<<<<< ");
        }
        return res;
    }

    /**
     * 处理下行（服务器到设备）消息
     * @param chargeDeviceProps
     *              充电设备属性
     * @return 消息处理回应
     */
    protected MsgHandleResponse handleDownMessage(ChargeDeviceProps chargeDeviceProps) {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> start handleDownMessage >>>>>> ");
            logger.debug(" chargeDeviceProps    : " + JsonConvertUtils.convertToString(chargeDeviceProps));
        }
        MsgHandleResponse res = new MsgHandleResponse();
        res.setStatus(0);
        res.setMsg("处理下行（服务器到设备）消息成功");
        if(logger.isDebugEnabled()) {
            logger.debug(" <<<<<<< end handleDownMessage <<<<<<< ");
        }
        return res;
    }

    /**
     * 处理下行（服务器到设备）消息
     * @param chargeDeviceProps
     *              充电设备属性
     * @param plugId
     *              插座通讯标识
     * @return 消息处理回应
     */
    protected MsgHandleResponse handleDownMessage(ChargeDeviceProps chargeDeviceProps, String plugId) {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> start handleDownMessage >>>>>> ");
            logger.debug(" chargeDeviceProps    : " + JsonConvertUtils.convertToString(chargeDeviceProps));
            logger.debug(" plugId               : " + plugId);
        }
        MsgHandleResponse res = new MsgHandleResponse();
        res.setStatus(0);
        res.setMsg("处理下行（服务器到设备）消息成功");
        if(logger.isDebugEnabled()) {
            logger.debug(" <<<<<<< end handleDownMessage <<<<<<< ");
        }
        return res;
    }

}
