package cn.yunrui.mqttclient.ebikesrv.mqttclient.handler.impl;

import cn.yunrui.mqttclient.ebikesrv.common.utils.*;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.constant.ChargeOpenFlag;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.constant.FinishReason;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.dao.MqttMsgLogDao;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.entity.*;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.handler.MqttMessageHandler;
import com.google.common.collect.Maps;
import org.eclipse.paho.client.mqttv3.MqttClient;
import org.eclipse.paho.client.mqttv3.MqttMessage;

import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * 设备各插座的状态的查询
 *
 * @author Nick Zhang
 * @date 2019-03-26
 *
 */
public class StatusHandlerImpl extends MqttMessageHandlerImpl implements MqttMessageHandler {

    private final static long START_RESULT_TIMEOUT = 10000L;

    public StatusHandlerImpl(MqttClient mqttClient, MqttMsgLogDao mqttMsgLogDao) {
        super("status", mqttClient, mqttMsgLogDao, true);
    }

    @Override
    public MsgHandleResponse handleUpMessage(ChargeDeviceProps chargeDeviceProps, MqttMessage mqttMessage) {
        // 设备到服务器[MQTT_YJM2M_201710]：<deviceId>#status#<remain1>#<power1>#<remain2>#<power2>#<remain3>#<power3>#...
        // 设备到服务器[MQTT_YJM2M_201712] / [MQTT_YJM2M_201801] / [MQTT_YJM2M_201804]：<deviceId>#status#<remain1>#<power1>#<remain2>#<power2>#<remain3>#<power3>#...#<time1>#<time2>#<time3>#...
        // 设备到服务器[MQTT_EBIKE_201804]：<deviceId>#status#<remain1>,<time1>,<power1>#<remain2>,<time2>,<power2>#<remain3>,<time3>,<power3>#…
        // <remain1>表示插座1剩余未消耗的电量，power1表示插座1当前的功率，如果没有在充电则返回的值都为0，如果remain为-1，表示该路的继电器出现故障
        // <time1>表示插座1剩余充电时间，如果没有在充电则返回的值都为0
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> start handleUpMessage >>>>>> ");
        }

        MsgHandleResponse res = new MsgHandleResponse();
        res.setStatus(-1);
        res.setMsg("处理上行（设备到服务器）消息失败 <<< 设备各插座的状态的查询");

        if(chargeDeviceProps != null && StringUtils.isNotBlank(chargeDeviceProps.getDeviceId())) {
            Integer plugSn = 0;
            // 获取未复归（设备离线复归 | 设备停电复归）告警列表
            List<OpsWorkOrder> opsWorkOrderList101 = opsWorkOrderCache.getOpsWorkOrderList(chargeDeviceProps.getDeviceId(), String.valueOf(plugSn), "101", "N");    // 101 - 设备离线
            List<OpsWorkOrder> opsWorkOrderList102 = opsWorkOrderCache.getOpsWorkOrderList(chargeDeviceProps.getDeviceId(), String.valueOf(plugSn), "102", "N");    // 102 - 设备停电
            List<OpsWorkOrder> opsWorkOrderList103 = opsWorkOrderCache.getOpsWorkOrderList(chargeDeviceProps.getDeviceId(), String.valueOf(plugSn), "103", "N");    // 103 - 设备物联卡疑似欠费
            List<OpsWorkOrder> opsWorkOrderList = new ArrayList<>();
            opsWorkOrderList.addAll(opsWorkOrderList101);
            opsWorkOrderList.addAll(opsWorkOrderList102);
            opsWorkOrderList.addAll(opsWorkOrderList103);
            if(CollectionUtils.sizeIsNotEmptyIgnoreNull(opsWorkOrderList)) {
                for(OpsWorkOrder opsWorkOrder : opsWorkOrderList) {
                    opsWorkOrder.setIsRecovery("Y");
                    opsWorkOrder.setRecoveryTime(new Date());
                    opsWorkOrderCache.recoveryOpsWorkOrder(opsWorkOrder);

                    if(StringUtils.equals("10", opsWorkOrder.getOpsworkorderStatus())) {        // 10 – 未处理
                        // 运维工单状态为未处理，则直接归档
                        Map<String, Object> archiveMap = new HashMap<>();
                        archiveMap.put("archiver", "EBIKE");
                        archiveMap.put("archiveType", "10");            // 10 – 直接归档
                        archiveMap.put("archiveTime", new Date());
                        archiveMap.put("archiveDesc", "直接归档");
                        opsWorkOrderCache.archiveOpsWorkOrder(opsWorkOrder, archiveMap);
                    }
                }
            }

            String deviceId = chargeDeviceProps.getDeviceId();
            String rcvdMsg = new String(mqttMessage.getPayload());
            String[] rcvdMsgArray = rcvdMsg.split("#");
            Integer plugCount = chargeDeviceProps.getPlugCount() != null ? chargeDeviceProps.getPlugCount() : 0;
            Double[] remains = new Double[plugCount];
            Double[] powers = new Double[plugCount];
            Double[] times = new Double[plugCount];
            Date currTime = new Date();
            if(chargeDeviceProps.getChargeDeviceProtocol().equals(ChargeDeviceProtocolEnum.MQTT_EBIKE_201804) || chargeDeviceProps.getChargeDeviceProtocol().equals(ChargeDeviceProtocolEnum.MQTT_EBIKE_201805)) {
                if(rcvdMsgArray.length >= plugCount + 2) {              // MQTT_EBIKE_201804
                    for(int i = 0; i < plugCount; i++) {
                        String strPlugValue = rcvdMsgArray[2 + i];
                        String[] plugValArray = strPlugValue.split(",");
                        String strRemain = plugValArray.length >= 1 ? plugValArray[0] : null;
                        String strTime = plugValArray.length >= 2 ? plugValArray[1] : null;
                        String strPower = plugValArray.length >= 3 ? plugValArray[2] : null;
                        remains[i] = NumberUtils.isParsable(strRemain) ? Double.parseDouble(strRemain) : null;
                        powers[i] = NumberUtils.isParsable(strPower) ? Double.parseDouble(strPower) : null;
                        times[i] = NumberUtils.isParsable(strTime) ? Double.parseDouble(strTime) : null;
                    }
                }
            }
            else {
                if(rcvdMsgArray.length >= plugCount * 3 + 2) {              // MQTT_YJM2M_201712 / MQTT_YJM2M_201801 / MQTT_YJM2M_201804
                    for(int i = 0; i < plugCount; i++) {
                        String strRemain = rcvdMsgArray[2 + i * 2];
                        String strPower = rcvdMsgArray[2 + i * 2 + 1];
                        String strTime = rcvdMsgArray[2 + plugCount * 3 - (plugCount - i)];
                        remains[i] = NumberUtils.isParsable(strRemain) ? Double.parseDouble(strRemain) : null;
                        powers[i] = NumberUtils.isParsable(strPower) ? Double.parseDouble(strPower) : null;
                        times[i] = NumberUtils.isParsable(strTime) ? Double.parseDouble(strTime) : null;
                    }
                }
                else if(rcvdMsgArray.length >= plugCount * 2 + 2) {         // MQTT_YJM2M_201710
                    for(int i = 0; i < plugCount; i++) {
                        String strRemain = rcvdMsgArray[2 + i * 2];
                        String strPower = rcvdMsgArray[2 + i * 2 + 1];
                        remains[i] = NumberUtils.isParsable(strRemain) ? Double.parseDouble(strRemain) : null;
                        powers[i] = NumberUtils.isParsable(strPower) ? Double.parseDouble(strPower) : null;
                    }
                }
            }

            if(logger.isDebugEnabled()) {
                logger.debug(" deviceId     : " + deviceId);
                logger.debug(" remains      : " + JsonConvertUtils.convertToString(remains));
                logger.debug(" powers       : " + JsonConvertUtils.convertToString(powers));
                logger.debug(" times        : " + JsonConvertUtils.convertToString(times));
            }

            Map<String, Object> chargedeviceLoadMap = new HashMap<>();
            chargedeviceLoadMap.put("CHGDEVICEID", chargeDeviceProps.getChargedeviceId());
            chargedeviceLoadMap.put("CHGDEVICENO", chargeDeviceProps.getChargedeviceNo());
            chargedeviceLoadMap.put("LOADTIME", DateFormatUtils.format(currTime, "yyyyMMddHHmmss"));
            chargedeviceLoadMap.put("DEVICEID", chargeDeviceProps.getDeviceId());
            chargedeviceLoadMap.put("PLUG_COUNT", plugCount);
            Double sumLoadValue = 0.0D;
            // 当前充电记录更新
            for(int i = 1; i <= plugCount; i++) {
                Map<String, Object> chargeplugLoadMap = new HashMap<>();
                chargeplugLoadMap.put("CHGPLUGNO", chargeDeviceProps.getChargedeviceNo() + String.format("%02d", i));
                chargeplugLoadMap.put("LOADTIME", DateFormatUtils.format(currTime, "yyyyMMddHHmmss"));
                String plugId = String.valueOf(i);
                Double remain = remains[i - 1];
                Double power = powers[i - 1];
                Double time = times[i - 1];
                Double loadValue = DoubleUtils.defaultIfNegValue(power, 0.0D, 0.0D) / 1000.0D;  // 单位转换：W >> kW
                chargeplugLoadMap.put("LOADVALUE", loadValue);
                chargedeviceLoadMap.put("P" + String.format("%02d", i) + "_LOADVALUE", loadValue);
                sumLoadValue += loadValue;
                if(chargeRecordCache.checkIsCharging(deviceId, plugId)) {
                    Map<String, Object> chargeorderLoadMap = new HashMap<>();
                    // 当前充电记录更新
                    RefreshChargeRecordResult result = refreshChargeRecord(deviceId, plugId, remain, power, time, currTime);
                    if(logger.isDebugEnabled()) {
                        logger.debug(" RefreshChargeRecordResult : " + JsonConvertUtils.convertToString(result));
                    }
                    boolean resendParam = result.isResendParamFlag();
                    Map<String, Object> valueMap = result.getChargeRecordValueMap();
                    chargeorderLoadMap.put("CHGORDERNO", valueMap.get("chargeRecordId"));
                    chargeorderLoadMap.put("LOADTIME", DateFormatUtils.format(currTime, "yyyyMMddHHmmss"));
                    chargeorderLoadMap.put("LOADVALUE", loadValue);
                    if(MapUtils.isNotEmpty(valueMap) && ObjectUtils.isNotNull(valueMap.get("chargeTime"))) {
                        Integer chargeOpenFlag = (Integer) ObjectUtils.defaultIfNull(valueMap.get("chargeOpenFlag"), ChargeOpenFlag.FLAG_WAITING);
                        Date chargeTime = (Date) valueMap.get("chargeTime");
                        long timeInterval = System.currentTimeMillis() - chargeTime.getTime();
                        if(logger.isDebugEnabled()) {
                            logger.debug(" chargeOpenFlag       : " + chargeOpenFlag);
                            logger.debug(" timeInterval         : " + timeInterval);
                            logger.debug(" chargeRecordValueMap : " + JsonConvertUtils.convertToString(valueMap));
                        }
                        if(chargeOpenFlag == ChargeOpenFlag.FLAG_SUCCESS || timeInterval > START_RESULT_TIMEOUT) {
                            if((remain != null && time != null && remain.compareTo(0.0D) > 0 && time.compareTo(0.0D) > 0) || (remain != null && time == null && remain.compareTo(0.0D) > 0) || (remain == null && time != null && time.compareTo(0.0D) > 0)) {
                                if(chargeOpenFlag == ChargeOpenFlag.FLAG_WAITING) {
                                    valueMap.put("chargeOpenFlag", ChargeOpenFlag.FLAG_SUCCESS);
                                    valueMap.put("chargeOpenTime", new Date());
                                    chargeRecordCache.updateChargeRecord(deviceId, plugId, valueMap);
                                    chargeOrderEventPublisher.publishEvent("open", chargeRecordCache.getBeChargingRecord((String) valueMap.get("chargeRecordId")));
                                }
                                else {
                                    chargeRecordCache.updateChargeRecord(deviceId, plugId, valueMap);
                                }

                                if(resendParam) {
                                    Double remainElecCons = (Double) valueMap.get("remainElecCons");
                                    Double remainTime = (Double) valueMap.get("remainTime");
                                    String msg2Device = "start" + "#" + plugId + "#" + Math.round(remainElecCons * 1000.0D) + "#" + Math.round(remainTime);
                                    response2device(deviceId, msg2Device, "start");
                                    try {
                                        TimeUnit.MILLISECONDS.sleep(100L);
                                    }
                                    catch(InterruptedException e) {
                                        logger.error(e.getMessage(), e.fillInStackTrace());
                                    }
                                }
                            }
                            else {                                      // 剩余充电电量或剩余充电时间小于等于0，结束充电记录
                                if(logger.isDebugEnabled()) {
                                    logger.debug(" >>> 剩余充电电量或剩余充电时间小于等于0，结束充电记录 >>> ");
                                    logger.debug(" chargeOpenFlag   : " + chargeOpenFlag);
                                    logger.debug(" remain           : " + remain);
                                    logger.debug(" time             : " + time);
                                }
                                // 结束充电记录
                                Map<String, Object> finishValueMap = Maps.newHashMap();
                                if(chargeOpenFlag == ChargeOpenFlag.FLAG_FAILURE_STARTWAITTIMEOUT) {
                                    Map<String, Object> mapTemp = Maps.newHashMap();
                                    mapTemp.put("chargeOpenFlag", chargeOpenFlag);
                                    mapTemp.put("chargeOpenTime", new Date());
                                    chargeRecordCache.updateChargeRecord(deviceId, plugId, mapTemp);
                                    finishValueMap.put("finishReason", FinishReason.REASON_STARTWAITTIMEOUT);       // -99表示充电开启超时，结束充电记录
                                }
                                else {
                                    chargeRecordCache.updateChargeRecord(deviceId, plugId, valueMap);
                                    finishValueMap.put("finishReason", FinishReason.REASON_SURPLUS_ZERO);           // -97剩余充电电量（或剩余充电时间）为零，但结束命令未收到，结束充电记录
                                }
                                finishValueMap.put("finishPower", power / 1000.0D);
                                chargeRecordCache.finishChargeRecord(deviceId, plugId, finishValueMap);
                                chargeOrderEventPublisher.publishEvent("finish", chargeRecordCache.getHistroyChargeRecord((String) valueMap.get("chargeRecordId")));
                                // 充电设备状态更新
                                if(StringUtils.equals("1", chargeDeviceProps.getOpStatus())) {
                                    chargeDeviceCache.updatePlugStatus(deviceId, plugId, "0");              // 当设备状态为在线时，插座状态置为 0 - 空闲
                                }
                                else {
                                    chargeDeviceCache.updatePlugStatus(deviceId, plugId, "9");              // 当设备状态为离线时，插座状态置为 9 - 不可用（设备离线）
                                }

                                // 如果剩余电量和剩余时间有一个不为0，则下发结束命令
                                if((remain != null && remain.compareTo(0.0D) > 0) || (time != null && time.compareTo(0.0D) > 0)) {
                                    String msg2Device = "stop" + "#" + plugId;
                                    response2device(deviceId, msg2Device, "stop");
                                }
                            }
                        }
                    }
                    chargeRecordCache.saveChargeorderLoadMap(chargeorderLoadMap);
                }
                else {
                    if((remain != null && time != null && remain.compareTo(0.0D) > 0 && time.compareTo(0.0D) > 0) || (remain != null && time == null && remain.compareTo(0.0D) > 0) || (remain == null && time != null && time.compareTo(0.0D) > 0)) {
                        if(chargeDeviceProps.getPlugPropsMap() != null && chargeDeviceProps.getPlugPropsMap().get(Integer.parseInt(plugId)) != null) {
                            chargeDeviceProps.getPlugPropsMap().get(Integer.parseInt(plugId)).setStatus("U");
                            chargeDeviceProps.getPlugPropsMap().get(Integer.parseInt(plugId)).setOpStatus("1");
                        }
                    }
                    else {
                        // 剩余充电电量或剩余充电时间小于等于0
                        if(fitconnDebuggingCache.checkIsDebugging(deviceId, plugId)) {
                            // 该插座在装接调试，结束装接调试日志记录
                            Map<String, Object> valueMap = new HashMap<>();
                            valueMap.put("debugFinishTime", new Date());
                            fitconnDebuggingCache.finishFitconnDebuggingLog(deviceId, plugId, valueMap);
                            // 充电设备状态更新
                            if(StringUtils.equals("1", chargeDeviceProps.getOpStatus())) {
                                chargeDeviceCache.updatePlugStatus(deviceId, plugId, "0");              // 当设备状态为在线时，插座状态置为 0 - 空闲
                            }
                            else {
                                chargeDeviceCache.updatePlugStatus(deviceId, plugId, "9");              // 当设备状态为离线时，插座状态置为 9 - 不可用（设备离线）
                            }
                        }
                    }
                }
                chargeRecordCache.saveChargeplugLoadMap(chargeplugLoadMap);
            }
            chargedeviceLoadMap.put("SUM_LOADVALUE", sumLoadValue);
            chargeRecordCache.saveChargedeviceLoadMap(chargedeviceLoadMap);

            res.setStatus(0);
            res.setMsg("处理上行（设备到服务器）消息成功 <<< 设备各插座的状态的查询");

            if(needRespond && res.getStatus() == 0) {
                res = handleDownMessage(chargeDeviceProps);
            }
        }

        if(logger.isDebugEnabled()) {
            logger.debug(" <<<<<<< end handleUpMessage <<<<<<< ");
        }
        return res;
    }

    @Override
    public MsgHandleResponse handleDownMessage(ChargeDeviceProps chargeDeviceProps) {
        // 服务器到设备：status#<status1>#<status2>#<status3>#....
        // <status1>表示插座1当前的状态:F－空闲 U－正在使用 R－已被预订
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> start handleDownMessage >>>>>> ");
        }
        MsgHandleResponse res = new MsgHandleResponse();
        res.setStatus(-1);
        res.setMsg("处理下行（服务器到设备）消息失败 >>> 设备各插座的状态的查询");
        if(chargeDeviceProps != null && chargeDeviceProps.getDeviceId() != null) {
            Integer plugCount = chargeDeviceProps.getPlugCount() != null ? chargeDeviceProps.getPlugCount() : 0;
            StringBuilder msg2Device = new StringBuilder();
            msg2Device.append(opType);
            for(int i = 1; i <= plugCount; i++) {
                String plugStatus = null;
                if(chargeDeviceProps.getPlugPropsMap() != null && chargeDeviceProps.getPlugPropsMap().get(i) != null) {
                    plugStatus = chargeDeviceProps.getPlugPropsMap().get(i).getStatus();
                }
                plugStatus = StringUtils.isNotBlank(plugStatus) ? plugStatus.trim() : "F";  // 插座当前状态，默认为F空闲
                msg2Device.append("#").append(plugStatus);                  // <statusi>
            }
            boolean bp = response2device(chargeDeviceProps.getDeviceId(), msg2Device.toString(), opType);
            if(bp) {
                res.setStatus(0);
                res.setMsg("处理下行（服务器到设备）消息成功 >>> 设备各插座的状态的查询");
            }
        }
        if(logger.isDebugEnabled()) {
            logger.debug(" <<<<<<< end handleDownMessage <<<<<<< ");
        }
        return res;
    }

}
