package cn.yunrui.mqttclient.ebikesrv.mqttclient.handler.impl;

import cn.yunrui.mqttclient.ebikesrv.mqttclient.constant.ChargeOpenFlag;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.constant.FinishReason;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.dao.MqttMsgLogDao;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.entity.ChargeDeviceProps;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.entity.ChargeDeviceProtocolEnum;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.entity.MsgHandleResponse;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.entity.RefreshChargeRecordResult;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.handler.MqttMessageHandler;
import cn.yunrui.mqttclient.ebikesrv.common.utils.DateFormatUtils;
import cn.yunrui.mqttclient.ebikesrv.common.utils.MapUtils;
import cn.yunrui.mqttclient.ebikesrv.common.utils.StringUtils;
import org.eclipse.paho.client.mqttv3.MqttClient;
import org.eclipse.paho.client.mqttv3.MqttMessage;

import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * 结束充电命令
 *
 * @author Nick Zhang
 * @date 2019-03-26
 *
 */
public class StopHandlerImpl extends MqttMessageHandlerImpl implements MqttMessageHandler {

    public StopHandlerImpl(MqttClient mqttClient, MqttMsgLogDao mqttMsgLogDao) {
        super("stop", mqttClient, mqttMsgLogDao, false);
    }

    @Override
    public MsgHandleResponse handleUpMessage(ChargeDeviceProps chargeDeviceProps, MqttMessage mqttMessage) {
        // 设备到服务器[MQTT_YJM2M_201710]：<deviceId>#stop#<plugId>#<remain>
        // 设备到服务器[MQTT_YJM2M_201712] / [MQTT_YJM2M_201801] / [MQTT_YJM2M_201804]：<deviceId>#stop#<plugId>#<remain>#<stopReason>#<stopPower>#<remainTime>
        // 设备到服务器[MQTT_EBIKE_201804]：<deviceId>#stop#<plugId>#<stopReason>#<stopPower>#<remain>#<remainTime>
        // <remain>表示剩余电量，单位为1/1000度；
        // <stopReason> 0:电量用完停止充电 1:浮充时间完成停止充电 2:插头拔掉停止充电 3:功率超限停止充电 4：计时时间到停止充电
        // <stopPower>停止供电时插座的功率
        // <remainTime>表示剩余充电时间，单位为分钟
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> start handleUpMessage >>>>>> ");
        }

        MsgHandleResponse res = new MsgHandleResponse();
        res.setStatus(-1);
        res.setMsg("处理上行（设备到服务器）消息失败 <<< 结束充电命令");

        if(chargeDeviceProps != null && StringUtils.isNotBlank(chargeDeviceProps.getDeviceId())) {
            boolean continueCharge = false;
            String deviceId = chargeDeviceProps.getDeviceId();
            String rcvdMsg = new String(mqttMessage.getPayload());
            String[] rcvdMsgArray = rcvdMsg.split("#");
            String plugId = null;
            String remain = null;
            String stopReason = null;
            String stopPower = null;
            String remainTime = null;
            Date currTime = new Date();
            if(rcvdMsgArray.length >= 3) {
                plugId = rcvdMsgArray[2];
            }
            if(rcvdMsgArray.length >= 4) {
                remain = rcvdMsgArray[chargeDeviceProps.getChargeDeviceProtocol().equals(ChargeDeviceProtocolEnum.MQTT_EBIKE_201804) || chargeDeviceProps.getChargeDeviceProtocol().equals(ChargeDeviceProtocolEnum.MQTT_EBIKE_201805) ? 5 : 3];
            }
            if(rcvdMsgArray.length >= 5) {
                stopReason = rcvdMsgArray[chargeDeviceProps.getChargeDeviceProtocol().equals(ChargeDeviceProtocolEnum.MQTT_EBIKE_201804) || chargeDeviceProps.getChargeDeviceProtocol().equals(ChargeDeviceProtocolEnum.MQTT_EBIKE_201805) ? 3 : 4];
            }
            if(rcvdMsgArray.length >= 6) {
                stopPower = rcvdMsgArray[chargeDeviceProps.getChargeDeviceProtocol().equals(ChargeDeviceProtocolEnum.MQTT_EBIKE_201804) || chargeDeviceProps.getChargeDeviceProtocol().equals(ChargeDeviceProtocolEnum.MQTT_EBIKE_201805) ? 4 : 5];
            }
            if(rcvdMsgArray.length >= 7) {
                remainTime = rcvdMsgArray[6];
            }

            Map<String, Object> chargeRecordMap = chargeRecordCache.getChargeRecordMap(deviceId, plugId);
            if(MapUtils.isNotEmpty(chargeRecordMap)) {      // 该插座有当前充电记录
                Map<String, Object> chargeorderLoadMap = new HashMap<>();
                // 当前充电记录更新
                Double valRemain = 0.0D;
                Double valStopPower = 0.0D;
                Double valRemainTime = 0.0D;
                if(StringUtils.isNotBlank(remain)) {
                    try {
                        valRemain = Double.parseDouble(remain);
                    }
                    catch(NumberFormatException _nfe) {
                        logger.error(_nfe.getMessage(), _nfe.fillInStackTrace());
                    }
                }
                if(StringUtils.isNotBlank(stopPower)) {
                    try {
                        valStopPower = Double.parseDouble(stopPower);
                    }
                    catch(NumberFormatException _nfe) {
                        logger.error(_nfe.getMessage(), _nfe.fillInStackTrace());
                    }
                }
                if(StringUtils.isNotBlank(remainTime)) {
                    try {
                        valRemainTime = Double.parseDouble(remainTime);
                    }
                    catch(NumberFormatException _nfe) {
                        logger.error(_nfe.getMessage(), _nfe.fillInStackTrace());
                    }
                }
                RefreshChargeRecordResult result = refreshChargeRecord(deviceId, plugId, valRemain, valStopPower, valRemainTime, currTime);
                chargeorderLoadMap.put("CHGORDERNO", result.getChargeRecordValueMap().get("chargeRecordId"));
                chargeorderLoadMap.put("LOADTIME", DateFormatUtils.format(currTime, "yyyyMMddHHmmss"));
                chargeorderLoadMap.put("LOADVALUE", valStopPower / 1000.0D);                                    // 单位转换：W >> kW
                chargeRecordCache.updateChargeRecord(deviceId, plugId, result.getChargeRecordValueMap());
                Integer chargeOpenFlag = (Integer) result.getChargeRecordValueMap().get("chargeOpenFlag");
                Date chargeOpenTime = (Date) result.getChargeRecordValueMap().get("chargeOpenTime");
                if(!(chargeDeviceProps.getChargeDeviceProtocol().equals(ChargeDeviceProtocolEnum.MQTT_EBIKE_201804) || chargeDeviceProps.getChargeDeviceProtocol().equals(ChargeDeviceProtocolEnum.MQTT_EBIKE_201805)) && !chargeDeviceProps.getChargeDeviceProtocol().equals(ChargeDeviceProtocolEnum.MQTT_YJM2M_201804)
                        && StringUtils.equals(stopReason, FinishReason.REASON_PLUGPULLOUT) && ((valStopPower.compareTo(0.0D) > 0 && valRemain.compareTo(0.0D) > 0 && valRemainTime.compareTo(0.0D) > 0) || (chargeOpenFlag == ChargeOpenFlag.FLAG_SUCCESS && currTime.getTime() - chargeOpenTime.getTime() < 120000L))) {
                    // 当结束原因是插头拔掉，结束功率不为0时或者充电持续时间不超过3分钟，继续下发充电命令（设备问题引发结束充电 0322）
                    // 有设备充了一段时间（有些半个小时，有些2个多小时，不一定）后就因为插头拔掉原因结束充电了，但实际上并没有拔掉插头，再次发起充电还是可以继续充电
                    // 根据观察，当结束功率大于0时，再次发起充电可以继续充电。
                    String msg2Device = "start" + "#" + plugId + "#" + Math.round(valRemain) + "#" + Math.round(valRemainTime);
                    response2device(deviceId, msg2Device, "start");
                    continueCharge = true;
                }
                if(!continueCharge) {
                    // 当结束原因不是插头拔掉，结束充电记录
                    // 当结束原因是插头拔掉，结束功率为0时且充电持续时间超过3分钟，结束充电记录
                    Map<String, Object> finishValueMap = new HashMap<>();
                    finishValueMap.put("finishReason", stopReason);
                    if(StringUtils.isNotBlank(stopPower)) {
                        try {
                            finishValueMap.put("finishPower", Double.valueOf(stopPower) / 1000.0D);
                        }
                        catch(NumberFormatException _nfe) {
                            logger.error(_nfe.getMessage(), _nfe.fillInStackTrace());
                        }
                    }
                    chargeRecordCache.finishChargeRecord(deviceId, plugId, finishValueMap);
                    chargeOrderEventPublisher.publishEvent("finish", chargeRecordCache.getHistroyChargeRecord((String) chargeRecordMap.get("chargeRecordId")));
                }
                chargeRecordCache.saveChargeorderLoadMap(chargeorderLoadMap);
            }
            else {
                // 该插座无当前充电记录，则判断是否在装接调试
                if(fitconnDebuggingCache.checkIsDebugging(deviceId, plugId)) {
                    // 结束装接调试日志记录
                    Map<String, Object> valueMap = new HashMap<>();
                    valueMap.put("debugFinishTime", new Date());
                    fitconnDebuggingCache.finishFitconnDebuggingLog(deviceId, plugId, valueMap);
                }
            }
            logger.debug(" chargedevice opstatus    : " + chargeDeviceProps.getOpStatus());
            logger.debug(" continueCharge           : " + continueCharge);
            // 充电设备状态更新
            if(StringUtils.equals("1", chargeDeviceProps.getOpStatus())) {
                if(!continueCharge) {
                    logger.debug(" deviceId : " + deviceId);
                    logger.debug(" plugId   : " + plugId);
                    chargeDeviceCache.updatePlugStatus(deviceId, plugId, "0");          // 当设备状态为在线时且不继续充电，插座状态置为 0 - 空闲
                }
            }
            else {
                chargeDeviceCache.updatePlugStatus(deviceId, plugId, "9");              // 当设备状态为离线时，插座状态置为 9 - 不可用（设备离线）
            }

            res.setStatus(0);
            res.setMsg("处理上行（设备到服务器）消息成功 <<< 结束充电命令");

            if(needRespond && res.getStatus() == 0) {
                res = handleDownMessage(chargeDeviceProps);
            }
        }

        if(logger.isDebugEnabled()) {
            logger.debug(" <<<<<<< end handleUpMessage <<<<<<< ");
        }
        return res;
    }

    @Override
    public MsgHandleResponse handleDownMessage(ChargeDeviceProps chargeDeviceProps) {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> start handleDownMessage >>>>>> ");
        }
        MsgHandleResponse res = new MsgHandleResponse();
        res.setStatus(0);
        res.setMsg("处理下行（服务器到设备）消息成功 >>> 结束充电命令");
        if(logger.isDebugEnabled()) {
            logger.debug(" <<<<<<< end handleDownMessage <<<<<<< ");
        }
        return res;
    }

}
