package cn.yunrui.mqttclient.ebikesrv.mqttclient.service.impl;

import cn.yunrui.mqttclient.ebikesrv.mqttclient.cache.ChargeDeviceCache;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.cache.ChargeRecordCache;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.cache.FitconnDebuggingCache;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.cache.OpsWorkOrderCache;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.constant.ChargeOpenFlag;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.dao.MqttMsgLogDao;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.entity.ChargeDeviceProps;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.entity.ChargePlugProps;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.entity.MsgHandleResponse;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.handler.MqttMessageHandler;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.handler.MqttMessageHandlerFactory;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.service.EBikeChargeProfile;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.service.EBikeChargeService;
import cn.yunrui.mqttclient.ebikesrv.common.utils.JsonConvertUtils;
import cn.yunrui.mqttclient.ebikesrv.common.utils.MapUtils;
import cn.yunrui.mqttclient.ebikesrv.common.utils.StringUtils;
import cn.yunrui.mqttclient.ebikesrv.syncevent.publisher.ChargeOrderEventPublisher;
import com.google.common.base.Splitter;
import com.google.common.collect.Iterables;
import org.eclipse.paho.client.mqttv3.*;
import org.eclipse.paho.client.mqttv3.persist.MemoryPersistence;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Resource;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * 电动自行车充电主服务（MQTT消息处理）
 */
public class EBikeChargeServiceImpl implements EBikeChargeService {

    private final static int QOS = 1;
    private final static boolean RETAINED = false;
    private final static long WAITTIMEOUT = 3000L;

    private final Logger logger = LoggerFactory.getLogger(getClass());

    private final ThreadPoolExecutor pool = new ThreadPoolExecutor(30, 300, 90, TimeUnit.SECONDS, new LinkedBlockingDeque<>());

    private final EBikeChargeProfile ebikeChargeProfile;

    private final MqttClient mqttClient;

    @Resource(name = "mqttMsgLogDao")
    private MqttMsgLogDao mqttMsgLogDao;

    @Resource(name = "chargeDeviceCache")
    private ChargeDeviceCache chargeDeviceCache;

    @Resource(name = "chargeRecordCache")
    private ChargeRecordCache chargeRecordCache;

    @Resource(name = "fitconnDebuggingCache")
    private FitconnDebuggingCache fitconnDebuggingCache;

    @Resource(name = "opsWorkOrderCache")
    private OpsWorkOrderCache opsWorkOrderCache;

    @Resource(name = "chargeOrderEventPublisher")
    private ChargeOrderEventPublisher chargeOrderEventPublisher;

    public EBikeChargeServiceImpl(EBikeChargeProfile ebikeChargeProfile) throws MqttException {
        this.ebikeChargeProfile = ebikeChargeProfile;
        String serverURI = "tcp://" + ebikeChargeProfile.getBrokerHostIp() + ":" + ebikeChargeProfile.getBrokerHostPort();
        String clientId = ebikeChargeProfile.getServerClientId();
        this.mqttClient = new MqttClient(serverURI, clientId, new MemoryPersistence());
        connectBroker();
    }

    @Override
    public boolean publish2device(String deviceId, byte[] payload, String type) {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> start publish2device >>>>>> ");
            logger.debug(" [ deviceId : " + deviceId + ", message : " + new String(payload) + ", type : " + type + " ] ");
        }

        try {
            String end = "";
            if(chargeDeviceCache.get(deviceId) != null && StringUtils.equals("hzchaoxiang", chargeDeviceCache.get(deviceId).getFactory())) {
                // 超翔设备结尾需加#
                end = "#";
            }
            else if(chargeDeviceCache.get(deviceId) != null && StringUtils.equals("silan", chargeDeviceCache.get(deviceId).getFactory())) {
                // 士兰微设备结尾需加#<time_stamp>
                end = "#" + String.format("%010d", new DateTime(DateTimeZone.forID("Etc/GMT-8")).getMillis() / 1000);
            }
            byte[] payloadExtend = new byte[payload.length + end.getBytes().length];
            System.arraycopy(payload, 0, payloadExtend, 0, payload.length);
            if(end.getBytes().length > 0) {
                System.arraycopy(end.getBytes(), 0, payloadExtend, payload.length, end.getBytes().length);
            }
            final String messageFinal = new String(payloadExtend);
            final MqttTopic mqttTopicDevice = mqttClient.getTopic((chargeDeviceCache.get(deviceId) != null && StringUtils.equals("hzchaoxiang", chargeDeviceCache.get(deviceId).getFactory())) ? "$client/" + deviceId : deviceId);
            final MqttMessage mm = new MqttMessage(payloadExtend);
            mm.setQos(QOS);
            mm.setRetained(RETAINED);
            MqttDeliveryToken token = mqttTopicDevice.publish(mm);
            token.waitForCompletion(WAITTIMEOUT);
            if(token.isComplete()) {
                if(logger.isDebugEnabled()) {
                    logger.debug(" >>>>>> mm         : " + new String(mm.getPayload()) + " >>>>>> ");
                    logger.debug(" >>>>>> completion : " + token.isComplete() + " >>>>>> ");
                }
                Thread thread = new Thread(() -> {
                    // 插入MQTT消息日志
                    mqttMsgLogDao.insertMqttMsgLog(mqttTopicDevice.getName(), mm, deviceId, type, "down", "request");
                    if(StringUtils.equals("start", type) || StringUtils.equals("cardCharge", type) || StringUtils.equals("cardCharge2", type) || StringUtils.equals("coinCharge", type)) {
                        String plugId = null;
                        if(StringUtils.isNotBlank(messageFinal)) {
                            String[] msgArray = messageFinal.split("#");
                            if(msgArray.length >= 2) {
                                plugId = msgArray[1];
                            }
                        }
                        if(StringUtils.isNotBlank(deviceId) && StringUtils.isNotBlank(plugId)) {
                            // 更新充电插座运行状态
                            chargeDeviceCache.updatePlugStatus(deviceId, plugId, "1");
                        }
                        // 重新加载缓存
                        ChargeDeviceProps cdp = chargeDeviceCache.reload(deviceId);
                        if(logger.isDebugEnabled()) {
                            logger.debug(" >>> 重新加载缓存 >>> ");
                            logger.debug(" ChargeDeviceProps : " + (cdp != null ? cdp.toString() : null));
                        }
                    }
                });
                pool.execute(thread);
            }
            if(logger.isDebugEnabled()) {
                logger.debug(" <<<<<<< end publish2device <<<<<<< ");
            }
            return token.isComplete();
        }
        catch(MqttException _me) {
            logger.error(_me.getMessage(), _me.fillInStackTrace());
            if(logger.isDebugEnabled()) {
                logger.debug(" <<<<<<< end publish2device <<<<<<< ");
            }
            return false;
        }
    }

    @Override
    public boolean publish2device(final String deviceId, String message, final String type) {
        return publish2device(deviceId, message.getBytes(), type);
    }

    @Override
    public void finishChargeRecord(String deviceId, String plugId, int openFlag, int reason) {
        // 当前充电记录更新
        Map<String, Object> valueMap = new HashMap<>();
        if(openFlag == ChargeOpenFlag.FLAG_WAITING) {
            valueMap.put("chargeOpenFlag", (reason >= 0 ? ChargeOpenFlag.FLAG_SUCCESS : reason));
            valueMap.put("chargeOpenTime", new Date());
            chargeRecordCache.updateChargeRecord(deviceId, plugId, valueMap);
        }
        // 结束充电记录
        // 充电设备状态更新
        Map<String, Object> finishValueMap = new HashMap<>();
        finishValueMap.put("finishReason", String.valueOf(reason));
        chargeRecordCache.finishChargeRecord(deviceId, plugId, finishValueMap);
        ChargeDeviceProps cdp = chargeDeviceCache.get(deviceId);
        if(StringUtils.equals("1", cdp.getOpStatus())) {
            chargeDeviceCache.updatePlugStatus(deviceId, plugId, "0");            // 当设备状态为在线时，插座状态置为 0 - 空闲
        }
        else {
            chargeDeviceCache.updatePlugStatus(deviceId, plugId, "9");            // 当设备状态为离线时，插座状态置为 9 - 不可用（设备离线）
        }
    }

    @Override
    public void finishFitconnDebugging(String deviceId, String plugId) {
        // TODO: 2018-02-08 finishFitconnDebugging
    }

    /**
     * 连接MQTT Broker
     */
    private void connectBroker() {
        MqttConnectOptions options = new MqttConnectOptions();
        options.setAutomaticReconnect(true);
        options.setCleanSession(false);
        // 设置用户名密码
        options.setUserName(ebikeChargeProfile.getBrokerUsername());
        options.setPassword(ebikeChargeProfile.getBrokerPassword().toCharArray());
        // 设置超时时间
        options.setConnectionTimeout(10);
        // 设置会话心跳时间
        options.setKeepAliveInterval(30);
        try {
            if(logger.isDebugEnabled()) {
                logger.debug(" >>>>>> connectBroker >>>>>> ");
            }
            // 连接
            mqttClient.connect(options);
            // 订阅消息
            String[] subTopics = Iterables.toArray(Splitter.on(',').trimResults().omitEmptyStrings().split(ebikeChargeProfile.getServerSubTopic()), String.class);
            int[] qoss = new int[subTopics.length];
            Arrays.fill(qoss, 1);
            if(logger.isDebugEnabled()) {
                logger.debug(" subTopics    : " + Arrays.toString(subTopics));
                logger.debug(" qoss         : " + Arrays.toString(qoss));
            }
            mqttClient.subscribe(subTopics, qoss);
            // 设置回调
            mqttClient.setCallback(new PushCallback());
            mqttClient.setManualAcks(false);
            mqttClient.setTimeToWait(WAITTIMEOUT);
        }
        catch(MqttException _me) {
            logger.error(_me.getMessage(), _me.fillInStackTrace());
        }
    }

    /**
     *
     */
    private class PushCallback implements MqttCallbackExtended {

        @Override
        public void connectComplete(boolean reconnect, String serverURI) {
            if(logger.isDebugEnabled()) {
                logger.debug(" >>>>>> [main service] start connectComplete >>>>>> ");
                logger.debug(" reconnect  : " + reconnect);
                logger.debug(" serverURI  : " + serverURI);
                logger.debug(" <<<<<<< [main service] end connectComplete <<<<<<< ");
            }
        }

        @Override
        public void connectionLost(Throwable throwable) {
            logger.warn(" >>>>>> [main service] start connectionLost >>>>>> ");
            logger.error(" Connection lost with cause \"" + throwable.getMessage() + "\", Reason code : " + ((MqttException) throwable).getReasonCode() + "\" Cause : \"" + throwable.getCause() +  "\"");
            logger.error(" error ", throwable);
            logger.warn(" <<<<<<< [main service] end connectionLost <<<<<<< ");
        }

        @Override
        public void messageArrived(final String topic, final MqttMessage mqttMessage) throws Exception {
            // subscribe后得到的消息会执行到这里面
            if(logger.isDebugEnabled()) {
                logger.debug(" >>>>>> [main service] start messageArrived >>>>>> ");
                logger.debug(" topic            : " + topic);
                if(mqttMessage != null && mqttMessage.getPayload() != null) {
                    logger.debug(" messageId        : " + mqttMessage.getId());
                    logger.debug(" messageContent   : " + new String(mqttMessage.getPayload()));
                    logger.debug(" qos              : " + mqttMessage.getQos());
                    logger.debug(" retained         : " + mqttMessage.isRetained());
                    logger.debug(" duplicate        : " + mqttMessage.isDuplicate());
                }
                else {
                    logger.debug(" mqttMessage is null ");
                }
            }

            if(mqttMessage != null && mqttMessage.getPayload() != null) {
                Thread thread = new Thread(() -> {
                    String receivedMsg = new String(mqttMessage.getPayload());
                    if(StringUtils.isNotBlank(receivedMsg) && receivedMsg.length() > 0) {
                        String[] rcvdMsgArray = receivedMsg.trim().split("#");
                        if(rcvdMsgArray.length >= 2) {
                            String deviceId = rcvdMsgArray[0].trim();
                            String opType = rcvdMsgArray[1].trim();
                            if(logger.isDebugEnabled()) {
                                logger.debug(" deviceId : " + deviceId + " , opType : " + opType);
                            }
                            // 插入MQTT消息日志
                            mqttMsgLogDao.insertMqttMsgLog(topic, mqttMessage, deviceId, opType, "up", "request");
                            //
                            ChargeDeviceProps cdp = chargeDeviceCache.get(deviceId);
                            if(cdp != null && MapUtils.isNotEmpty(cdp.getPlugPropsMap())) {
                                // 更新运行状态（包括充电设备及充电插座）
                                String[] plugStatus = null;
                                if(StringUtils.equals("0", cdp.getOpStatus())) {   // 0 - 不在线
                                    plugStatus = new String[cdp.getPlugPropsMap().size()];
                                    for(int i = 0; i < cdp.getPlugPropsMap().size(); i++) {
                                        ChargePlugProps cpp = cdp.getPlugPropsMap().get(i + 1);
                                        if(cpp != null && StringUtils.equals("8", cpp.getOpStatus())) {
                                            // 插座状态：8 - 正在检修
                                            plugStatus[i] = "8";                    // 8 - 正在检修
                                        }
                                        else {
                                            if(cpp != null && StringUtils.isNotBlank(cpp.getStatus())) {
                                                if(StringUtils.equals("F", cpp.getStatus())) {
                                                    plugStatus[i] = "0";            // 0 - 空闲
                                                }
                                                else if(StringUtils.equals("U", cpp.getStatus())) {
                                                    plugStatus[i] = "1";            // 1 - 正在使用
                                                }
                                                else if(StringUtils.equals("R", cpp.getStatus())) {
                                                    plugStatus[i] = "2";            // 2 - 已被预订
                                                }
                                                else {
                                                    plugStatus[i] = "0";            // 0 - 空闲
                                                }
                                            }
                                            else {
                                                plugStatus[i] = "0";                // 0 - 空闲
                                            }
                                        }
                                    }
                                }
                                chargeDeviceCache.updateOpStatus(deviceId, "1", plugStatus);
                                // 更新设备资产状态
                                if(StringUtils.equals("02", cdp.getStatus())) {
                                    chargeDeviceCache.updateDeviceAssertStatus(deviceId, "03");
                                }
                            }

                            MqttMessageHandler handler = MqttMessageHandlerFactory.getInstance(opType, mqttClient, mqttMsgLogDao);
                            if(handler != null) {
                                handler.setChargeDeviceCache(chargeDeviceCache);
                                handler.setChargeRecordCache(chargeRecordCache);
                                handler.setFitconnDebuggingCache(fitconnDebuggingCache);
                                handler.setOpsWorkOrderCache(opsWorkOrderCache);
                                handler.setChargeOrderEventPublisher(chargeOrderEventPublisher);
                                MsgHandleResponse res = handler.handleUpMessage(cdp, mqttMessage);
                                if(logger.isDebugEnabled()) {
                                    logger.debug("ChargeDeviceProps : " + JsonConvertUtils.convertToString(cdp));
                                    logger.debug("MsgHandleResponse : " + JsonConvertUtils.convertToString(res));
                                }
                            }
                            else {
                                logger.error(" MqttMessageHandlerFactory getInstance is null. [ opType : " + opType + " ] ");
                                logger.error(" receivedMsg : " + receivedMsg);
                            }
                        }
                    }
                });
                pool.execute(thread);
            }

            if(logger.isDebugEnabled()) {
                logger.debug(" <<<<<<< [main service] end messageArrived <<<<<<< ");
            }
        }

        @Override
        public void deliveryComplete(IMqttDeliveryToken token) {
            if(logger.isDebugEnabled()) {
                logger.debug(" >>>>>> [main service] start deliveryComplete >>>>>> ");
                logger.debug(" deliveryComplete : " + token.isComplete());
                logger.debug(" <<<<<<< [main service] end deliveryComplete <<<<<<< ");
            }
        }

    }

}
