package cn.yunrui.mqttclient.ebikesrv.mqttclient.service.impl;

import cn.yunrui.mqttclient.ebikesrv.mqttclient.dao.MqttMsgLogDao;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.dao.TestingDao;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.entity.ChargeDeviceProtocolEnum;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.entity.TestingChargeDevice;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.entity.TestingChargePlug;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.service.EBikeChargeProfile;
import cn.yunrui.mqttclient.ebikesrv.mqttclient.service.EBikeTestingService;
import cn.yunrui.mqttclient.ebikesrv.common.utils.JsonConvertUtils;
import cn.yunrui.mqttclient.ebikesrv.common.utils.StringUtils;
import com.google.common.base.Splitter;
import com.google.common.collect.Iterables;
import org.eclipse.paho.client.mqttv3.*;
import org.eclipse.paho.client.mqttv3.persist.MemoryPersistence;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Resource;
import java.util.Arrays;
import java.util.Date;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * 电动自行车充电检测服务（MQTT消息处理）
 */
public class EBikeTestingServiceImpl implements EBikeTestingService {

    private final static int QOS = 1;
    private final static boolean RETAINED = false;
    private final static long WAITTIMEOUT = 3000L;

    private final Logger logger = LoggerFactory.getLogger(getClass());
    private final ThreadPoolExecutor pool = new ThreadPoolExecutor(10, 30, 60, TimeUnit.SECONDS, new LinkedBlockingDeque<>());

    private final EBikeChargeProfile ebikeTestingProfile;

    private final MqttClient mqttClient;

    @Resource(name = "testingDao")
    private TestingDao testingDao;

    @Resource(name = "mqttMsgLogDao")
    private MqttMsgLogDao mqttMsgLogDao;

    public EBikeTestingServiceImpl(EBikeChargeProfile ebikeTestingProfile) throws MqttException {
        this.ebikeTestingProfile = ebikeTestingProfile;
        String serverURI = "tcp://" + ebikeTestingProfile.getBrokerHostIp() + ":" + ebikeTestingProfile.getBrokerHostPort();
        String clientId = ebikeTestingProfile.getServerClientId();
        this.mqttClient = new MqttClient(serverURI, clientId, new MemoryPersistence());
        connectBroker();
    }


    @Override
    public boolean publish2device(String deviceId, byte[] payload, String type) {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>> start publish2device >>>>>> ");
            logger.debug(" [ deviceId : " + deviceId + ", message : " + new String(payload) + ", type : " + type + " ] ");
        }

        try {
            TestingChargeDevice tcd = testingDao.getTestingChargeDeviceByDeviceId(deviceId);
            String end = "";
            if(tcd != null && StringUtils.equals("hzchaoxiang", tcd.getFactory())) {
                // 超翔设备结尾需加#
                end = "#";
            }
            else if(tcd != null && StringUtils.equals("silan", tcd.getFactory())) {
                // 士兰微设备结尾需加#<time_stamp>
                end = "#" + String.format("%010d", new DateTime(DateTimeZone.forID("Etc/GMT-8")).getMillis() / 1000);
            }
            byte[] payloadExtend = new byte[payload.length + end.getBytes().length];
            System.arraycopy(payload, 0, payloadExtend, 0, payload.length);
            if(end.getBytes().length > 0) {
                System.arraycopy(end.getBytes(), 0, payloadExtend, payload.length, end.getBytes().length);
            }
            //final String messageFinal = new String(payloadExtend);
            final MqttTopic mqttTopicDevice = mqttClient.getTopic((tcd != null && StringUtils.equals("hzchaoxiang", tcd.getFactory())) ? "$client/" + deviceId : deviceId);
            final MqttMessage mm = new MqttMessage(payloadExtend);
            mm.setQos(QOS);
            mm.setRetained(RETAINED);
            MqttDeliveryToken token = mqttTopicDevice.publish(mm);
            token.waitForCompletion(WAITTIMEOUT);
            if(token.isComplete()) {
                Thread thread = new Thread(() -> {
                    // 插入MQTT消息日志
                    mqttMsgLogDao.insertTestingMqttMsgLog(mqttTopicDevice.getName(), mm, deviceId, type, "down", "request");
                });
                pool.execute(thread);
            }
            if(logger.isDebugEnabled()) {
                logger.debug(" <<<<<<< end publish2device <<<<<<< ");
            }
            return token.isComplete();
        }
        catch(MqttException _me) {
            logger.error(_me.getMessage(), _me.fillInStackTrace());
            if(logger.isDebugEnabled()) {
                logger.debug(" <<<<<<< end publish2device <<<<<<< ");
            }
            return false;
        }
    }

    @Override
    public boolean publish2device(String deviceId, String message, String type) {
        return publish2device(deviceId, message.getBytes(), type);
    }

    /**
     * 连接MQTT Broker
     */
    private void connectBroker() {
        MqttConnectOptions options = new MqttConnectOptions();
        options.setAutomaticReconnect(true);
        options.setCleanSession(false);
        // 设置用户名密码
        options.setUserName(ebikeTestingProfile.getBrokerUsername());
        options.setPassword(ebikeTestingProfile.getBrokerPassword().toCharArray());
        // 设置超时时间
        options.setConnectionTimeout(10);
        // 设置会话心跳时间
        options.setKeepAliveInterval(30);
        try {
            if(logger.isDebugEnabled()) {
                logger.debug(" >>>>>> connectBroker >>>>>> ");
            }
            // 连接
            mqttClient.connect(options);
            // 订阅消息
            String[] subTopics = Iterables.toArray(Splitter.on(',').trimResults().omitEmptyStrings().split(ebikeTestingProfile.getServerSubTopic()), String.class);
            int[] qoss = new int[subTopics.length];
            Arrays.fill(qoss, 1);
            if(logger.isDebugEnabled()) {
                logger.debug(" subTopics    : " + Arrays.toString(subTopics));
                logger.debug(" qoss         : " + Arrays.toString(qoss));
            }
            mqttClient.subscribe(subTopics, qoss);
            // 设置回调
            mqttClient.setCallback(new EBikeTestingServiceImpl.PushCallback());
            mqttClient.setManualAcks(false);
            mqttClient.setTimeToWait(WAITTIMEOUT);
        }
        catch(MqttException _me) {
            logger.error(_me.getMessage(), _me.fillInStackTrace());
        }
    }

    /**
     *
     */
    private class PushCallback implements MqttCallbackExtended {

        @Override
        public void connectComplete(boolean reconnect, String serverURI) {
            if(logger.isDebugEnabled()) {
                logger.debug(" >>>>>> [testing service] start connectComplete >>>>>> ");
                logger.debug(" reconnect  : " + reconnect);
                logger.debug(" serverURI  : " + serverURI);
                logger.debug(" <<<<<<< [testing service] end connectComplete <<<<<<< ");
            }
        }

        @Override
        public void connectionLost(Throwable throwable) {
            logger.warn(" >>>>>> [testing service] start connectionLost >>>>>> ");
            logger.error(" Connection lost with cause \"" + throwable.getMessage() + "\", Reason code : " + ((MqttException) throwable).getReasonCode() + "\" Cause : \"" + throwable.getCause() +  "\"");
            logger.error(" error ", throwable);
            logger.warn(" <<<<<<< [testing service] end connectionLost <<<<<<< ");
        }

        @Override
        public void messageArrived(final String topic, final MqttMessage mqttMessage) throws Exception {
            // subscribe后得到的消息会执行到这里面
            if(logger.isDebugEnabled()) {
                logger.debug(" >>>>>> [testing service] start messageArrived >>>>>> ");
                logger.debug(" topic            : " + topic);
                if(mqttMessage != null) {
                    logger.debug(" mqttMessage      : " + new String(mqttMessage.getPayload()));
                }
                else {
                    logger.debug(" mqttMessage is null ");
                }
            }

            if(mqttMessage != null) {
                Thread thread = new Thread(() -> {
                    String strMsg = new String(mqttMessage.getPayload());
                    if(StringUtils.isNotBlank(strMsg) && strMsg.length() > 0) {
                        String[] strMsgArray = strMsg.trim().split("#");
                        if(strMsgArray.length >= 2) {
                            String deviceId = strMsgArray[0].trim();
                            String opType = strMsgArray[1].trim();
                            if(logger.isDebugEnabled()) {
                                logger.debug(" deviceId : " + deviceId + " , opType : " + opType);
                                logger.debug(" msgArray : " + Arrays.toString(strMsgArray));
                                logger.debug(" size     : " + strMsgArray.length);
                            }
                            // 插入MQTT消息日志
                            mqttMsgLogDao.insertTestingMqttMsgLog(topic, mqttMessage, deviceId, opType, "up", "request");
                            //
                            TestingChargeDevice tcd = testingDao.getTestingChargeDeviceByDeviceId(deviceId);
                            if(tcd != null) {
                                // 更新运行状态
                                tcd.setOpStatus("1");       // 在线
                                testingDao.updateTestingChargeDevice(tcd);

                            }

                            if(StringUtils.equals("mqtt", opType)) {
                                // <<< 充电桩设备到主站：<deviceId>#mqtt
                                // >>> 主站到充电桩设备：mqtt#<ip>#<port>#<topic>
                                String msg2Device = "mqtt#ERROR";
                                publish2device(deviceId, msg2Device, opType);
                            }
                            else if(StringUtils.equals("param", opType)) {
                                // <<< 充电桩设备到主站：<deviceId>#param
                                //
                                // >>> 主站到充电桩设备：param#<maxPlugPower>#<maxDevicePower>#<factor>#<remain1>,<time1>#<remain2>,<time2>#<remain3>,<time3>#…
                                if(tcd != null) {
                                    String maxPlugPower = "600";
                                    String maxDevicePower = "2500";
                                    String cardPassword = "000000000000";
                                    String factor = "1000";
                                    String reporttime = "1";
                                    //String remain = "0";
                                    //String time = "0";
                                    StringBuilder msg2Device = new StringBuilder();
                                    msg2Device.append("param");
                                    if(ChargeDeviceProtocolEnum.MQTT_EBIKE_201804.equals(tcd.getChargeDeviceProtocol())) {
                                        msg2Device.append("#").append(maxPlugPower);                // <maxPlugPower>
                                        msg2Device.append("#").append(maxDevicePower);              // <maxDevicePower>
                                        msg2Device.append("#").append(factor);                      // <factor>
                                    }
                                    else if(ChargeDeviceProtocolEnum.MQTT_EBIKE_201805.equals(tcd.getChargeDeviceProtocol())) {
                                        msg2Device.append("#").append(maxPlugPower);                // <maxPlugPower>
                                        msg2Device.append("#").append(maxDevicePower);              // <maxDevicePower>
                                        msg2Device.append("#").append(factor);                      // <factor>
                                        msg2Device.append("#").append(reporttime);                  // <reporttime>
                                    }
                                    else {
                                        msg2Device.append("#").append(maxPlugPower);                // <maxPlugPower>
                                        msg2Device.append("#").append(maxDevicePower);              // <maxDevicePower>
                                        msg2Device.append("#").append(cardPassword);                // <cardPassword>
                                        msg2Device.append("#").append(factor);                      // <factor>
                                    }
                                    String r = "0";
                                    String t = "0";
                                    for(int i = 0; i < tcd.getPlugCount(); i++) {
                                        msg2Device.append("#").append(r).append(",").append(t);     // <remaini>,<timei>
                                    }
                                    publish2device(deviceId, msg2Device.toString(), opType);
                                }
                            }
                            else if(StringUtils.equals("attachParam", opType)) {
                                // <<< 充电桩设备到主站：<deviceId>#attachParam
                                //
                                // >>> 主站到充电桩设备：attachParam#<floatChargeTime>#<noloadTime>#<unitLimit>#<unitLimitTime>
                                if(tcd != null) {
                                    String floatChargeTime = "120";                     // 浮充充电时间，120分钟
                                    String noloadTime = "180";                          // 空载（功率为0）持续供电时间，3分钟
                                    String unitLimit = "1000";                          // 单元充电量，500Wh/元
                                    String unitLimitTime = "180";                       // 单元充电时间，180分钟/元
                                    StringBuilder msg2Device = new StringBuilder();
                                    msg2Device.append("attachParam");
                                    if(ChargeDeviceProtocolEnum.MQTT_EBIKE_201804.equals(tcd.getChargeDeviceProtocol())) {
                                        if(StringUtils.equals("hzchaoxiang", tcd.getFactory())) {
                                            msg2Device.append("#").append(String.valueOf(Integer.parseInt(floatChargeTime) * 60));          // <floatChargeTime> 转换为秒
                                        }
                                        else {
                                            msg2Device.append("#").append(floatChargeTime);                                                 // <floatChargeTime>
                                        }
                                        msg2Device.append("#").append(noloadTime);                          // <noloadTime>
                                        msg2Device.append("#").append(unitLimit);                           // <unitLimit>
                                        msg2Device.append("#").append(unitLimitTime);                       // <unitLimitTime>
                                    }
                                    else if(ChargeDeviceProtocolEnum.MQTT_EBIKE_201805.equals(tcd.getChargeDeviceProtocol())) {
                                        msg2Device.append("#").append(floatChargeTime);                     // <floatChargeTime>
                                        msg2Device.append("#").append(noloadTime);                          // <noloadTime>
                                        if(StringUtils.equals("gaea", tcd.getFactory())) {
                                            msg2Device.append("#").append(unitLimit);                       // <unitLimit>
                                            msg2Device.append("#").append(unitLimitTime);                   // <unitLimitTime>
                                        }
                                    }
                                    else {
                                        msg2Device.append("#").append(floatChargeTime);                     // <floatChargeTime>
                                    }
                                    publish2device(deviceId, msg2Device.toString(), opType);
                                }
                            }
                            else if(StringUtils.equals("status", opType)) {
                                // <<< 充电桩设备到主站：<deviceId>#status#<remain1>,<time1>,<power1>#<remain2>,<time2>,<power2>#<remain3>,<time3>,<power3>#…
                                //
                                // >>> 主站到充电桩设备：status#<status1>#<status2>#<status3>#....
                                if(tcd != null) {
                                    String rcvdMsg = new String(mqttMessage.getPayload());
                                    String[] rcvdMsgArray = rcvdMsg.split("#");
                                    Integer plugCount = tcd.getPlugCount() != null ? tcd.getPlugCount() : 0;
                                    Double[] remains = new Double[plugCount];
                                    Double[] powers = new Double[plugCount];
                                    Double[] times = new Double[plugCount];
                                    //Date currTime = new Date();
                                    if(ChargeDeviceProtocolEnum.MQTT_EBIKE_201804.equals(tcd.getChargeDeviceProtocol()) || ChargeDeviceProtocolEnum.MQTT_EBIKE_201805.equals(tcd.getChargeDeviceProtocol())) {
                                        if(rcvdMsgArray.length >= plugCount + 2) {              // MQTT_EBIKE_201804 / MQTT_EBIKE_201805
                                            for(int i = 0; i < plugCount; i++) {
                                                String[] plugValArray = rcvdMsgArray[2 + i].split(",");
                                                String strRemain = plugValArray.length >= 1 ? plugValArray[0] : null;
                                                String strTime = plugValArray.length >= 2 ? plugValArray[1] : null;
                                                String strPower = plugValArray.length >= 3 ? plugValArray[2] : null;
                                                remains[i] = StringUtils.isNotBlank(strRemain) ? Double.parseDouble(strRemain) : null;
                                                powers[i] = StringUtils.isNotBlank(strPower) ? Double.parseDouble(strPower) : null;
                                                times[i] = StringUtils.isNotBlank(strTime) ? Double.parseDouble(strTime) : null;
                                            }
                                        }
                                    }
                                    else {
                                        if(rcvdMsgArray.length >= plugCount * 3 + 2) {
                                            for(int i = 0; i < plugCount; i++) {
                                                remains[i] = StringUtils.isNotBlank(rcvdMsgArray[2 + i * 2]) ? Double.parseDouble(rcvdMsgArray[2 + i * 2]) : null;
                                                powers[i] = StringUtils.isNotBlank(rcvdMsgArray[2 + i * 2 + 1]) ? Double.parseDouble(rcvdMsgArray[2 + i * 2 + 1]) : null;
                                                times[i] = StringUtils.isNotBlank(rcvdMsgArray[2 + plugCount * 3 - (plugCount - i)]) ? Double.parseDouble(rcvdMsgArray[2 + plugCount * 3 - (plugCount - i)]) : null;
                                            }
                                        }
                                    }

                                    if(logger.isDebugEnabled()) {
                                        logger.debug(" remains      : " + JsonConvertUtils.convertToString(remains));
                                        logger.debug(" powers       : " + JsonConvertUtils.convertToString(powers));
                                        logger.debug(" times        : " + JsonConvertUtils.convertToString(times));
                                    }

                                    StringBuilder msg2Device = new StringBuilder();
                                    msg2Device.append(opType);
                                    for(int i = 0; i < plugCount; i++) {
                                        Double remain = remains[i];
                                        //Double power = powers[i];
                                        Double time = times[i];
                                        String plugStatus = "F";
                                        if((remain != null && time != null && remain.compareTo(0.0D) > 0 && time.compareTo(0.0D) > 0) || (remain != null && time == null && remain.compareTo(0.0D) > 0) || (remain == null && time != null && time.compareTo(0.0D) > 0)) {
                                            plugStatus = "U";
                                        }
                                        msg2Device.append("#").append(plugStatus);                  // <statusi>
                                    }
                                    publish2device(deviceId, msg2Device.toString(), opType);
                                }
                            }
                            else if(StringUtils.equals("start", opType)) {
                                // <<< 充电桩设备到主站：<deviceId>#start#<plugId>#<result>
                                //
                                // 处理开启充电回应
                                String rcvdMsg = new String(mqttMessage.getPayload());
                                String[] rcvdMsgArray = rcvdMsg.split("#");
                                String plugId = null;
                                String result = null;                   // 为0表示开启充电成功；为负数表示开启充电失败，-1表示设备总体已经超过最大功率，-2表示被投币充电占用，-99表示等待返回超时
                                //Date currTime = new Date();
                                if(rcvdMsgArray.length >= 4) {
                                    plugId = rcvdMsgArray[2];
                                    result = rcvdMsgArray[3];
                                }

                                if(logger.isDebugEnabled()) {
                                    logger.debug(" plugId   : " + plugId);
                                    logger.debug(" result   : " + result);
                                }
                                if(StringUtils.isNotBlank(plugId)) {
                                    TestingChargePlug tcp = testingDao.getTestingChargePlugByDeviceIdAndPlugSn(deviceId, Integer.parseInt(plugId));
                                    if(StringUtils.equals("0", result)) {
                                        tcp.setTestStatus("02");        // 02 – 开启成功；
                                        tcp.setTestStatusDesc("开启成功");
                                    }
                                    else {
                                        tcp.setTestStatus("03");        // 03 – 开启失败；
                                        tcp.setTestStatusDesc("开启失败");
                                    }
                                    testingDao.updateTestingChargePlug(tcp);
                                }
                            }
                            else if(StringUtils.equals("stop", opType)) {
                                // <<< 充电桩设备到主站：<deviceId>#stop#<plugId>#<stopReason>#<stopPower>#<remain>#<remainTime>
                                //
                                // 处理结束充电回应
                                String rcvdMsg = new String(mqttMessage.getPayload());
                                String[] rcvdMsgArray = rcvdMsg.split("#");
                                String plugId = null;
                                // String remain = null;
                                // String stopReason = null;
                                // String stopPower = null;
                                // String remainTime = null;
                                if(rcvdMsgArray.length >= 3) {
                                    plugId = rcvdMsgArray[2];
                                }
                                if(logger.isDebugEnabled()) {
                                    logger.debug(" plugId   : " + plugId);
                                }
                                if(StringUtils.isNotBlank(plugId)) {
                                    TestingChargePlug tcp = testingDao.getTestingChargePlugByDeviceIdAndPlugSn(deviceId, Integer.parseInt(plugId));
                                    tcp.setTestStatus("07");        // 07 – 充电结束；
                                    tcp.setTestStatusDesc("充电结束");
                                    tcp.setEndTestTime(new Date());
                                    testingDao.updateTestingChargePlug(tcp);
                                }
                            }
                            /*else if(StringUtils.equals("coinCharge", opType)) {
                                // <<< 充电桩设备到主站：<deviceId>#coinCharge#<plugId>#<money>
                                //
                                // >>> 主站到充电桩设备：coinCharge#<plugId>#<limit>#<limitTime>
                            }*/
                        }
                    }
                });
                pool.execute(thread);
            }

            if(logger.isDebugEnabled()) {
                logger.debug(" <<<<<<< [testing service] end messageArrived <<<<<<< ");
            }
        }

        @Override
        public void deliveryComplete(IMqttDeliveryToken token) {
            if(logger.isDebugEnabled()) {
                logger.debug(" >>>>>> [testing service] start deliveryComplete >>>>>> ");
                logger.debug(" deliveryComplete : " + token.isComplete());
                logger.debug(" <<<<<<< [testing service] end deliveryComplete <<<<<<< ");
            }
        }

    }

}
