package com.ebike.openservices.cscmgmt.dao;

import com.ebike.openservices.cscmgmt.vo.CarExcelTemplate;
import com.ebike.openservices.cscmgmt.vo.CarServiceOrgVo;
import com.ebike.openservices.cscmgmt.vo.CustomerCarInfoVo;
import com.ebike.openservices.cscmgmt.vo.CustomerInfoVo;
import org.apache.ibatis.annotations.Mapper;
import org.apache.ibatis.annotations.Param;

import java.util.List;

@Mapper
public interface CarDao {


    /**
     * 获取运营商列表
     *
     * @return List<CarServiceOrgVo>
     */
    List<CarServiceOrgVo> selectOrgs();

    /**
     * 客户列表检索（所有运营商）
     *
     * @param name    姓名
     * @param phone   电话
     * @param type    客户类型
     * @param card    车牌号
     * @param vin     车辆VIN码
     * @param current 当前页
     * @param size    页数据长度
     * @return List<CustomerInfoVo>
     */
    List<CustomerInfoVo> searchCustomerAll(
        @Param("name") String name,
        @Param("phone") String phone,
        @Param("type") String type,
        @Param("card") String card,
        @Param("vin") String vin,
        @Param("current") int current,
        @Param("size") int size);

    /**
     * 客户列表检索（当前运营商）
     *
     * @param orgNo     业务单位编号
     * @param name      姓名
     * @param phone     电话
     * @param type      客户类型
     * @param card      车牌号
     * @param vin       车辆VIN码
     * @param current   当前页
     * @param size      页数据长度
     * @return List<CustomerInfoVo>
     */
    List<CustomerInfoVo> searchCustomer(
        @Param("orgNo") String orgNo,
        @Param("name") String name,
        @Param("phone") String phone,
        @Param("type") String type,
        @Param("card") String card,
        @Param("vin") String vin,
        @Param("current") int current,
        @Param("size") int size);

    /**
     * 条件检索客户记录计数(所有运营商)
     *
     * @param name  姓名
     * @param phone 电话
     * @param type  客户类型
     * @param card  车牌号
     * @param vin   车辆VIN码
     * @return int
     */
    int customerCountAll(@Param("name") String name,
                         @Param("phone") String phone,
                         @Param("type") String type,
                         @Param("card") String card,
                         @Param("vin") String vin);

    /**
     * 条件检索客户记录计数(当前运营商)
     *
     * @param orgNo 业务单位编号
     * @param name  姓名
     * @param phone 电话
     * @param type  客户类型
     * @param card  车牌号
     * @param vin   车辆VIN码
     * @return int
     */
    int customerCount(@Param("orgNo") String orgNo,
                      @Param("name") String name,
                      @Param("phone") String phone,
                      @Param("type") String type,
                      @Param("card") String card,
                      @Param("vin") String vin);

    /**
     * 添加客户信息
     *
     * @param customer 客户信息
     */
    void insertCustomer(CustomerInfoVo customer);

    /**
     * 添加车辆信息
     *
     * @param car 车辆信息
     */
    void insertCar(CustomerCarInfoVo car);

    /**
     * 通过客户编号（账号）查询客户信息
     *
     * @param orgNo      业务系统编号
     * @param customerNo 客户编号
     * @return String
     */
    String selectCustomerByCustomerNo(@Param("orgNo") String orgNo, @Param("customerNo") String customerNo);

    /**
     * 通过客户id查询客户
     *
     * @param orgNo 业务单位标识
     * @param id    客户id
     * @return String
     */
    String selectCustomerById(@Param("orgNo") String orgNo, @Param("id") String id);

    /**
     * 通过用户id查询实时订单（是否有订单？删除客户的前置条件）
     *
     * @param orgNo      业务单位标识
     * @param customerId 客户id
     * @return String
     */
    String selectCurrentOrderByCustomerId(@Param("orgNo") String orgNo, @Param("customerId") String customerId);

    /**
     * 通过用户id查询历史订单（是否有订单？删除客户的前置条件）
     *
     * @param orgNo      业务单位标识
     * @param customerId 客户id
     * @return String
     */
    String selectHistoryOrderByCustomerId(@Param("orgNo") String orgNo, @Param("customerId") String customerId);

    /**
     * 通过客户id删除客户
     *
     * @param orgNo  业务单位标识
     * @param userNo 操作员
     * @param id     记录id
     */
    void logicalDeleteCustomerById(@Param("orgNo") String orgNo, @Param("userNo") String userNo, @Param("id") String id);

    /**
     * 通过业务单位标识删除客户
     *
     * @param orgNo 业务单位标识
     */
    void delCustomerByOrgno(@Param("orgNo") String orgNo);

    /**
     * 更新客户信息
     *
     * @param orgNo   业务单位标识
     * @param id      客户id
     * @param name    姓名
     * @param account 账号
     * @param phone   电话
     * @param type    类型
     */
    void updateCustomerById(@Param("orgNo") String orgNo,
                            @Param("id") String id,
                            @Param("name") String name,
                            @Param("account") String account,
                            @Param("phone") String phone,
                            @Param("type") String type);


    /**
     * 车辆信息列表检索
     *
     * @param orgNo      业务单位标识
     * @param customerId 客户id
     * @param card       车牌号
     * @param vin        车辆vin码
     * @param current    当前页
     * @param size       数据长度
     * @return List<CustomerCarInfoVo>
     */
    List<CustomerCarInfoVo> searchCar(
        @Param("orgNo") String orgNo,
        @Param("customerId") String customerId,
        @Param("card") String card,
        @Param("vin") String vin,
        @Param("current") int current,
        @Param("size") int size);

    /**
     * 查询车辆信息
     *
     * @param orgNo      业务单位编号
     * @param customerId 客户id
     * @return Car
     */
    List<CarExcelTemplate> selectCars(@Param("orgNo") String orgNo, @Param("customerId") String customerId);

    /**
     * 条件检索车辆记录计数
     *
     * @param orgNo      业务单位标识
     * @param customerId 客户id
     * @param card       车牌号
     * @param vin        车辆vin码
     * @return int
     */
    int carCount(@Param("orgNo") String orgNo,
                 @Param("customerId") String customerId,
                 @Param("card") String card,
                 @Param("vin") String vin);

    /**
     * 查询客户是否绑定了车辆信息
     *
     * @param orgNo      业务单位标识
     * @param customerId 客户id
     * @return String
     */
    String selectCarByCustomerId(@Param("orgNo") String orgNo, @Param("customerId") String customerId);

    /**
     * 根据客户id修改车辆信息
     *
     * @param orgNo      业务单位标识
     * @param customerId 用户id
     * @param card       车牌号
     * @param vin        车辆VIN码
     */
    void updateCarByCustomerId(
        @Param("orgNo") String orgNo,
        @Param("customerId") String customerId,
        @Param("card") String card,
        @Param("vin") String vin);

    /**
     * 根据id修改车辆信息
     *
     * @param orgNo 业务单位标识
     * @param id    记录id
     * @param card  车牌号
     * @param vin   车辆VIN码
     */
    void updateCarById(
        @Param("orgNo") String orgNo,
        @Param("id") String id,
        @Param("card") String card,
        @Param("vin") String vin);

    /**
     * 根据客户id删除车辆信息
     *
     * @param orgNo      业务单位标识
     * @param customerId 客户id
     */
    void delCarByCustomerId(@Param("orgNo") String orgNo, @Param("customerId") String customerId);

    /**
     * 根据id删除车辆信息
     *
     * @param orgNo 业务单位标识
     * @param id    记录id
     */
    void delCarById(@Param("orgNo") String orgNo, @Param("id") String id);
}
