package com.ebike.openservices.oprmgmt.service.impl;

import cn.hutool.core.lang.tree.Tree;
import cn.hutool.core.lang.tree.TreeNode;
import cn.hutool.core.lang.tree.TreeUtil;
import com.easesource.data.bean.Pager;
import com.ebike.openservices.base.service.AbstractEbikeBaseService;
import com.ebike.openservices.oprmgmt.dao.OperEvChargingStationBillingInfoLogDao;
import com.ebike.openservices.oprmgmt.dao.PowerStationSafeguardDao;
import com.ebike.openservices.oprmgmt.dto.PageResponse;
import com.ebike.openservices.oprmgmt.entity.po.OperEvChargingStation;
import com.ebike.openservices.oprmgmt.entity.po.OperEvChargingStationBillingInfo;
import com.ebike.openservices.oprmgmt.entity.po.OperEvChargingStationBillingInfoLog;
import com.ebike.openservices.oprmgmt.entity.vo.DetailPowerStationSafeguardChargingGunVO;
import com.ebike.openservices.oprmgmt.entity.vo.DetailPowerStationSafeguardChargingPileVO;
import com.ebike.openservices.oprmgmt.entity.vo.DetailPowerStationSafeguardVO;
import com.ebike.openservices.oprmgmt.entity.vo.ListPowerStationSafeguardVO;
import com.ebike.openservices.oprmgmt.enums.ErrorCodeEnum;
import com.ebike.openservices.oprmgmt.exception.BizException;
import com.ebike.openservices.oprmgmt.request.*;
import com.ebike.openservices.oprmgmt.service.PowerStationSafeguardService;
import com.ebike.openservices.oprmgmt.util.ConvertUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author wuyuxuan
 * @version 1.0.0
 * @since 2021/4/14
 */
@Service
public class PowerStationSafeguardServiceImpl extends AbstractEbikeBaseService implements PowerStationSafeguardService {

    private final PowerStationSafeguardDao powerStationSafeguardDao;

    private final OperEvChargingStationBillingInfoLogDao operEvChargingStationBillingInfoLogDao;

    @Autowired
    public PowerStationSafeguardServiceImpl(PowerStationSafeguardDao powerStationSafeguardDao,
                                            OperEvChargingStationBillingInfoLogDao operEvChargingStationBillingInfoLogDao) {
        this.powerStationSafeguardDao = powerStationSafeguardDao;
        this.operEvChargingStationBillingInfoLogDao = operEvChargingStationBillingInfoLogDao;
    }

    @Override
    public PageResponse<ListPowerStationSafeguardVO> listPowerStationSafeguard(ListPowerStationSafeguardParam param, Pager pager) {
        List<ListPowerStationSafeguardVO> list = powerStationSafeguardDao.listPowerStationSafeguard(param, pager);
        long total = powerStationSafeguardDao.listPowerStationSafeguardTotal(param);
        return PageResponse.of(list, pager.getPageNumber(), pager.getPageSize(), total);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public int insertPowerStation(InsertPowerStationParam param) {
        OperEvChargingStation po = ConvertUtils.convert(param, OperEvChargingStation.class);
        po.setId(idGenerator.nextId())
            .setChargingStationDesc(param.getChargingStationName())
            .setSortSn(1)
            .setGmtCreate(System.currentTimeMillis())
            .setParkingUnitPrice(BigDecimal.ZERO)
            .setVersion(1)
            .setPrimaryOrgNo(param.getOrgNo());
        ConvertUtils.toEmptyDefaultValue(po);
        int count = powerStationSafeguardDao.insertOperEvChargingStation(po);
        if (count > 0) {
            //关联计费模板
            OperEvChargingStationBillingInfo o = new OperEvChargingStationBillingInfo();
            o.setId(idGenerator.nextId())
                .setPrimaryOrgNo(param.getOrgNo())
                .setOrgNo(param.getOrgNo())
                .setChargingStationId(po.getId())
                .setStationBillingType(1)
                .setStationBillingStatus(1)
                .setBillingTmplId(param.getBillingTmplId())
                .setSortSn(1)
                .setGmtCreate(System.currentTimeMillis())
                .setVersion(1);
            ConvertUtils.toEmptyDefaultValue(o);
            count = powerStationSafeguardDao.insertOperEvChargingStationBillingInfo(o);
            //添加操作日志
            OperEvChargingStationBillingInfoLog log = OperEvChargingStationBillingInfoLog.builder()
                .id(idGenerator.nextId())
                .primaryOrgNo(param.getOrgNo())
                .orgNo(param.getOrgNo())
                .stationBillingInfoId(o.getId())
                .logType(10)
                .logValue("")
                .stationBillingStatus(1)
                .creator(param.getCreator())
                .build().defaultField();
            operEvChargingStationBillingInfoLogDao.insert(log);
        }
        return count;
    }

    @Override
    public int updateChargingPile(UpdateChargingPileParam param) {
        return powerStationSafeguardDao.updateChargingPile(param);
    }

    @Override
    public int updateChargingGun(UpdateChargingGunParam param) {
        return powerStationSafeguardDao.updateChargingGun(param);
    }

    @Override
    public DetailPowerStationSafeguardVO getDetailPowerStationSafeguardById(Long id) {
        return powerStationSafeguardDao.getDetailPowerStationSafeguardById(id);
    }

    @Override
    public DetailPowerStationSafeguardChargingPileVO getPowerStationSafeguardChargingPileById(Long id) {
        return powerStationSafeguardDao.getPowerStationSafeguardChargingPileById(id);
    }

    @Override
    public DetailPowerStationSafeguardChargingGunVO getPowerStationSafeguardChargingGunById(Long id) {
        return powerStationSafeguardDao.getPowerStationSafeguardChargingGunById(id);
    }

    @Override
    public void deletePowerStationById(Long id) {
        /*
        * 若电站下存在充电桩则不可删除
        */
        long pileTotal = powerStationSafeguardDao.countChargingPileByStationId(id);
        if (pileTotal > 0) {
            throw new BizException(ErrorCodeEnum.POWER_STATION_CHARGING_PILE_ERROR);
        }
        /*
         * 若电站下存在充电枪不可删除
         */
        long gunTotal = powerStationSafeguardDao.countChargingGunByStationId(id);
        if (gunTotal > 0) {
            throw new BizException(ErrorCodeEnum.POWER_STATION_CHARGING_GUN_ERROR);
        }
        powerStationSafeguardDao.deletePowerStationById(id);
    }

    @Override
    public String getPowerStationNo() {
        String no = powerStationSafeguardDao.getPowerStationNo();
        if (StringUtils.isBlank(no)) {
            return "100001";
        }
        Long noL = Long.valueOf(no);
        return String.valueOf(noL + 1);
    }

    @Override
    public int updatePowerStation(UpdatePowerStationParam param) {
        param.setGmtModified(System.currentTimeMillis());
        return powerStationSafeguardDao.updateById(param);
    }

    @Override
    public Object treePowerStation(String orgNo) {
        //查出所有区域编码
        List<Map<String, String>> adCodeList = powerStationSafeguardDao.listAdCode();
        List<TreeNode<String>> codeNodeList = adCodeList.stream().map(map -> {
            TreeNode<String> node = new TreeNode<String>()
                .setId(map.get("adcode"))
                .setName(map.get("adname"))
                .setParentId(map.get("parentAdcode"));
            return node;
        }).collect(Collectors.toList());
        //查出所有的电站
        List<Map<String, String>> stationList = powerStationSafeguardDao.listTreePowerStation(orgNo);
        List<TreeNode<String>> stationNodeList = stationList.stream().map(map -> {
            TreeNode<String> node = new TreeNode<String>()
                .setId(map.get("id"))
                .setName(map.get("chargingStationName"))
                .setParentId(map.get("adcode"));
            return node;
        }).collect(Collectors.toList());
        codeNodeList.addAll(stationNodeList);
        List<Tree<String>> treeNodes = TreeUtil.build(codeNodeList, "100000");
        return treeNodes;
    }

}
