package com.iesms.openservices.photovoltaic.common;

import com.baomidou.mybatisplus.core.conditions.Wrapper;
import com.baomidou.mybatisplus.core.enums.SqlMethod;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.metadata.TableInfo;
import com.baomidou.mybatisplus.core.metadata.TableInfoHelper;
import com.baomidou.mybatisplus.core.toolkit.*;
import com.baomidou.mybatisplus.extension.toolkit.SqlHelper;
import org.apache.ibatis.binding.MapperMethod;
import org.apache.ibatis.logging.Log;
import org.apache.ibatis.logging.LogFactory;
import org.apache.ibatis.session.SqlSession;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.ResolvableType;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.validation.annotation.Validated;

import java.io.Serializable;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author wuyuxuan
 * @version 1.0.0
 * @since 2021/10/13
 */
@SuppressWarnings("all")
@Validated
public abstract class BaseCrudServiceImpl<M extends BaseMapper<T>, T> implements BaseCrudService<T> {

    private static final Logger LOGGER = LoggerFactory.getLogger(BaseCrudServiceImpl.class);
    private static final int DEFAULT_BATCH_SIZE = 1000;

    protected Log log = LogFactory.getLog(getClass());
    protected Class<T> entityClass = currentModelClass();
    protected Class<M> mapperClass = currentMapperClass();

    @Autowired
    protected M baseCrudMapper;

    public BaseCrudServiceImpl() {
    }

    @Override
    public Class<T> getEntityClass() {
        return entityClass;
    }

    @Override
    public Class<M> getMapperClass() {
        return mapperClass;
    }

    @Override
    public boolean save(T entity) {
        if(LOGGER.isDebugEnabled()) {
            LOGGER.debug(">>>>>> start save >>>>>>");
            LOGGER.debug("entity : " + (entity != null ? entity.toString(): null));
        }
        if(null != entity) {
            return SqlHelper.retBool(baseCrudMapper.insert(entity));
        }
        return false;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean saveBatch(Collection<T> entityList) {
        return saveBatch(entityList, DEFAULT_BATCH_SIZE);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean saveBatch(Collection<T> entityList, int batchSize) {
        if(LOGGER.isDebugEnabled()) {
            LOGGER.debug(">>>>>> start saveBatch >>>>>>");
            LOGGER.debug("entityList    : " + entityList);
            LOGGER.debug("batchSize     : " + batchSize);
        }
        String sqlStatement = getSqlStatement(SqlMethod.INSERT_ONE);
        return executeBatch(entityList, batchSize, (sqlSession, entity) -> sqlSession.insert(sqlStatement, entity));
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean saveOrUpdate(T entity) {
        if(LOGGER.isDebugEnabled()) {
            LOGGER.debug(">>>>>> start saveOrUpdate >>>>>>");
            LOGGER.debug("entity : " + (entity != null ? entity.toString(): null));
        }
        if(null != entity) {
            TableInfo tableInfo = TableInfoHelper.getTableInfo(this.entityClass);
            Assert.notNull(tableInfo, "error: can not execute. because can not find cache of TableInfo for entity!");
            String keyProperty = tableInfo.getKeyProperty();
            Assert.notEmpty(keyProperty, "error: can not execute. because can not find column for id from entity!");
            Object idVal = ReflectionKit.getFieldValue(entity, tableInfo.getKeyProperty());
            return StringUtils.checkValNull(idVal) || Objects.isNull(getById((Serializable) idVal)) ? save(entity) : updateById(entity);
        }
        return false;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean saveOrUpdate(T entity, Wrapper<T> updateWrapper) {
        if(LOGGER.isDebugEnabled()) {
            LOGGER.debug(">>>>>> start saveOrUpdate >>>>>>");
            LOGGER.debug("entity        : " + (entity != null ? entity.toString(): null));
            LOGGER.debug("updateWrapper : " + (updateWrapper != null ? updateWrapper.toString(): null));
        }
        return update(entity, updateWrapper) || saveOrUpdate(entity);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean saveOrUpdateBatch(Collection<T> entityList) {
        return saveOrUpdateBatch(entityList, DEFAULT_BATCH_SIZE);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean saveOrUpdateBatch(Collection<T> entityList, int batchSize) {
        if(LOGGER.isDebugEnabled()) {
            LOGGER.debug(">>>>>> start saveOrUpdateBatch >>>>>>");
            LOGGER.debug("entityList    : " + entityList);
            LOGGER.debug("batchSize     : " + batchSize);
        }
        TableInfo tableInfo = TableInfoHelper.getTableInfo(entityClass);
        Assert.notNull(tableInfo, "error: can not execute. because can not find cache of TableInfo for entity!");
        String keyProperty = tableInfo.getKeyProperty();
        Assert.notEmpty(keyProperty, "error: can not execute. because can not find column for id from entity!");
        return SqlHelper.saveOrUpdateBatch(this.entityClass, this.mapperClass, this.log, entityList, batchSize, (sqlSession, entity) -> {
            Object idVal = ReflectionKit.getFieldValue(entity, keyProperty);
            return StringUtils.checkValNull(idVal)
                    || CollectionUtils.isEmpty(sqlSession.selectList(getSqlStatement(SqlMethod.SELECT_BY_ID), entity));
        }, (sqlSession, entity) -> {
            MapperMethod.ParamMap<T> param = new MapperMethod.ParamMap<>();
            param.put(Constants.ENTITY, entity);
            sqlSession.update(getSqlStatement(SqlMethod.UPDATE_BY_ID), param);
        });
    }

    @Override
    public boolean updateById(T entity) {
        if(LOGGER.isDebugEnabled()) {
            LOGGER.debug(">>>>>> start updateById >>>>>>");
            LOGGER.debug("entity : " + (entity != null ? entity.toString(): null));
        }
        if(null != entity) {
            return SqlHelper.retBool(baseCrudMapper.updateById(entity));
        }
        return false;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean updateBatchById(Collection<T> entityList) {
        return updateBatchById(entityList, DEFAULT_BATCH_SIZE);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean updateBatchById(Collection<T> entityList, int batchSize) {
        if(LOGGER.isDebugEnabled()) {
            LOGGER.debug(">>>>>> start updateBatchById >>>>>>");
            LOGGER.debug("entityList    : " + entityList);
            LOGGER.debug("batchSize     : " + batchSize);
        }
        String sqlStatement = getSqlStatement(SqlMethod.UPDATE_BY_ID);
        return executeBatch(entityList, batchSize, (sqlSession, entity) -> {
            MapperMethod.ParamMap<T> param = new MapperMethod.ParamMap<>();
            param.put(Constants.ENTITY, entity);
            sqlSession.update(sqlStatement, param);
        });
    }

    @Override
    public boolean update(Wrapper<T> updateWrapper) {
        return update(null, updateWrapper);
    }

    @Override
    public boolean update(T entity, Wrapper<T> updateWrapper) {
        if(LOGGER.isDebugEnabled()) {
            LOGGER.debug(">>>>>> start update >>>>>>");
            LOGGER.debug("entity        : " + (entity != null ? entity.toString(): null));
            LOGGER.debug("updateWrapper : " + (updateWrapper != null ? updateWrapper.toString(): null));
        }
        return SqlHelper.retBool(baseCrudMapper.update(entity, updateWrapper));
    }

    @Override
    public boolean removeById(Serializable id) {
        if(LOGGER.isDebugEnabled()) {
            LOGGER.debug(">>>>>> start removeById >>>>>>");
            LOGGER.debug("id    : " + id);
        }
        return SqlHelper.retBool(baseCrudMapper.deleteById(id));
    }

    @Override
    public boolean removeByIds(Collection<? extends Serializable> idList) {
        if(LOGGER.isDebugEnabled()) {
            LOGGER.debug(">>>>>> start removeByIds >>>>>>");
            LOGGER.debug("idList    : " + idList);
        }
        if(CollectionUtils.isEmpty(idList)) {
            return false;
        }
        return SqlHelper.retBool(baseCrudMapper.deleteBatchIds(idList));
    }

    @Override
    public boolean remove(Wrapper<T> queryWrapper) {
        if(LOGGER.isDebugEnabled()) {
            LOGGER.debug(">>>>>> start remove >>>>>>");
            LOGGER.debug("queryWrapper  : " + (queryWrapper != null ? queryWrapper.toString(): null));
        }
        return SqlHelper.retBool(baseCrudMapper.delete(queryWrapper));
    }

    @Override
    public boolean removeByMap(Map<String, Object> columnMap) {
        if(LOGGER.isDebugEnabled()) {
            LOGGER.debug(">>>>>> start removeByMap >>>>>>");
            LOGGER.debug("columnMap     : " + (columnMap != null ? columnMap.toString(): null));
        }
        Assert.notEmpty(columnMap, "error: columnMap must not be empty");
        return SqlHelper.retBool(baseCrudMapper.deleteByMap(columnMap));
    }

    @Override
    public T getById(Serializable id) {
        if(LOGGER.isDebugEnabled()) {
            LOGGER.debug(">>>>>> start getById >>>>>>");
            LOGGER.debug("id    : " + id);
        }
        return baseCrudMapper.selectById(id);
    }

    @Override
    public T getOne(Wrapper<T> queryWrapper) {
        return getOne(queryWrapper, true);
    }

    @Override
    public T getOne(Wrapper<T> queryWrapper, boolean throwEx) {
        if(LOGGER.isDebugEnabled()) {
            LOGGER.debug(">>>>>> start getOne >>>>>>");
            LOGGER.debug("queryWrapper  : " + (queryWrapper != null ? queryWrapper.toString() : null));
            LOGGER.debug("throwEx       : " + throwEx);
        }
        if(throwEx) {
            return baseCrudMapper.selectOne(queryWrapper);
        }
        return SqlHelper.getObject(log, baseCrudMapper.selectList(queryWrapper));
    }

    @Override
    public Map<String, Object> getMap(Wrapper<T> queryWrapper) {
        if(LOGGER.isDebugEnabled()) {
            LOGGER.debug(">>>>>> start getMap >>>>>>");
            LOGGER.debug("queryWrapper  : " + (queryWrapper != null ? queryWrapper.toString() : null));
        }
        return SqlHelper.getObject(log, baseCrudMapper.selectMaps(queryWrapper));
    }

    @Override
    public <V> V getObj(Wrapper<T> queryWrapper, Function<? super Object, V> mapper) {
        if(LOGGER.isDebugEnabled()) {
            LOGGER.debug(">>>>>> start getObj >>>>>>");
            LOGGER.debug("queryWrapper  : " + (queryWrapper != null ? queryWrapper.toString() : null));
            LOGGER.debug("mapper        : " + (mapper != null ? mapper.toString() : null));
        }
        return SqlHelper.getObject(log, listObjs(queryWrapper, mapper));
    }

    @Override
    public List<T> list(Wrapper<T> queryWrapper) {
        if(LOGGER.isDebugEnabled()) {
            LOGGER.debug(">>>>>> start list >>>>>>");
            LOGGER.debug("queryWrapper  : " + (queryWrapper != null ? queryWrapper.toString() : null));
        }
        return baseCrudMapper.selectList(queryWrapper);
    }

    @Override
    public List<T> listByIds(Collection<? extends Serializable> idList) {
        if(LOGGER.isDebugEnabled()) {
            LOGGER.debug(">>>>>> start listByIds >>>>>>");
            LOGGER.debug("idList  : " + idList);
        }
        return baseCrudMapper.selectBatchIds(idList);
    }

    @Override
    public List<T> listByMap(Map<String, Object> columnMap) {
        if(LOGGER.isDebugEnabled()) {
            LOGGER.debug(">>>>>> start listByMap >>>>>>");
            LOGGER.debug("columnMap  : " + columnMap);
        }
        return baseCrudMapper.selectByMap(columnMap);
    }

    @Override
    public List<Map<String, Object>> listMaps(Wrapper<T> queryWrapper) {
        if(LOGGER.isDebugEnabled()) {
            LOGGER.debug(">>>>>> start listMaps >>>>>>");
            LOGGER.debug("queryWrapper  : " + (queryWrapper != null ? queryWrapper.toString() : null));
        }
        return baseCrudMapper.selectMaps(queryWrapper);
    }

    @Override
    public <V> List<V> listObjs(Wrapper<T> queryWrapper, Function<? super Object, V> mapper) {
        if(LOGGER.isDebugEnabled()) {
            LOGGER.debug(">>>>>> start listObjs >>>>>>");
            LOGGER.debug("queryWrapper  : " + (queryWrapper != null ? queryWrapper.toString() : null));
            LOGGER.debug("mapper        : " + (mapper != null ? mapper.toString() : null));
        }
        return baseCrudMapper.selectObjs(queryWrapper).stream().filter(Objects::nonNull).map(mapper).collect(Collectors.toList());
    }

    @Override
    public <E extends IPage<T>> E page(E page) {
        return page(page, Wrappers.emptyWrapper());
    }

    @Override
    public <E extends IPage<T>> E page(E page, Wrapper<T> queryWrapper) {
        if(LOGGER.isDebugEnabled()) {
            LOGGER.debug(">>>>>> start page >>>>>>");
            LOGGER.debug("page          : " + (page != null ? page.toString() : null));
            LOGGER.debug("queryWrapper  : " + (queryWrapper != null ? queryWrapper.toString() : null));
        }
        return baseCrudMapper.selectPage(page, queryWrapper);
    }

    @Override
    public <E extends IPage<Map<String, Object>>> E pageMaps(E page) {
        return pageMaps(page, Wrappers.emptyWrapper());
    }

    @Override
    public <E extends IPage<Map<String, Object>>> E pageMaps(E page, Wrapper<T> queryWrapper) {
        if(LOGGER.isDebugEnabled()) {
            LOGGER.debug(">>>>>> start pageMaps >>>>>>");
            LOGGER.debug("page          : " + (page != null ? page.toString() : null));
            LOGGER.debug("queryWrapper  : " + (queryWrapper != null ? queryWrapper.toString() : null));
        }
        return baseCrudMapper.selectMapsPage(page, queryWrapper);
    }

    @Override
    public long count(Wrapper<T> queryWrapper) {
        if(LOGGER.isDebugEnabled()) {
            LOGGER.debug(">>>>>> start count >>>>>>");
            LOGGER.debug("queryWrapper  : " + (queryWrapper != null ? queryWrapper.toString() : null));
        }
        return (int) SqlHelper.retCount(baseCrudMapper.selectCount(queryWrapper));
    }

    protected Class<M> currentMapperClass() {
        return (Class<M>) this.getResolvableType().as(BaseCrudServiceImpl.class).getGeneric(0).getType();
    }

    protected Class<T> currentModelClass() {
        return (Class<T>) this.getResolvableType().as(BaseCrudServiceImpl.class).getGeneric(1).getType();
    }

    /**
     * @see ResolvableType
     */
    protected ResolvableType getResolvableType() {
        return ResolvableType.forClass(ClassUtils.getUserClass(getClass()));
    }

    /**
     * 获取mapperStatementId
     *
     * @param sqlMethod 方法名
     * @return 命名id
     * @since 3.4.0
     */
    protected String getSqlStatement(SqlMethod sqlMethod) {
        return SqlHelper.getSqlStatement(mapperClass, sqlMethod);
    }

    /**
     * 执行批量操作
     *
     * @param list      数据集合
     * @param batchSize 批量大小
     * @param consumer  执行方法
     * @param <E>       泛型
     * @return 操作结果
     */
    protected <E> boolean executeBatch(Collection<E> list, int batchSize, BiConsumer<SqlSession, E> consumer) {
        return SqlHelper.executeBatch(this.entityClass, this.log, list, batchSize, consumer);
    }

    /**
     * 执行批量操作（默认批次提交数量 DEFAULT_BATCH_SIZE）
     *
     * @param list     数据集合
     * @param consumer 执行方法
     * @param <E>      泛型
     * @return 操作结果
     */
    protected <E> boolean executeBatch(Collection<E> list, BiConsumer<SqlSession, E> consumer) {
        return executeBatch(list, DEFAULT_BATCH_SIZE, consumer);
    }
}
