package com.iesms.openservices.photovoltaic.typehandler;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategies;
import com.fasterxml.jackson.databind.SerializationFeature;
import org.apache.ibatis.type.BaseTypeHandler;
import org.apache.ibatis.type.JdbcType;
import org.apache.ibatis.type.MappedJdbcTypes;
import org.apache.ibatis.type.MappedTypes;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.sql.CallableStatement;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

/**
 * @author wuyuxuan
 * @version 1.0.0
 * @since 2021/10/12
 */
@MappedTypes({Object.class})
@MappedJdbcTypes(JdbcType.VARCHAR)
@SuppressWarnings("unused")
public class JsonObjectTypeHandler<T> extends BaseTypeHandler<T> {

    private static final Logger LOGGER = LoggerFactory.getLogger(JsonObjectTypeHandler.class);

    protected static ObjectMapper objectMapper;
    protected final Class<T> type;

    static {
        objectMapper = new ObjectMapper();

        // 序列化
        /// >>> 格式化输出（以缩进的方式展示json）
        //objectMapper.enable(SerializationFeature.INDENT_OUTPUT);  # 不按缩进的方式展示，不然存入数据库的也是按缩进方式 2021-05-26
        /// >>> 遇到空对象不抛异常
        objectMapper.disable(SerializationFeature.FAIL_ON_EMPTY_BEANS);
        /// >>> 当出现 Java 类中未知的属性时不报错，而是忽略此 JSON 字段
        objectMapper.disable(SerializationFeature.FAIL_ON_UNWRAPPED_TYPE_IDENTIFIERS);

        // 反序列化
        /// >>> 遇到未知属性时不引起结果失败
        objectMapper.disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);

        // 禁用科学记数法
        objectMapper.configure(JsonGenerator.Feature.WRITE_BIGDECIMAL_AS_PLAIN, true);

        // 忽略未知的JSON属性
        objectMapper.configure(JsonGenerator.Feature.IGNORE_UNKNOWN, true);

        // 属性为NULL 不序列化
        objectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);

        // 开启下划线驼峰命名法，例如："number_value", "naming_strategy", "the_definite_proof"
        objectMapper.setPropertyNamingStrategy(PropertyNamingStrategies.SNAKE_CASE);
    }

    public JsonObjectTypeHandler(Class<T> type) {
        if(LOGGER.isDebugEnabled()) {
            LOGGER.debug("JsonTypeHandler(" + type + ")");
        }
        this.type = type;
    }

    @Override
    public void setNonNullParameter(PreparedStatement ps, int i, T parameter, JdbcType jdbcType) throws SQLException {
        ps.setString(i, toJson(parameter));
    }

    @Override
    public T getNullableResult(ResultSet rs, String columnName) throws SQLException {
        final String json = rs.getString(columnName);
        return json == null || json.trim().length() == 0 ? null : parse(json);
    }

    @Override
    public T getNullableResult(ResultSet rs, int columnIndex) throws SQLException {
        final String json = rs.getString(columnIndex);
        return json == null || json.trim().length() == 0 ? null : parse(json);
    }

    @Override
    public T getNullableResult(CallableStatement cs, int columnIndex) throws SQLException {
        final String json = cs.getString(columnIndex);
        return json == null || json.trim().length() == 0 ? null : parse(json);
    }

    public T parse(String json) {
        try {
            return objectMapper.readValue(json, type);
        }
        catch(IOException e) {
            LOGGER.error(e.getMessage(), e);
            throw new RuntimeException(e);
        }
    }

    public String toJson(T obj) {
        try {
            return objectMapper.writeValueAsString(obj);
        }
        catch(JsonProcessingException e) {
            LOGGER.error(e.getMessage(), e);
            throw new RuntimeException(e);
        }
    }

    public static void setObjectMapper(ObjectMapper objectMapper) {
        if(objectMapper == null) {
            LOGGER.error("ObjectMapper should not be null");
        }
        JsonObjectTypeHandler.objectMapper = objectMapper;
    }

}
