package com.gaea.spring.cloud.starter.component.aspect;

import cn.hutool.json.JSONUtil;
import com.gaea.utils.ClassUtils;
import com.gaea.utils.StringUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.aspectj.lang.annotation.Pointcut;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.util.Objects;

/**
 * @author wuyuxuan
 * @version 1.0.0
 * @since 2022/5/24
 */
@Aspect
@Component
@ConditionalOnProperty(name = "request-log", havingValue = "true")
public class WebLogAspect {

    private static final Logger LOGGER = LoggerFactory.getLogger(WebLogAspect.class);

    @Pointcut("execution(public * com..*.controller..*.*(..))")
    public void webLog() {
    }

    @Before("webLog()")
    public void doBefore(JoinPoint joinPoint) {
        // 开始打印请求日志
        ServletRequestAttributes attributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
        if (Objects.isNull(attributes)) {
            return;
        }
        HttpServletRequest request = attributes.getRequest();

        if (StringUtils.equals(joinPoint.getSignature().getName(), "getStdoutLog")) {
            return;
        }

        String enter = "\r\n";
        String prefix = "\r\n >>>>>>   ";

        String url = prefix + "URL      : {} ( {} )";
        String cls = prefix + "Class    : {}.{}";
        String ip = prefix + "IP       : {}";
        String req = prefix + "Request  : {}";

        LOGGER.info(enter + url + cls + ip + req + enter,
                request.getRequestURL().toString(),
                request.getMethod(),
                joinPoint.getSignature().getDeclaringTypeName(),
                joinPoint.getSignature().getName(),
                request.getRemoteAddr(),
                JSONUtil.toJsonStr(joinPoint.getArgs()));
    }

    /**
     * 环绕
     */
    @Around("webLog()")
    public Object doAround(ProceedingJoinPoint proceedingJoinPoint) throws Throwable {
        long startTime = System.currentTimeMillis();
        Object result = proceedingJoinPoint.proceed();

        if (!StringUtils.equals(proceedingJoinPoint.getSignature().getName(), "getStdoutLog")) {
            long time = System.currentTimeMillis() - startTime;
            if (time >= 2000) {
                LOGGER.error(">>> {} >> Response Time : {} ms",
                        ClassUtils.getClassNameByPath(proceedingJoinPoint.getSignature().getDeclaringTypeName()) + "." + proceedingJoinPoint.getSignature().getName(),
                        time);
            } else {
                LOGGER.info(">>> {} >> Response Time : {} ms",
                        ClassUtils.getClassNameByPath(proceedingJoinPoint.getSignature().getDeclaringTypeName()) + "." + proceedingJoinPoint.getSignature().getName(),
                        time);
            }
        }

        return result;
    }
}
