package com.gaea.spring.cloud.starter.component.exception;

import com.gaea.base.core.R;
import com.gaea.base.enums.CommonCodeEnum;
import com.gaea.base.error.BizException;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Component;
import org.springframework.validation.BindException;
import org.springframework.validation.BindingResult;
import org.springframework.validation.ObjectError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;

import javax.validation.ConstraintViolationException;
import javax.validation.UnexpectedTypeException;
import java.util.List;

/**
 * 全局异常处理
 *
 * @author wuyuxuan
 * @version 1.0.0
 * @since 2021/6/22
 */
@Order(-1)
@ResponseBody
@Component
@ControllerAdvice
public class GlobalException {

    private static final Logger LOGGER = LoggerFactory.getLogger(GlobalException.class);

    /**
     * 处理请求中
     */
    @ExceptionHandler(BindException.class)
    public R<?> bindExceptionHandler(BindException e) {
        BindingResult bindingResult = e.getBindingResult();
        if (bindingResult.hasErrors()) {
            return paramError(bindingResult);
        }
        return R.ok();
    }

    /**
     * 处理请求中
     */
    @ExceptionHandler(MethodArgumentNotValidException.class)
    public R<?> paramExceptionHandler(MethodArgumentNotValidException e) {
        BindingResult bindingResult = e.getBindingResult();
        if (bindingResult.hasErrors()) {
            return paramError(bindingResult);
        }
        return R.ok();
    }

    /**
     * 拦截业务异常处理
     */
    @ExceptionHandler(BizException.class)
    public R<?> bizException(BizException e) {
        LOGGER.error(">>> 业务异常 : {}", e.getMsg());
        return R.biz(e);
    }

    /**
     * 拦截系统错误处理
     */
    @ExceptionHandler(Exception.class)
    public R<?> exception(Exception e) {
        // TODO 系统异常
        LOGGER.error(">>> 系统异常 : ", e);
        return R.enums(CommonCodeEnum.SYSTEM_ERROR);
    }

    private static R<?> paramError(BindingResult bindingResult) {
        List<ObjectError> ls = bindingResult.getAllErrors();
        for (ObjectError item : ls) {
            if (StringUtils.isNotBlank(item.getDefaultMessage())) {
                LOGGER.error(">>> 请求参数异常 : {}", item.getDefaultMessage());
                return R.failure().msg(item.getDefaultMessage());
            } else {
                LOGGER.error(">>> 缺少参数或参数类型不匹配");
                return R.failure().msg("缺少参数或参数类型不匹配");
            }
        }
        return R.failure();
    }
}
