package com.gaea.spring.cloud.starter.component.swagger.config;

import cn.hutool.core.convert.Convert;
import cn.hutool.core.util.EnumUtil;
import com.gaea.base.core.IBaseEnum;
import com.gaea.utils.StringUtils;
import io.swagger.annotations.ApiModelProperty;
import lombok.SneakyThrows;
import org.springframework.context.annotation.Configuration;
import springfox.documentation.schema.Annotations;
import springfox.documentation.service.AllowableListValues;
import springfox.documentation.spi.DocumentationType;
import springfox.documentation.spi.schema.ModelPropertyBuilderPlugin;
import springfox.documentation.spi.schema.contexts.ModelPropertyContext;
import springfox.documentation.spi.service.ExpandedParameterBuilderPlugin;
import springfox.documentation.spi.service.contexts.ParameterExpansionContext;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;

/**
 * @author wuyuxuan
 * @version 1.0.0
 * @since 2022/8/5
 */
@Configuration
public class SwaggerPluginConfig implements ModelPropertyBuilderPlugin, ExpandedParameterBuilderPlugin {

    /**
     * 响应参数扩展
     *
     * @param context
     */
    @SneakyThrows
    @Override
    public void apply(ModelPropertyContext context) {
        if (context.getBeanPropertyDefinition().isPresent()) {
            Optional<ApiModelProperty> optional = Annotations.findPropertyAnnotation(context.getBeanPropertyDefinition().get(), ApiModelProperty.class);
            if (optional.isPresent()) {

                boolean isEnum;
                Class<?> enumClass;
                String fieldType = context.getBuilder().build().getType().toString();
                List<String> list;

                try {
                    isEnum = EnumUtil.isEnum(enumClass = Class.forName(fieldType));
                    // 如果请求参数字段类型是枚举类
                    if (!isEnum) {
                        isEnum = EnumUtil.isEnum(enumClass = Class.forName(optional.get().notes()));
                    } else {
                        context.getBuilder().allowableValues(new AllowableListValues(allowableListValues(enumClass, true), "LIST"));
                    }
                } catch (ClassNotFoundException e) {
                    isEnum = false;
                    enumClass = null;
                }

                if (isEnum) {
                    list = message(enumClass, true);
                    context.getBuilder().description(context.getBuilder().build().getDescription() + "：" + StringUtils.join("，", list));
                }
            }
        }
    }

    @Override
    public boolean supports(DocumentationType delimiter) {
        return true;
    }

    /**
     * 请求参数扩展
     *
     * @param context
     */
    @Override
    public void apply(ParameterExpansionContext context) {
        Optional<ApiModelProperty> annotation = context.findAnnotation(ApiModelProperty.class);
        if (annotation.isPresent()) {

            boolean isEnum = false;
            Class<?> enumClass;

            AtomicReference<String> fieldType = new AtomicReference<>("");
            context.getParameterBuilder().build().getType().ifPresent(type -> {
                fieldType.set(type.toString());
            });

            List<String> list;
            try {
                isEnum = EnumUtil.isEnum(enumClass = Class.forName(fieldType.get()));
                // 如果请求参数字段类型是枚举类
                if (isEnum) {
                    list = message(enumClass, false);
                    context.getParameterBuilder().description(context.getParameterBuilder().build().getDescription() + "：" + StringUtils.join("，", list) + "<br/>");
                    context.getParameterBuilder().allowableValues(new AllowableListValues(allowableListValues(enumClass, false), "LIST"));
                } else {
                    enumClass = Class.forName(annotation.get().notes());
                    isEnum = EnumUtil.isEnum(enumClass);
                    if (isEnum) {
                        list = message(enumClass, true);
                        context.getParameterBuilder().description(context.getParameterBuilder().build().getDescription() + "：" + StringUtils.join("，", list));
                    }
                }
            } catch (ClassNotFoundException e) {
            }
        }
    }

    private List<String> message(Class<?> unknown, boolean isShowValue) {
        List<String> list = new ArrayList<>();
        Object[] objects = unknown.getEnumConstants();
        for (Object enumObject : objects) {
            IBaseEnum<?> iBaseEnum = Convert.convert(IBaseEnum.class, enumObject);
            if (isShowValue) {
                list.add(iBaseEnum.getValue() + " - " + iBaseEnum.getName());
            } else {
                list.add(enumObject.toString() + " - " + iBaseEnum.getName());
            }
        }
        return list;
    }

    private List<String> allowableListValues(Class<?> unknown, boolean isShowValue) {
        List<String> list = new ArrayList<>();
        Object[] objects = unknown.getEnumConstants();
        for (Object enumObject : objects) {
            IBaseEnum<?> iBaseEnum = Convert.convert(IBaseEnum.class, enumObject);
            if (isShowValue) {
                list.add(iBaseEnum.getValue().toString());
            } else {
                list.add(enumObject.toString());
            }
        }
        return list;
    }
}