package com.gaea.spring.cloud.starter.context;

import cn.hutool.json.JSONUtil;
import com.gaea.base.core.R;
import com.gaea.base.pool.StringPool;
import com.gaea.utils.StringUtils;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Objects;

import static java.nio.charset.StandardCharsets.UTF_8;

/**
 * @author wyx
 * @version 1.0.0
 * @since 2021/12/29
 */
public class HttpContext {

    public static final String AUTHORIZATION = "Authorization";
    public static final String APPLICATION_JSON = "application/json";
    public static final String BEARER = "bearer ";

    /**
     * 获取当前请求 request 对象
     *
     * @return HttpServletRequest
     */
    public static HttpServletRequest getRequest() {
        return ((ServletRequestAttributes) Objects.requireNonNull(RequestContextHolder.getRequestAttributes())).getRequest();
    }

    /**
     * 获取当前请求 response 对象
     *
     * @return HttpServletResponse
     */
    public static HttpServletResponse getResponse() {
        return ((ServletRequestAttributes) Objects.requireNonNull(RequestContextHolder.getRequestAttributes())).getResponse();
    }

    /**
     * HttpServletResponse 下载文件
     *
     * @param in
     * @param fileName
     */
    public static void downloadFile(InputStream in, String fileName) {
        HttpServletResponse response = getResponse();
        OutputStream out = null;
        try {
            out = response.getOutputStream();
            response.setContentType("application/octet-stream;charset=utf-8");
            response.setHeader("Content-disposition", StringUtils.format("attachment; filename=\"{}\"; filename*=utf-8''{}", URLEncoder.encode(fileName, UTF_8.name()), URLEncoder.encode(fileName, UTF_8.name())));
            byte[] buff = new byte[1024];
            int length;
            while ((length = in.read(buff)) > 0) {
                out.write(buff, 0, length);
            }
            out.flush();
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (out != null) {
                try {
                    out.close();
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
            if (in != null) {
                try {
                    in.close();
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }
    }

    /**
     * HttpServletResponse 渲染 json 数据
     *
     * @param object Object
     */
    public static void renderJson(R<?> object) {
        HttpServletResponse response = getResponse();
        ServletOutputStream writer = null;
        try {
            response.setContentType(APPLICATION_JSON);
            response.setCharacterEncoding(StringPool.UTF_8);
            writer = response.getOutputStream();
            writer.write(JSONUtil.toJsonStr(object).getBytes(StandardCharsets.UTF_8));
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (writer != null) {
                try {
                    writer.close();
                } catch (Exception e) {
                    e.printStackTrace();
                }

            }
        }
    }

}