package com.gaea.spring.cloud.starter.util;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.concurrent.*;

/**
 * 全局公共线程池（工具类）
 *
 * @author wuyuxuan
 * @version 1.0.0
 * @since 2024/05/16
 */
public class ThreadPoolUtil {

    private static final Logger LOGGER = LoggerFactory.getLogger(ThreadPoolUtil.class);
    private static ExecutorService executor;

    static {
        init();
    }

    /**
     * 初始化全局线程池
     */
    synchronized private static void init() {
        if (null != executor) {
            executor.shutdownNow();
        }
        int cpuNum = Runtime.getRuntime().availableProcessors();
        executor = threadPoolExecutor(cpuNum + 1, cpuNum * 2, 300, TimeUnit.SECONDS, new ArrayBlockingQueue<>(5000), Executors.defaultThreadFactory(), new ThreadPoolExecutor.AbortPolicy());
    }

    /**
     * 直接在公共线程池中执行线程
     *
     * @param runnable 可运行对象
     */
    public static void execute(Runnable runnable) {
        try {
            executor.execute(runnable);
        } catch (Exception e) {
            LOGGER.error(">>> e : ", e);
            throw new RuntimeException(e);
        }
    }

    /**
     * 获取公共线程池
     *
     * @return
     */
    public static ExecutorService getExecutor() {
        return executor;
    }

    /**
     * 自定义新的线程池
     *
     * @param coreSize
     * @param maxSize
     * @param time
     * @param unit
     * @param queue
     * @param factory
     * @param handler
     * @return
     */
    public static ExecutorService threadPoolExecutor(int coreSize, int maxSize, int time, TimeUnit unit, BlockingQueue<Runnable> queue, ThreadFactory factory, RejectedExecutionHandler handler) {
        return new ThreadPoolExecutor(coreSize, maxSize, time, unit, queue, factory, handler);
    }


}
