package com.easesource.iot.datacenter.openservice.tablestore.dao.impl;

import com.alicloud.openservices.tablestore.ClientException;
import com.alicloud.openservices.tablestore.TableStoreException;
import com.alicloud.openservices.tablestore.model.*;
import com.easesource.commons.util.NumberUtils;
import com.easesource.commons.util.ObjectUtils;
import com.easesource.commons.util.StringUtils;
import com.easesource.commons.util.convert.JsonConvertUtils;
import com.easesource.data.jdbc.SqlParamsBuilder;
import com.easesource.iot.datacenter.openservice.entity.MeasDataLatestInfo;
import com.easesource.iot.datacenter.openservice.tablestore.AbstractEaseIotBaseTableStoreDao;
import com.easesource.iot.datacenter.openservice.tablestore.TableStoreTemplate;
import com.easesource.iot.datacenter.openservice.tablestore.dao.MeasDataLatestDao;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.collections4.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Repository;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;

/**
 * @author Nick Zhang
 */
@Repository
public class MeasDataLatestDaoImpl extends AbstractEaseIotBaseTableStoreDao implements MeasDataLatestDao {

    private final Logger logger = LoggerFactory.getLogger(getClass());

    @Resource(name = "easeIotTableStoreTemplate")
    private TableStoreTemplate easeIotTableStoreTemplate;

    @Qualifier("easeIotJdbcTemplate")
    @Resource(name = "easeIotJdbcTemplate")
    private JdbcTemplate easeIotJdbcTemplate;

    @Override
    public int updateMeasDataInfoLatest(Long measPointId, String measItemCode, Map<String, Object> measDataInfoMap, int measDataSource, long gmtMeasDataLatest) {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>>> start updateMeasDataInfoLatest >>>>>>> ");
            logger.debug(" measPointId          : " + measPointId);
            logger.debug(" measItemCode         : " + measItemCode);
            logger.debug(" measDataInfoMap      : " + JsonConvertUtils.convertToString(measDataInfoMap));
            logger.debug(" measDataSource       : " + measDataSource);
            logger.debug(" gmtMeasDataLatest    : " + gmtMeasDataLatest);
        }
        // 构造主键
        PrimaryKeyBuilder primaryKeyBuilder = PrimaryKeyBuilder.createPrimaryKeyBuilder();
        primaryKeyBuilder.addPrimaryKeyColumn("meas_point_id", PrimaryKeyValue.fromLong(measPointId));      // 分片键
        primaryKeyBuilder.addPrimaryKeyColumn("meas_item_code", PrimaryKeyValue.fromString(measItemCode));
        PrimaryKey primaryKey = primaryKeyBuilder.build();
        RowPutChange rowPutChange = new RowPutChange("meas_data_latest", primaryKey);
        //加入一些属性列
        rowPutChange.addColumn(new Column("meas_data_info", ColumnValue.fromString(JsonConvertUtils.convertToString(measDataInfoMap)), System.currentTimeMillis()));
        rowPutChange.addColumn(new Column("meas_data_source", ColumnValue.fromLong(measDataSource), System.currentTimeMillis()));
        rowPutChange.addColumn(new Column("gmt_meas_data_latest", ColumnValue.fromLong(System.currentTimeMillis()), System.currentTimeMillis()));
        PutRowRequest putRowRequest = new PutRowRequest(rowPutChange);
        easeIotTableStoreTemplate.putRowSync(putRowRequest);
        return 1;
    }

    @Override
    public int batchUpdateMeasDataInfoLatest(Map<Long, Map<String, Map<String, Object>>> batchMeasDataInfosMap) {
        if(logger.isDebugEnabled()) {
            logger.debug(" >>>>>>> start batchUpdateMeasDataInfoLatest >>>>>>> ");
            logger.debug(" batchMeasDataInfosMap    : " + JsonConvertUtils.convertToString(batchMeasDataInfosMap));
        }
        BatchWriteRowRequest batchWriteRowRequest = new BatchWriteRowRequest();
        int n = 0;
        int i = 0;
        for(Long measPointId : batchMeasDataInfosMap.keySet()) {
            Map<String, Map<String, Object>> insideMap = batchMeasDataInfosMap.get(measPointId);
            /*if(measPointId == 104409354721034285L) {
                logger.info("measPointId    : " + measPointId);
                logger.info("insideMap      : " + JsonConvertUtils.convertToString(insideMap));
            }*/
            for(String measItemCode : insideMap.keySet()) {
                n++;
                i++;
                Map<String, Object> measDataInfoMap = insideMap.get(measItemCode);
                // 构造rowPutChange
                PrimaryKeyBuilder primaryKeyBuilder = PrimaryKeyBuilder.createPrimaryKeyBuilder();
                primaryKeyBuilder.addPrimaryKeyColumn("meas_point_id", PrimaryKeyValue.fromLong(measPointId));      // 分片键
                primaryKeyBuilder.addPrimaryKeyColumn("meas_item_code", PrimaryKeyValue.fromString(measItemCode));
                PrimaryKey primaryKey = primaryKeyBuilder.build();
                RowPutChange rowPutChange = new RowPutChange("meas_data_latest", primaryKey);
                //加入一些属性列
                int measDataSource = NumberUtils.toInt(ObjectUtils.toStringDefaultEmpty(measDataInfoMap.get("measDataSource")), 0);
                long gmtMeasDataLatest = NumberUtils.toLong(ObjectUtils.toStringDefaultEmpty(measDataInfoMap.get("gmtMeasDataLatest")), System.currentTimeMillis());
                rowPutChange.addColumn(new Column("meas_data_info", ColumnValue.fromString(JsonConvertUtils.convertToString(measDataInfoMap)), System.currentTimeMillis()));
                rowPutChange.addColumn(new Column("meas_data_source", ColumnValue.fromLong(measDataSource), System.currentTimeMillis()));
                rowPutChange.addColumn(new Column("gmt_meas_data_latest", ColumnValue.fromLong(System.currentTimeMillis()), System.currentTimeMillis()));
                // 添加到batch操作中
                batchWriteRowRequest.addRowChange(rowPutChange);
                if(i == 199) {
                    easeIotTableStoreTemplate.batchWriteRowSync(batchWriteRowRequest);
                    i = 0;
                    batchWriteRowRequest = new BatchWriteRowRequest();
                }
            }
        }
        if(i > 0) {
            easeIotTableStoreTemplate.batchWriteRowSync(batchWriteRowRequest);
        }
        return n;
    }

    @Override
    public int deleteByMeasPointIdAndMeasItemCode(Long measPointId, String measItemCode) {
        // 构造主键
        PrimaryKeyBuilder primaryKeyBuilder = PrimaryKeyBuilder.createPrimaryKeyBuilder();
        primaryKeyBuilder.addPrimaryKeyColumn("meas_point_id", PrimaryKeyValue.fromLong(measPointId));      // 分片键
        primaryKeyBuilder.addPrimaryKeyColumn("meas_item_code", PrimaryKeyValue.fromString(measItemCode));
        PrimaryKey primaryKey = primaryKeyBuilder.build();
        RowDeleteChange rowDeleteChange = new RowDeleteChange("meas_data_latest", primaryKey);
        DeleteRowRequest deleteRowRequest = new DeleteRowRequest(rowDeleteChange);
        easeIotTableStoreTemplate.deleteRowSync(deleteRowRequest);
        return 1;
    }

    @Override
    public Map<String, MeasDataLatestInfo> getMeasDataLatestInfoMap(Long measPointId, Set<String> measItemCodeSet, Integer measDataSideType) {
        SqlParamsBuilder sqlParamsBuilder = new SqlParamsBuilder();
        sqlParamsBuilder.append("SELECT gdm.meas_point_id, gdm.self_rate, gdm.ct_rate, gdm.pt_rate ");
        sqlParamsBuilder.append(" FROM iesms.gm_dev_meter gdm ");
        sqlParamsBuilder.append(" WHERE gdm.meas_point_id = ? ", measPointId);
        sqlParamsBuilder.append(" LIMIT 1 ");
        Map<String, Object> gdmMap;
        try {
            gdmMap = easeIotJdbcTemplate.queryForObject(sqlParamsBuilder.getSql(), sqlParamsBuilder.getParams(), mapRowMapper);
        }
        catch(EmptyResultDataAccessException _erdae) {
            gdmMap = null;
        }
        BigDecimal selfRate = gdmMap != null ? (BigDecimal) gdmMap.get("self_rate") : new BigDecimal("1.0");
        if(!(selfRate != null && selfRate.compareTo(BigDecimal.ZERO) > 0)) {
            selfRate = new BigDecimal("1.0");
        }
        BigDecimal ctRate = gdmMap != null ? (BigDecimal) gdmMap.get("ct_rate") : new BigDecimal("1.0");
        if(!(ctRate != null && ctRate.compareTo(BigDecimal.ZERO) > 0)) {
            ctRate = new BigDecimal("1.0");
        }
        BigDecimal ptRate = gdmMap != null ? (BigDecimal) gdmMap.get("pt_rate") : new BigDecimal("1.0");
        if(!(ptRate != null && ptRate.compareTo(BigDecimal.ZERO) > 0)) {
            ptRate = new BigDecimal("1.0");
        }

        //
        Map<String, MeasDataLatestInfo> measDataLatestInfoMap = Maps.newHashMap();
        List<String> measItemCodeList = Lists.newArrayList(measItemCodeSet);
        List<List<String>> listList = Lists.partition(measItemCodeList, 100);
        for(List<String> list : listList) {
            Set<String> measItemCodeSetTemp = Sets.newHashSet(list);
            MultiRowQueryCriteria multiRowQueryCriteria = new MultiRowQueryCriteria("meas_data_latest");
            for(String measItemCode : measItemCodeSetTemp) {
                if(StringUtils.isNotBlank(measItemCode)) {
                    try {
                        PrimaryKeyBuilder primaryKeyBuilder = PrimaryKeyBuilder.createPrimaryKeyBuilder();
                        primaryKeyBuilder.addPrimaryKeyColumn("meas_point_id", PrimaryKeyValue.fromLong(measPointId));      // 分片键
                        primaryKeyBuilder.addPrimaryKeyColumn("meas_item_code", PrimaryKeyValue.fromString(measItemCode));
                        PrimaryKey primaryKey = primaryKeyBuilder.build();
                        multiRowQueryCriteria.addRow(primaryKey);
                    }
                    catch(Exception _e) {
                        logger.error(_e.getMessage(), _e.fillInStackTrace());
                    }
                }
            }
            if(multiRowQueryCriteria.isEmpty()) {
                return measDataLatestInfoMap;
            }
            // 设置读取最新版本
            multiRowQueryCriteria.setMaxVersions(1);
            // 设置获取列名
            multiRowQueryCriteria.addColumnsToGet(Arrays.asList("meas_data_info", "meas_data_source", "gmt_meas_data_latest"));

            BatchGetRowRequest batchGetRowRequest = new BatchGetRowRequest();
            // batchGetRow支持读取多个表的数据, 一个multiRowQueryCriteria对应一个表的查询条件, 可以添加多个multiRowQueryCriteria.
            batchGetRowRequest.addMultiRowQueryCriteria(multiRowQueryCriteria);
            BatchGetRowResponse batchGetRowResponse = null;
            try {
                batchGetRowResponse = easeIotTableStoreTemplate.batchGetRowSync(batchGetRowRequest);
                List<BatchGetRowResponse.RowResult> rowResults = batchGetRowResponse.getBatchGetRowResult("meas_data_latest");
                if(rowResults != null && rowResults.size() > 0) {
                    for(BatchGetRowResponse.RowResult rowResult : rowResults) {
                        if(rowResult.isSucceed()) {
                            Row row = rowResult.getRow();
                            if(row != null) {
                                MeasDataLatestInfo measDataLatestInfo = row2MeasDataLatestInfo(row, measDataSideType, selfRate, ctRate, ptRate);
                                measDataLatestInfoMap.put(measDataLatestInfo.getMeasItemCode(), measDataLatestInfo);
                            }
                        }
                        else {
                            PrimaryKey primaryKey = batchGetRowRequest.getPrimaryKey(rowResult.getTableName(), rowResult.getIndex());
                            if(primaryKey != null) {
                                // 读一行
                                SingleRowQueryCriteria criteria = new SingleRowQueryCriteria("meas_data_latest", primaryKey);
                                // 设置读取最新版本
                                criteria.setMaxVersions(1);
                                // 设置获取列名
                                criteria.addColumnsToGet(Arrays.asList("meas_data_info", "meas_data_source", "gmt_meas_data_latest"));
                                //
                                GetRowRequest getRowRequest = new GetRowRequest(criteria);
                                GetRowResponse getRowResponse = easeIotTableStoreTemplate.getRowSync(getRowRequest);
                                Row row = getRowResponse.getRow();
                                if(row != null) {
                                    MeasDataLatestInfo measDataLatestInfo = row2MeasDataLatestInfo(row, measDataSideType, selfRate, ctRate, ptRate);
                                    measDataLatestInfoMap.put(measDataLatestInfo.getMeasItemCode(), measDataLatestInfo);
                                }
                            }
                        }
                    }
                }
            }
            catch(TableStoreException | ClientException e) {
                logger.error(e.getMessage(), e.fillInStackTrace());
            }
            finally {
                if(batchGetRowResponse != null) {
                    batchGetRowResponse = null;
                }
            }
        }
        return measDataLatestInfoMap;
    }

    @Override
    public Map<Long, Map<String, MeasDataLatestInfo>> batchGetMeasDataLatestInfoMap(Set<Long> measPointIdSet, Set<String> measItemCodeSet, Integer measDataSideType) {
        Map<Long, Map<String, MeasDataLatestInfo>> batchMeasDataLatestInfoMap = Maps.newHashMap();
        for(Long measPointId : measPointIdSet) {
            batchMeasDataLatestInfoMap.put(measPointId, getMeasDataLatestInfoMap(measPointId, measItemCodeSet, measDataSideType));
        }
        return batchMeasDataLatestInfoMap;
    }

    private MeasDataLatestInfo row2MeasDataLatestInfo(Row row, Integer measDataSideType, BigDecimal selfRate, BigDecimal ctRate, BigDecimal ptRate) {
        MeasDataLatestInfo measDataLatestInfo = null;
        if(row != null) {
            measDataLatestInfo = new MeasDataLatestInfo();
            PrimaryKey primaryKey = row.getPrimaryKey();
            if(primaryKey != null) {
                PrimaryKeyColumn measPointIdCol = primaryKey.getPrimaryKeyColumn("meas_point_id");
                PrimaryKeyColumn measItemCodeCol = primaryKey.getPrimaryKeyColumn("meas_item_code");
                Long measPointId = measPointIdCol.getValue().asLong();
                String measItemCode = measItemCodeCol.getValue().asString();
                measDataLatestInfo.setMeasPointId(measPointId);
                measDataLatestInfo.setMeasItemCode(measItemCode);

                Column measDataInfoCol = row.getLatestColumn("meas_data_info");
                if(measDataInfoCol != null) {
                    String measDataInfoStr = measDataInfoCol.getValue().asString();
                    Map measDataInfoMap = JsonConvertUtils.convertFromString(measDataInfoStr, Map.class);
                    String measDataClassName = (String) measDataInfoMap.get("meas_data_class");
                    if(StringUtils.isBlank(measDataClassName)) {
                        measDataClassName = (String) measDataInfoMap.get("measDataClass");
                    }
                    Object measDataValueObj = measDataInfoMap.get("meas_data_value");
                    if(measDataValueObj == null) {
                        measDataValueObj = measDataInfoMap.get("measDataValue");
                    }
                    Class measDataClass = null;
                    Object measDataValue = null;
                    /*BigDecimal selfRate = new BigDecimal("1.0");
                    BigDecimal ctRate = new BigDecimal("1.0");
                    BigDecimal ptRate = new BigDecimal("1.0");*/
                    if(StringUtils.equals(measDataClassName, "java.math.BigDecimal") && measDataValueObj != null) {
                        measDataClass = java.math.BigDecimal.class;
                        measDataValue = new BigDecimal(measDataValueObj.toString()).setScale(8, BigDecimal.ROUND_HALF_UP);
                        if(measDataSideType != null && measDataSideType == 1) {
                            //measDataValue = new BigDecimal(measDataInfoMap.get("measDataValue").toString()).multiply(new BigDecimal("2.0")).setScale(6, BigDecimal.ROUND_HALF_UP);
                            if(StringUtils.containsAny(measItemCode, "gen_watt_ua", "gen_watt_ub", "gen_watt_uc", "gen_watt_upv", "gen_watt_ur", "gen_watt_ubus")) {
                                // 电压 * PT
                                measDataValue = new BigDecimal(measDataValueObj.toString()).multiply(selfRate).multiply(ptRate).setScale(8, BigDecimal.ROUND_HALF_UP);
                            }
                            else if(StringUtils.containsAny(measItemCode, "gen_watt_ia", "gen_watt_ib", "gen_watt_ic", "gen_watt_in", "gen_watt_ipv", "gen_watt_ir", "gen_watt_ibus")) {
                                // 电流 * CT
                                measDataValue = new BigDecimal(measDataValueObj.toString()).multiply(selfRate).multiply(ctRate).setScale(8, BigDecimal.ROUND_HALF_UP);
                            }
                            else if(StringUtils.containsAny(measItemCode, "gen_watt_acp", "gen_watt_rep", "gen_watt_app", "gen_watt_ppv", "gen_watt_pr", "gen_watt_pbus")) {
                                // 功率 * CT * PT
                                measDataValue = new BigDecimal(measDataValueObj.toString()).multiply(selfRate).multiply(ctRate).multiply(ptRate).setScale(8, BigDecimal.ROUND_HALF_UP);
                            }
                            else if(StringUtils.containsAny(measItemCode, "gen_watt_pae", "gen_watt_rae", "gen_watt_cae", "gen_watt_q1re", "gen_watt_q2re", "gen_watt_q3re", "gen_watt_q4re", "gen_watt_pre", "gen_watt_rre")) {
                                // 电量 * CT * PT
                                measDataValue = new BigDecimal(measDataValueObj.toString()).multiply(selfRate).multiply(ctRate).multiply(ptRate).setScale(8, BigDecimal.ROUND_HALF_UP);
                            }
                        }
                    }
                    else if(StringUtils.equals(measDataClassName, "java.lang.Boolean") && measDataValueObj != null) {
                        measDataClass = java.lang.Boolean.class;
                        measDataValue = Boolean.valueOf(measDataValueObj.toString());
                    }
                    else if(StringUtils.equals(measDataClassName, "java.lang.String") && measDataValueObj != null) {
                        measDataClass = java.lang.String.class;
                        measDataValue = measDataValueObj.toString();
                    }
                    else if(StringUtils.equals(measDataClassName, "java.lang.Integer") && measDataValueObj != null) {
                        measDataClass = java.lang.Integer.class;
                        measDataValue = Integer.valueOf(measDataValueObj.toString());
                    }
                    else if(StringUtils.equals(measDataClassName, "java.lang.Long") && measDataValueObj != null) {
                        measDataClass = java.lang.Long.class;
                        measDataValue = Long.valueOf(measDataValueObj.toString());
                    }
                    else if(StringUtils.equals(measDataClassName, "java.util.Date") && measDataValueObj != null) {
                        measDataClass = java.util.Date.class;
                        measDataValue = new Date(Long.parseLong(measDataValueObj.toString()));
                    }
                    measDataLatestInfo.setMeasDataClass(measDataClass);
                    measDataLatestInfo.setMeasDataValue(measDataValue);
                }
                Column measDataSourceCol = row.getLatestColumn("meas_data_source");
                if(measDataSourceCol != null) {
                    measDataLatestInfo.setMeasDataSource(NumberUtils.toInt(String.valueOf(measDataSourceCol.getValue().asLong()), 0));
                }
                Column gmtMeasDataLatestCol = row.getLatestColumn("gmt_meas_data_latest");
                if((gmtMeasDataLatestCol != null)) {
                    long gmtMeasDataLatest = gmtMeasDataLatestCol.getValue().asLong();
                    measDataLatestInfo.setGmtMeasDataLatest(gmtMeasDataLatest);
                    measDataLatestInfo.setGmtCreate(gmtMeasDataLatestCol.getTimestamp());
                    measDataLatestInfo.setGmtModified(gmtMeasDataLatestCol.getTimestamp());
                }
            }
        }
        return measDataLatestInfo;
    }

}
