package com.easesource.iot.datacenter.openservice.tablestore;

import com.easesource.commons.util.MapUtils;
import com.easesource.data.bean.Pager;
import com.google.common.collect.Maps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.core.RowMapper;

import java.math.BigDecimal;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Types;
import java.util.Collections;
import java.util.Date;
import java.util.Map;
import java.util.Set;

/**
 * @author Nick Zhang
 */
public abstract class AbstractEaseIotBaseTableStoreDao {

    private final Logger logger = LoggerFactory.getLogger(getClass());

    protected final RowMapper<Map<String, Object>> mapRowMapper = (rs, rowNum) -> {
        ResultSetMetaData rsmd = rs.getMetaData();
        Map<String, Object> map = Maps.newHashMap();
        for(int i = 1; i <= rsmd.getColumnCount(); i++) {
            if(rs.getObject(i) != null) {
                switch(rsmd.getColumnType(i)) {
                    case Types.TINYINT:
                        if(rsmd.getScale(i) == 0 && rsmd.getPrecision(i) == 1) {
                            map.put(rsmd.getColumnLabel(i).trim(), rs.getBoolean(i));
                        }
                        else {
                            map.put(rsmd.getColumnLabel(i).trim(), rs.getInt(i));
                        }
                        break;
                    case Types.SMALLINT:
                        map.put(rsmd.getColumnLabel(i).trim(), rs.getInt(i));
                        break;
                    case Types.INTEGER:
                        map.put(rsmd.getColumnLabel(i).trim(), rs.getInt(i));
                        break;
                    case Types.BIGINT:
                        map.put(rsmd.getColumnLabel(i).trim(), rs.getLong(i));
                        break;
                    case Types.FLOAT:
                        map.put(rsmd.getColumnLabel(i).trim(), new BigDecimal(Float.toString(rs.getFloat(i))));
                        break;
                    case Types.REAL:
                        map.put(rsmd.getColumnLabel(i).trim(), new BigDecimal(Float.toString(rs.getFloat(i))));
                        break;
                    case Types.DOUBLE:
                        map.put(rsmd.getColumnLabel(i).trim(), new BigDecimal(Double.toString(rs.getDouble(i))));
                        break;
                    case Types.DECIMAL:
                        if(rsmd.getScale(i) == 0) {
                            map.put(rsmd.getColumnLabel(i).trim(), rs.getBigDecimal(i).longValue());
                        }
                        else {
                            map.put(rsmd.getColumnLabel(i).trim(), rs.getBigDecimal(i));
                        }
                        break;
                    case Types.NUMERIC:
                        if(rsmd.getScale(i) == 0) {
                            map.put(rsmd.getColumnLabel(i).trim(), rs.getBigDecimal(i).longValue());
                        }
                        else {
                            map.put(rsmd.getColumnLabel(i).trim(), rs.getBigDecimal(i));
                        }
                        break;
                    case Types.DATE:
                        map.put(rsmd.getColumnLabel(i).trim(), new Date(rs.getDate(i).getTime()));
                        break;
                    case Types.TIME:
                        map.put(rsmd.getColumnLabel(i).trim(), new Date(rs.getTime(i).getTime()));
                        break;
                    case Types.TIMESTAMP:
                        map.put(rsmd.getColumnLabel(i).trim(), new Date(rs.getTimestamp(i).getTime()));
                        break;
                    case Types.CHAR:
                        map.put(rsmd.getColumnLabel(i).trim(), rs.getString(i));
                        break;
                    case Types.NCHAR:
                        map.put(rsmd.getColumnLabel(i).trim(), rs.getString(i));
                        break;
                    case Types.VARCHAR:
                        map.put(rsmd.getColumnLabel(i).trim(), rs.getString(i));
                        break;
                    case Types.NVARCHAR:
                        map.put(rsmd.getColumnLabel(i).trim(), rs.getString(i));
                        break;
                    case Types.LONGVARCHAR:
                        map.put(rsmd.getColumnLabel(i).trim(), rs.getString(i));
                        break;
                    case Types.LONGNVARCHAR:
                        map.put(rsmd.getColumnLabel(i).trim(), rs.getString(i));
                        break;
                    default:
                        map.put(rsmd.getColumnLabel(i).trim(), rs.getObject(i));
                        break;
                }
            }
        }
        return map;
    };

    protected boolean isExistColumn(ResultSet rs, String columnName) {
        try {
            if(rs.findColumn(columnName) > 0 ) {
                return true;
            }
        }
        catch(SQLException sqle) {
            // logger.error(sqle.getMessage(), sqle.fillInStackTrace());
            return false;
        }
        return false;
    }

    protected Map<String, Object> handleParamMapForInList(Map<String, Object> params) {
        params = MapUtils.emptyIfNull(params);
        Set<String> keys = params.keySet();
        for(String key : keys) {
            if(params.get(key) != null && params.get(key).getClass().isArray()) {
                params.put(key, Collections.singletonList(params.get(key)));
            }
        }
        return params;
    }

    protected StringBuilder generateSqlForSorter() {
        StringBuilder sqlPaginationBuilder = new StringBuilder();
        sqlPaginationBuilder.append(" LIMIT :start, :size ");
        return sqlPaginationBuilder;
    }

    protected StringBuilder generateSqlForPager() {
        StringBuilder sqlPaginationBuilder = new StringBuilder();
        sqlPaginationBuilder.append(" LIMIT :start, :size ");
        return sqlPaginationBuilder;
    }

    protected Map<String, Object> handleParamMapForPager(Map<String, Object> params, Pager pager) {
        if(pager != null) {
            params = MapUtils.emptyIfNull(params);
            params.put("start", pager.getStart());
            params.put("size", pager.getSize());
        }
        return params;
    }

}
