package com.easesource.iot.datacenter.openservice.tablestore.dao.impl;

import com.alicloud.openservices.tablestore.ClientException;
import com.alicloud.openservices.tablestore.TableStoreException;
import com.alicloud.openservices.tablestore.model.*;
import com.easesource.commons.util.DateUtils;
import com.easesource.commons.util.NumberUtils;
import com.easesource.commons.util.StringUtils;
import com.easesource.commons.util.convert.DateConvertUtils;
import com.easesource.data.jdbc.SqlParamsBuilder;
import com.easesource.iot.datacenter.openservice.entity.MeasDataInsFreezeCurveValue;
import com.easesource.iot.datacenter.openservice.entity.MeasDataLatestInfo;
import com.easesource.iot.datacenter.openservice.tablestore.AbstractEaseIotBaseTableStoreDao;
import com.easesource.iot.datacenter.openservice.tablestore.TableStoreTemplate;
import com.easesource.iot.datacenter.openservice.tablestore.dao.MeasDataInsFreezeCurveDao;
import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Repository;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;

/**
 * @author Nick Zhang
 */
@Repository
public class MeasDataInsFreezeCurveDaoImpl extends AbstractEaseIotBaseTableStoreDao implements MeasDataInsFreezeCurveDao {

    private final Logger logger = LoggerFactory.getLogger(getClass());

    @Resource(name = "easeIotTableStoreTemplate")
    private TableStoreTemplate easeIotTableStoreTemplate;

    @Qualifier("easeIotJdbcTemplate")
    @Resource(name = "easeIotJdbcTemplate")
    private JdbcTemplate easeIotJdbcTemplate;

    @Override
    public int updateMeasDataInsFreezeCurve(Long measPointId, String measItemCode, long gmtMeasFreeze, BigDecimal measDataValue, BigDecimal measDataRate, int measDataSource, long gmtMeasData, int freezeMode) {
        String partKey = Joiner.on('_').skipNulls().join(DateConvertUtils.convertToString(gmtMeasFreeze, DateConvertUtils.PATTERN_YEAR), measPointId);   // 分片键（YYYY_measPointId）
        // 构造主键
        PrimaryKeyBuilder primaryKeyBuilder = PrimaryKeyBuilder.createPrimaryKeyBuilder();
        primaryKeyBuilder.addPrimaryKeyColumn("part_key", PrimaryKeyValue.fromString(partKey));
        primaryKeyBuilder.addPrimaryKeyColumn("meas_point_id", PrimaryKeyValue.fromLong(measPointId));
        primaryKeyBuilder.addPrimaryKeyColumn("meas_item_code", PrimaryKeyValue.fromString(measItemCode));
        primaryKeyBuilder.addPrimaryKeyColumn("gmt_meas_freeze", PrimaryKeyValue.fromLong(gmtMeasFreeze));
        PrimaryKey primaryKey = primaryKeyBuilder.build();
        RowPutChange rowPutChange = new RowPutChange("meas_data_ins_freeze_curve", primaryKey);
        //加入一些属性列
        rowPutChange.addColumn(new Column("meas_data_value", ColumnValue.fromDouble(measDataValue.doubleValue()), System.currentTimeMillis()));
        rowPutChange.addColumn(new Column("meas_data_rate", ColumnValue.fromDouble(measDataRate.doubleValue()), System.currentTimeMillis()));
        rowPutChange.addColumn(new Column("meas_data_source", ColumnValue.fromLong(measDataSource), System.currentTimeMillis()));
        rowPutChange.addColumn(new Column("gmt_meas_data", ColumnValue.fromLong(gmtMeasData), System.currentTimeMillis()));
        rowPutChange.addColumn(new Column("freeze_mode", ColumnValue.fromLong(freezeMode), System.currentTimeMillis()));
        PutRowRequest putRowRequest = new PutRowRequest(rowPutChange);
        easeIotTableStoreTemplate.putRowSync(putRowRequest);
        return 1;
    }

    @Override
    public int deleteMeasDataInsFreezeCurve(Long measPointId, String measItemCode, long gmtMeasFreeze) {
        String partKey = Joiner.on('_').skipNulls().join(DateConvertUtils.convertToString(gmtMeasFreeze, DateConvertUtils.PATTERN_YEAR), measPointId);   // 分片键（YYYY_measPointId）
        // 构造主键
        PrimaryKeyBuilder primaryKeyBuilder = PrimaryKeyBuilder.createPrimaryKeyBuilder();
        primaryKeyBuilder.addPrimaryKeyColumn("part_key", PrimaryKeyValue.fromString(partKey));
        primaryKeyBuilder.addPrimaryKeyColumn("meas_point_id", PrimaryKeyValue.fromLong(measPointId));
        primaryKeyBuilder.addPrimaryKeyColumn("meas_item_code", PrimaryKeyValue.fromString(measItemCode));
        primaryKeyBuilder.addPrimaryKeyColumn("gmt_meas_freeze", PrimaryKeyValue.fromLong(gmtMeasFreeze));
        PrimaryKey primaryKey = primaryKeyBuilder.build();
        RowDeleteChange rowDeleteChange = new RowDeleteChange("meas_data_ins_freeze_curve", primaryKey);
        DeleteRowRequest deleteRowRequest = new DeleteRowRequest(rowDeleteChange);
        easeIotTableStoreTemplate.deleteRowSync(deleteRowRequest);
        return 1;
    }

    @Override
    public int batchUpdateMeasDataInsFreezeCurve(Long measPointId, Map<String, MeasDataLatestInfo> measDataLatestInfoMap, long gmtMeasFreeze, int freezeMode) {
        return 0;
    }

    @Override
    public Map<String, MeasDataInsFreezeCurveValue> getMeasDataInsFreezeCurveValueMap(Long measPointId, Set<String> measItemCodeSet, long gmtMeasFreeze, Integer measDataSideType) {
        SqlParamsBuilder sqlParamsBuilder = new SqlParamsBuilder();
        sqlParamsBuilder.append("SELECT gdm.meas_point_id, gdm.self_rate, gdm.ct_rate, gdm.pt_rate ");
        sqlParamsBuilder.append(" FROM iesms.gm_dev_meter gdm ");
        sqlParamsBuilder.append(" WHERE gdm.meas_point_id = ? ", measPointId);
        sqlParamsBuilder.append(" LIMIT 1 ");
        Map<String, Object> gdmMap;
        try {
            gdmMap = easeIotJdbcTemplate.queryForObject(sqlParamsBuilder.getSql(), sqlParamsBuilder.getParams(), mapRowMapper);
        }
        catch(EmptyResultDataAccessException _erdae) {
            gdmMap = null;
        }
        BigDecimal selfRate = gdmMap != null ? (BigDecimal) gdmMap.get("self_rate") : new BigDecimal("1.0");
        if(!(selfRate != null && selfRate.compareTo(BigDecimal.ZERO) > 0)) {
            selfRate = new BigDecimal("1.0");
        }
        BigDecimal ctRate = gdmMap != null ? (BigDecimal) gdmMap.get("ct_rate") : new BigDecimal("1.0");
        if(!(ctRate != null && ctRate.compareTo(BigDecimal.ZERO) > 0)) {
            ctRate = new BigDecimal("1.0");
        }
        BigDecimal ptRate = gdmMap != null ? (BigDecimal) gdmMap.get("pt_rate") : new BigDecimal("1.0");
        if(!(ptRate != null && ptRate.compareTo(BigDecimal.ZERO) > 0)) {
            ptRate = new BigDecimal("1.0");
        }

        Map<String, MeasDataInsFreezeCurveValue> measDataInsFreezeCurveValueMap = Maps.newHashMap();
        List<String> measItemCodeList = Lists.newArrayList(measItemCodeSet);
        List<List<String>> listList = Lists.partition(measItemCodeList, 100);
        for(List<String> list : listList) {
            Set<String> measItemCodeSetTemp = Sets.newHashSet(list);
            MultiRowQueryCriteria multiRowQueryCriteria = new MultiRowQueryCriteria("meas_data_ins_freeze_curve");
            for(String measItemCode : measItemCodeSetTemp) {
                if(StringUtils.isNotBlank(measItemCode)) {
                    String partKey = Joiner.on('_').skipNulls().join(DateConvertUtils.convertToString(gmtMeasFreeze, DateConvertUtils.PATTERN_YEAR), measPointId);   // 分片键（YYYY_measPointId）
                    try {
                        PrimaryKeyBuilder primaryKeyBuilder = PrimaryKeyBuilder.createPrimaryKeyBuilder();
                        primaryKeyBuilder.addPrimaryKeyColumn("part_key", PrimaryKeyValue.fromString(partKey));
                        primaryKeyBuilder.addPrimaryKeyColumn("meas_point_id", PrimaryKeyValue.fromLong(measPointId));
                        primaryKeyBuilder.addPrimaryKeyColumn("meas_item_code", PrimaryKeyValue.fromString(measItemCode));
                        primaryKeyBuilder.addPrimaryKeyColumn("gmt_meas_freeze", PrimaryKeyValue.fromLong(gmtMeasFreeze));
                        PrimaryKey primaryKey = primaryKeyBuilder.build();
                        multiRowQueryCriteria.addRow(primaryKey);
                    }
                    catch(Exception _e) {
                        logger.error(_e.getMessage(), _e.fillInStackTrace());
                    }
                }
            }
            if(multiRowQueryCriteria.isEmpty()) {
                return measDataInsFreezeCurveValueMap;
            }
            // 设置读取最新版本
            multiRowQueryCriteria.setMaxVersions(1);
            // 设置获取列名
            multiRowQueryCriteria.addColumnsToGet(Arrays.asList("meas_data_value", "meas_data_rate", "meas_data_source", "gmt_meas_data", "freeze_mode"));

            BatchGetRowRequest batchGetRowRequest = new BatchGetRowRequest();
            // batchGetRow支持读取多个表的数据, 一个multiRowQueryCriteria对应一个表的查询条件, 可以添加多个multiRowQueryCriteria.
            batchGetRowRequest.addMultiRowQueryCriteria(multiRowQueryCriteria);
            BatchGetRowResponse batchGetRowResponse = null;

            try {
                batchGetRowResponse = easeIotTableStoreTemplate.batchGetRowSync(batchGetRowRequest);
                List<BatchGetRowResponse.RowResult> rowResults = batchGetRowResponse.getBatchGetRowResult("meas_data_ins_freeze_curve");
                if(rowResults != null && rowResults.size() > 0) {
                    for(BatchGetRowResponse.RowResult rowResult : rowResults) {
                        if(rowResult.isSucceed()) {
                            Row row = rowResult.getRow();
                            if(row != null) {
                                MeasDataInsFreezeCurveValue measDataInsFreezeCurveValue = row2MeasDataInsFreezeCurveValue(row, measDataSideType, selfRate, ctRate, ptRate);
                                if(measDataInsFreezeCurveValue != null && StringUtils.isNotBlank(measDataInsFreezeCurveValue.getMeasItemCode())) {
                                    measDataInsFreezeCurveValueMap.put(measDataInsFreezeCurveValue.getMeasItemCode(), measDataInsFreezeCurveValue);
                                }
                            }
                        }
                        else {
                            PrimaryKey primaryKey = batchGetRowRequest.getPrimaryKey(rowResult.getTableName(), rowResult.getIndex());
                            if(primaryKey != null) {
                                // 读一行
                                SingleRowQueryCriteria criteria = new SingleRowQueryCriteria("meas_data_ins_freeze_curve", primaryKey);
                                // 设置读取最新版本
                                criteria.setMaxVersions(1);
                                // 设置获取列名
                                criteria.addColumnsToGet(Arrays.asList("meas_point_id", "meas_item_code", "meas_data_value", "meas_data_rate", "meas_data_source", "gmt_meas_data", "freeze_mode"));
                                //
                                GetRowRequest getRowRequest = new GetRowRequest(criteria);
                                GetRowResponse getRowResponse = easeIotTableStoreTemplate.getRowSync(getRowRequest);
                                Row row = getRowResponse.getRow();
                                if(row != null) {
                                    MeasDataInsFreezeCurveValue measDataInsFreezeCurveValue = row2MeasDataInsFreezeCurveValue(row, measDataSideType, selfRate, ctRate, ptRate);
                                    if(measDataInsFreezeCurveValue != null && StringUtils.isNotBlank(measDataInsFreezeCurveValue.getMeasItemCode())) {
                                        measDataInsFreezeCurveValueMap.put(measDataInsFreezeCurveValue.getMeasItemCode(), measDataInsFreezeCurveValue);
                                    }
                                }
                            }
                        }
                    }
                }
            }
            catch(TableStoreException | ClientException e) {
                logger.error(e.getMessage(), e.fillInStackTrace());
            }
            finally {
                if(batchGetRowResponse != null) {
                    batchGetRowResponse = null;
                }
            }
        }
        return measDataInsFreezeCurveValueMap;
    }

    @Override
    public Map<String, List<MeasDataInsFreezeCurveValue>> getRangeMeasDataInsFreezeCurveValueMap(Long measPointId, Set<String> measItemCodeSet, long gmtMeasFreezeStart, long gmtMeasFreezeEnd, Integer measDataSideType) {
        if(gmtMeasFreezeStart >= gmtMeasFreezeEnd) {
            logger.warn("gmtMeasFreezeStart must be less than gmtMeasFreezeEnd.");
            return Maps.newHashMap();
        }

        int gmtMeasFreezeStartYear = DateUtils.getYear(new Date(gmtMeasFreezeStart));
        int gmtMeasFreezeEndYear = DateUtils.getYear(new Date(gmtMeasFreezeEnd));
        List<Long> gmtMeasFreezeStartList = Lists.newArrayList();
        List<Long> gmtMeasFreezeEndList = Lists.newArrayList();
        if(gmtMeasFreezeStartYear == gmtMeasFreezeEndYear) {
            gmtMeasFreezeStartList.add(gmtMeasFreezeStart);
            gmtMeasFreezeEndList.add(gmtMeasFreezeEnd);
        }
        else if(gmtMeasFreezeStartYear + 1 == gmtMeasFreezeEndYear) {
            gmtMeasFreezeStartList.add(gmtMeasFreezeStart);
            gmtMeasFreezeEndList.add(DateUtils.addMillis(DateConvertUtils.convertFromString(Joiner.on('-').join(DateConvertUtils.convertToString(gmtMeasFreezeEnd, DateConvertUtils.PATTERN_YEAR), "01", "01"), DateConvertUtils.PATTERNS_DATE), -1).getTime());
            gmtMeasFreezeStartList.add(DateConvertUtils.convertFromString(Joiner.on('-').join(DateConvertUtils.convertToString(gmtMeasFreezeEnd, DateConvertUtils.PATTERN_YEAR), "01", "01"), DateConvertUtils.PATTERNS_DATE).getTime());
            gmtMeasFreezeEndList.add(gmtMeasFreezeEnd);
        }
        else {
            logger.warn("gmtMeasFreezeStart ~ gmtMeasFreezeEnd is not beyond two years.");
            return Maps.newHashMap();
        }

        SqlParamsBuilder sqlParamsBuilder = new SqlParamsBuilder();
        sqlParamsBuilder.append("SELECT gdm.meas_point_id, gdm.self_rate, gdm.ct_rate, gdm.pt_rate ");
        sqlParamsBuilder.append(" FROM iesms.gm_dev_meter gdm ");
        sqlParamsBuilder.append(" WHERE gdm.meas_point_id = ? ", measPointId);
        sqlParamsBuilder.append(" LIMIT 1 ");
        Map<String, Object> gdmMap;
        try {
            gdmMap = easeIotJdbcTemplate.queryForObject(sqlParamsBuilder.getSql(), sqlParamsBuilder.getParams(), mapRowMapper);
        }
        catch(EmptyResultDataAccessException _erdae) {
            gdmMap = null;
        }
        BigDecimal selfRate = gdmMap != null ? (BigDecimal) gdmMap.get("self_rate") : new BigDecimal("1.0");
        if(!(selfRate != null && selfRate.compareTo(BigDecimal.ZERO) > 0)) {
            selfRate = new BigDecimal("1.0");
        }
        BigDecimal ctRate = gdmMap != null ? (BigDecimal) gdmMap.get("ct_rate") : new BigDecimal("1.0");
        if(!(ctRate != null && ctRate.compareTo(BigDecimal.ZERO) > 0)) {
            ctRate = new BigDecimal("1.0");
        }
        BigDecimal ptRate = gdmMap != null ? (BigDecimal) gdmMap.get("pt_rate") : new BigDecimal("1.0");
        if(!(ptRate != null && ptRate.compareTo(BigDecimal.ZERO) > 0)) {
            ptRate = new BigDecimal("1.0");
        }

        Map<String, List<MeasDataInsFreezeCurveValue>> rangeMeasDataInsFreezeCurveValueMap = Maps.newHashMap();
        for(String measItemCode : measItemCodeSet) {
            if(StringUtils.isNotBlank(measItemCode)) {
                List<MeasDataInsFreezeCurveValue> results = Lists.newArrayList();
                for(int i = 0; i < gmtMeasFreezeStartList.size(); i++) {
                    results.addAll(getMeasDataInsFreezeCurveValueList(measPointId, measItemCode, gmtMeasFreezeStartList.get(i), gmtMeasFreezeEndList.get(i), measDataSideType, selfRate, ctRate, ptRate));
                }
                rangeMeasDataInsFreezeCurveValueMap.put(measItemCode, results);
            }
        }
        return rangeMeasDataInsFreezeCurveValueMap;
    }

    private List<MeasDataInsFreezeCurveValue> getMeasDataInsFreezeCurveValueList(Long measPointId, String measItemCode, long gmtMeasFreezeStart, long gmtMeasFreezeEnd, Integer measDataSideType, BigDecimal selfRate, BigDecimal ctRate, BigDecimal ptRate) {
        int gmtMeasFreezeStartYear = DateUtils.getYear(new Date(gmtMeasFreezeStart));
        int gmtMeasFreezeEndYear = DateUtils.getYear(new Date(gmtMeasFreezeEnd));
        if(gmtMeasFreezeStartYear != gmtMeasFreezeEndYear) {
            logger.warn("gmtMeasFreezeStart ~ gmtMeasFreezeEnd must be the same year.");
            return Lists.newArrayList();
        }
        String startPartKey = Joiner.on('_').skipNulls().join(DateConvertUtils.convertToString(gmtMeasFreezeStart, DateConvertUtils.PATTERN_YEAR), measPointId);   // 分片键（YYYY_measPointId）
        String endPartKey = Joiner.on('_').skipNulls().join(DateConvertUtils.convertToString(gmtMeasFreezeEnd, DateConvertUtils.PATTERN_YEAR), measPointId);        // 分片键（YYYY_measPointId）
        RangeRowQueryCriteria rangeRowQueryCriteria = new RangeRowQueryCriteria("meas_data_ins_freeze_curve");
        // 设置读取最新版本
        rangeRowQueryCriteria.setMaxVersions(1);
        // 设置起始主键
        rangeRowQueryCriteria.setInclusiveStartPrimaryKey(createMeasDataInsFreezeCurvePrimaryKey(startPartKey, measPointId, measItemCode, gmtMeasFreezeStart));
        // 设置结束主键
        rangeRowQueryCriteria.setExclusiveEndPrimaryKey(createMeasDataInsFreezeCurvePrimaryKey(endPartKey, measPointId, measItemCode, gmtMeasFreezeEnd));
        // 设置获取列名
        rangeRowQueryCriteria.addColumnsToGet(Arrays.asList("meas_data_value", "meas_data_rate", "meas_data_source", "gmt_meas_data", "freeze_mode"));
        List<MeasDataInsFreezeCurveValue> results = Lists.newArrayList();
        GetRangeResponse getRangeResponse = null;
        try {
            while(true) {
                getRangeResponse = easeIotTableStoreTemplate.getRangeSync(new GetRangeRequest(rangeRowQueryCriteria));
                for(Row row : getRangeResponse.getRows()) {
                    results.add(row2MeasDataInsFreezeCurveValue(row, measDataSideType, selfRate, ctRate, ptRate));
                }
                // 若nextStartPrimaryKey不为null, 则继续读取.
                if(getRangeResponse.getNextStartPrimaryKey() != null) {
                    rangeRowQueryCriteria.setInclusiveStartPrimaryKey(getRangeResponse.getNextStartPrimaryKey());
                }
                else {
                    break;
                }
            }
        }
        catch(TableStoreException | ClientException e) {
            logger.error(e.getMessage(), e.fillInStackTrace());
        }
        finally {
            if(getRangeResponse != null) {
                getRangeResponse = null;
            }
        }
        return results;
    }

    private PrimaryKey createMeasDataInsFreezeCurvePrimaryKey(String partKey, Long measPointId, String measItemCode, long gmtMeasFreeze) {
        PrimaryKeyBuilder primaryKeyBuilder = PrimaryKeyBuilder.createPrimaryKeyBuilder();
        primaryKeyBuilder.addPrimaryKeyColumn("part_key", PrimaryKeyValue.fromString(partKey));    // 分片键
        primaryKeyBuilder.addPrimaryKeyColumn("meas_point_id", PrimaryKeyValue.fromLong(measPointId));
        primaryKeyBuilder.addPrimaryKeyColumn("meas_item_code", PrimaryKeyValue.fromString(measItemCode));
        primaryKeyBuilder.addPrimaryKeyColumn("gmt_meas_freeze", PrimaryKeyValue.fromLong(gmtMeasFreeze));
        return primaryKeyBuilder.build();
    }

    private MeasDataInsFreezeCurveValue row2MeasDataInsFreezeCurveValue(Row row, Integer measDataSideType, BigDecimal selfRate, BigDecimal ctRate, BigDecimal ptRate) {
        MeasDataInsFreezeCurveValue measDataInsFreezeCurveValue = null;
        if(row != null) {
            PrimaryKey primaryKey = row.getPrimaryKey();
            if(primaryKey != null) {
                PrimaryKeyColumn measPointIdCol = primaryKey.getPrimaryKeyColumn("meas_point_id");
                PrimaryKeyColumn measItemCodeCol = primaryKey.getPrimaryKeyColumn("meas_item_code");
                PrimaryKeyColumn gmtMeasFreezeCol = primaryKey.getPrimaryKeyColumn("gmt_meas_freeze");
                Long measPointId = measPointIdCol.getValue().asLong();
                String measItemCode = measItemCodeCol.getValue().asString();
                long gmtMeasFreeze = gmtMeasFreezeCol.getValue().asLong();
                measDataInsFreezeCurveValue = new MeasDataInsFreezeCurveValue();
                measDataInsFreezeCurveValue.setMeasPointId(measPointId);
                measDataInsFreezeCurveValue.setMeasItemCode(measItemCode);
                measDataInsFreezeCurveValue.setGmtMeasFreeze(gmtMeasFreeze);

                Column measDataValueCol = row.getLatestColumn("meas_data_value");
                if(measDataValueCol != null) {
                    BigDecimal measDataValue = new BigDecimal(String.valueOf(measDataValueCol.getValue().asDouble())).setScale(8, BigDecimal.ROUND_HALF_EVEN);
                    if(measDataSideType != null && measDataSideType == 1) {
                        //measDataValue = measDataValue.multiply(new BigDecimal("2.0")).setScale(6, BigDecimal.ROUND_HALF_UP);
                        if(StringUtils.containsAny(measItemCode, "gen_watt_ua", "gen_watt_ub", "gen_watt_uc", "gen_watt_upv", "gen_watt_ur", "gen_watt_ubus")) {
                            // 电压 * PT
                            measDataValue = measDataValue.multiply(selfRate).multiply(ptRate).setScale(6, BigDecimal.ROUND_HALF_UP);
                        }
                        else if(StringUtils.containsAny(measItemCode, "gen_watt_ia", "gen_watt_ib", "gen_watt_ic", "gen_watt_in", "gen_watt_ipv", "gen_watt_ir", "gen_watt_ibus")) {
                            // 电流 * CT
                            measDataValue = measDataValue.multiply(selfRate).multiply(ctRate).setScale(6, BigDecimal.ROUND_HALF_UP);
                        }
                        else if(StringUtils.containsAny(measItemCode, "gen_watt_acp", "gen_watt_rep", "gen_watt_app", "gen_watt_ppv", "gen_watt_pr", "gen_watt_pbus")) {
                            // 功率 * CT * PT
                            measDataValue = measDataValue.multiply(selfRate).multiply(ctRate).multiply(ptRate).setScale(6, BigDecimal.ROUND_HALF_UP);
                        }
                    }
                    measDataInsFreezeCurveValue.setMeasDataValue(measDataValue);
                    measDataInsFreezeCurveValue.setGmtCreate(measDataValueCol.getTimestamp());
                    measDataInsFreezeCurveValue.setGmtModified(measDataValueCol.getTimestamp());
                }
                Column measDataRateCol = row.getLatestColumn("meas_data_rate");
                if(measDataRateCol != null) {
                    measDataInsFreezeCurveValue.setMeasDataRate(new BigDecimal(String.valueOf(measDataRateCol.getValue().asDouble())).setScale(8, BigDecimal.ROUND_HALF_EVEN));
                }
                Column measDataSourceCol = row.getLatestColumn("meas_data_source");
                if(measDataSourceCol != null) {
                    measDataInsFreezeCurveValue.setMeasDataSource(NumberUtils.toInt(String.valueOf(measDataSourceCol.getValue().asLong()), 0));
                }
                Column gmtMeasDataCol = row.getLatestColumn("gmt_meas_data");
                if((gmtMeasDataCol != null)) {
                    long gmtMeasData = gmtMeasDataCol.getValue().asLong();
                    measDataInsFreezeCurveValue.setGmtMeasDataReading(gmtMeasData);
                }
                Column freezeModeCol = row.getLatestColumn("freeze_mode");
                if((freezeModeCol != null)) {
                    int freezeMode = NumberUtils.toInt(String.valueOf(freezeModeCol.getValue().asLong()), 0);
                    measDataInsFreezeCurveValue.setFreezeMode(freezeMode);
                }
            }
        }
        return measDataInsFreezeCurveValue;
    }

}
