package com.easesource.iot.springbootapps.gateway.socket.iec104.receiver;

import com.easesource.commons.util.convert.JsonConvertUtils;
import com.easesource.iot.protoparser.iec104.writer.PostmanHandler;
import com.rabbitmq.client.Channel;
import org.apache.dubbo.common.utils.Assert;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.amqp.core.Message;
import org.springframework.amqp.rabbit.annotation.RabbitHandler;
import org.springframework.amqp.rabbit.annotation.RabbitListener;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import java.io.IOException;

import static com.easesource.iot.gateway.base.model.MqConstants.*;


@Component
public class RabbitmqToNettyReceiver {
    private static final Logger LOGGER = LoggerFactory.getLogger(RabbitmqToNettyReceiver.class);

    @Autowired
    private PostmanHandler postmanHandler;

    @RabbitHandler
    @RabbitListener(queues = DIRECT_QUEUE3)
    public void directMessage(Message message, Channel channel) throws IOException {

        try {
            Assert.notNull(message, "sendMessage 消息体不能为NULL");
            // prefetchCount限制每个消费者在收到下一个确认回执前一次可以最大接受多少条消息,通过basic.qos方法设置prefetch_count=1,这样RabbitMQ就会使得每个Consumer在同一个时间点最多处理一个Message
            channel.basicQos(1);
            LOGGER.info("DirectConsumer {} directMessage :" + message);

            String iec = new String(message.getBody(), "UTF-8");
            com.easesource.iot.protoparser.iec104.model.Message msg = JsonConvertUtils.convertFromString(iec, com.easesource.iot.protoparser.iec104.model.Message.class);
            //将数据保存到缓存中，通过不断轮询，将数据发送出去
            postmanHandler.handler(msg);
            // 确认消息已经消费成功
            channel.basicAck(message.getMessageProperties().getDeliveryTag(), false);
        } catch (IOException e) {
            e.printStackTrace();
            if (message.getMessageProperties().getRedelivered()) {
                LOGGER.info("消息已重复处理失败,拒绝再次接收...");
                // deliveryTag：消息传送的次数,发布的每一条消息都会获得一个唯一的deliveryTag，deliveryTag在channel范围内是唯一的
                // multiple：批量确认标志。如果值为true，包含本条消息在内的、所有比该消息deliveryTag值小的 消息都被拒绝了（除了已经被 ack 的以外）;如果值为false，只拒绝三本条消息
                // requeue：如果值为true，则重新放入RabbitMQ的发送队列，如果值为false，则通知RabbitMQ销毁这条消息
                channel.basicReject(message.getMessageProperties().getDeliveryTag(), true);
            } else {
                LOGGER.info("消息即将再次返回队列处理...");
                // deliveryTag：消息传送的次数,发布的每一条消息都会获得一个唯一的deliveryTag，deliveryTag在channel范围内是唯一的
                // requeue：如果值为true，则重新放入RabbitMQ的发送队列，如果值为false，则通知RabbitMQ销毁这条消息
                channel.basicNack(message.getMessageProperties().getDeliveryTag(), false, true);
            }
        }
    }
}
