package com.easesource.iot.protoparser.gaeadcu.handler.codec.decoder;


import com.easesource.iot.protoparser.base.utils.HexDump;
import com.easesource.iot.protoparser.cjt188.v2004.mbus.MBus;
import com.easesource.iot.protoparser.gaeadcu.message.asdu.DataCommand;
import com.easesource.iot.protoparser.gaeadcu.message.asdu.DataItem;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufUtil;
import io.netty.buffer.Unpooled;
import io.netty.util.ReferenceCountUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import java.util.Calendar;
import java.util.Date;
import static com.easesource.iot.protoparser.gaeadcu.constant.Constants.*;


abstract class BaseDataItemDecoder {
    protected final Logger log = LoggerFactory.getLogger(IDataItemDecoder.class);

    abstract void decode(ByteBuf buf, DataCommand hc) throws Exception;

    protected String parseDataTimeStr(ByteBuf buf) {
        int milliseconds = buf.order(BYTE_ORDER).readUnsignedShort();
        int millis = milliseconds % 1000;
        int seconds = (milliseconds - millis) / 1000;

        int minute = buf.readByte() & 0x3F;
        minute = minute > 59 ? 59 : minute;

        int hour = buf.readByte() & 0x1F;
        hour = hour > 23 ? 23 : hour;

        int day = buf.readByte() & 0x1F;

        int month = buf.readByte() & 0x0F;
        month = month > 12 ? 12 : month;

        int year = 2000 + (buf.readByte() & 0x7F);

        StringBuilder builder = new StringBuilder();
        builder.append(year).append("-");
        builder.append(month < 10 ? "0" : "").append(month).append("-");
        builder.append(day < 10 ? "0" : "").append(day).append(" ");
        builder.append(hour < 10 ? "0" : "").append(hour).append(":");
        builder.append(minute < 10 ? "0" : "").append(minute).append(":");
        builder.append(seconds < 10 ? "0" : "").append(seconds).append(".");
        builder.append(millis);
        return builder.toString();
    }

    // 按字节倒置
    protected String toLittleEndianHex(ByteBuf buf) {
        int bytes = buf.readableBytes();
        String hex = "";
        for (int i = 0; i < bytes; i++) {
            hex = HexDump.toHex(buf.readByte()) + hex;
        }
        return hex;
    }

    protected String toAsciiString(ByteBuf buf) {
        int bytes = buf.readableBytes();
        char[] chrList = new char[bytes];
        for (int i = 0; i < bytes; i++) {
            chrList[i] = (char) buf.readUnsignedByte();
        }
        return (new String(chrList)).trim();
    }

    protected void setFailedDataItem(DataCommand command) {
        DataItem item = new DataItem();
        item.setCode(HexDump.toHex(command.getType()));
        item.setValue("");
        item.setStatus(ITEM_STATUS_FAILED);
        command.addItem(item);
        command.setStatus(STATUS_RTU_FAILED);
    }

    /**
     * 针对f9透明转发命令 数据组装
     * @param command
     * @param value
     * @return
     */
    protected DataItem setF9SuccessDataItem(DataCommand command, MBus mBus, String value, ByteBuf param){
        DataItem item = new DataItem();
        item.setCode(HexDump.toHex(command.getType()));
        item.setValue(value);
        item.setPortName(param.readByte());
        item.setBaudRate(param.readByte());
        item.setParity(param.readByte());
        item.setStopBites(param.readByte());
        item.setCot(command.getCot());
        item.setStatus(ITEM_STATUS_SUCCESS);
        item.setmBus(mBus);
        command.addItem(item);
        command.setStatus(STATUS_SUCCESS);
        return item;
    }
    protected DataItem setSuccessDataItem(DataCommand command, String value) {
        return setSuccessDataItem(command, value, null, null, null);
    }

    protected DataItem setSuccessDataItem(DataCommand command, String value, String tn, String channel) {
        return setSuccessDataItem(command, value, tn, channel, null);
    }

    protected DataItem setSuccessDataItem(DataCommand command, String value, String tn, String channel, Date time) {
        DataItem item = new DataItem();
        item.setCode(HexDump.toHex(command.getType()));
        item.setTn(tn);
        item.setChannel(channel);
        item.setValue(value);
        item.setTime(time);
        item.setCot(command.getCot());
        item.setStatus(ITEM_STATUS_SUCCESS);
        command.addItem(item);
        command.setStatus(STATUS_SUCCESS);
        return item;
    }

    protected Date parseDataTime(ByteBuf buf) {
        int milliseconds = buf.order(BYTE_ORDER).readUnsignedShort();
        int millis = milliseconds % 1000;
        int seconds = (milliseconds - millis) / 1000;

        int minute = buf.readByte() & 0x3F;
        minute = minute > 59 ? 59 : minute;

        int hour = buf.readByte() & 0x1F;
        hour = hour > 23 ? 23 : hour;

        int day = buf.readByte() & 0x1F;

        int month = buf.readByte() & 0x0F;
        month = month > 12 ? 12 : month;

        int year = 2000 + (buf.readByte() & 0x7F);

        Calendar cal = Calendar.getInstance();
        // Calendar的月份从0开始，0是一月份
        cal.set(year, month - 1, day, hour, minute, seconds);
        cal.set(Calendar.MILLISECOND, millis);
        return cal.getTime();
    }

    protected void explainDIItem(ByteBuf buf, int count, boolean withTime, DataCommand command, String sMeterAddr, Date time, String code) {
        DataItem result;
        // 4字节点号+1字节读数
        for (int i = 0; i < count; i++) {
            result = new DataItem();
            ByteBuf stnBuf = null;
            ByteBuf timeBuf = null;
            try {
                result.setCot(command.getCot());
                result.setCode(code);
                result.setType("YX");
                stnBuf = buf.readBytes(4);
                String sTn = toLittleEndianHex(stnBuf).toUpperCase();
                result.setTn(sTn);
                result.setValue(HexDump.toHex(buf.readByte()));
                if (withTime) {
                    timeBuf = buf.readBytes(7);
                    result.setTime(parseDataTime(timeBuf));
                } else {
                    result.setTime(time);
                }
                result.setChannel(sMeterAddr);
                result.setStatus(ITEM_STATUS_SUCCESS);
                command.addItem(result);
            } catch (Exception e) {
                e.printStackTrace();
            }finally {
                ReferenceCountUtil.release(stnBuf);
                ReferenceCountUtil.release(timeBuf);
            }
        }
    }

    protected void explainAIItem(ByteBuf buf, int count, boolean withTime, DataCommand command, String sMeterAddr, Date time, String code) {
        DataItem result;
        // 4字节点号+4字节读数
        for (int i = 0; i < count; i++) {
            result = new DataItem();
            ByteBuf stnBuf = null;
            ByteBuf timeBuf = null;
            try {
                result.setCot(command.getCot());
                result.setCode(code);
                result.setType("YC");
                stnBuf = buf.readBytes(4);
                String sTn = toLittleEndianHex(stnBuf).toUpperCase();
                result.setTn(sTn);
                float value = buf.order(BYTE_ORDER).readFloat();
                result.setValue(Float.toString(value));
                if (withTime) {
                    timeBuf = buf.readBytes(7);
                    result.setTime(parseDataTime(timeBuf));
                } else {
                    result.setTime(time);
                }
                result.setChannel(sMeterAddr);
                result.setStatus(ITEM_STATUS_SUCCESS);
                command.addItem(result);
            } catch (Exception e) {
                e.printStackTrace();
            }finally {
                ReferenceCountUtil.release(stnBuf);
                ReferenceCountUtil.release(timeBuf);
            }
        }
    }

    protected void explainPIItem(ByteBuf buf, int count, boolean withTime, DataCommand command, String sMeterAddr, Date time, String code) {
        DataItem result;
        // 4字节点号+8字节读数
        for (int i = 0; i < count; i++) {
            result = new DataItem();
            ByteBuf stnBuf = null;
            ByteBuf timeBuf = null;
            try {
                result.setCot(command.getCot());
                result.setCode(code);
                result.setType("LJ");
                stnBuf = buf.readBytes(4);
                String sTn = toLittleEndianHex(stnBuf).toUpperCase();
                result.setTn(sTn);
                double value = buf.order(BYTE_ORDER).readDouble();
                if (value > MAX_LJL_VAL) {
                    log.error("DCU：{}，累积量数据大小越界,value:{}，hex:{}", HexDump.toHex(command.getRtuId()), value, toHex(value));
                }
                result.setValue(Double.toString(value));
                if (withTime) {
                    timeBuf = buf.readBytes(7);
                    result.setTime(parseDataTime(timeBuf));
                } else {
                    result.setTime(time);
                }
                result.setChannel(sMeterAddr);
                result.setStatus(ITEM_STATUS_SUCCESS);
                command.addItem(result);
            } catch (Exception e) {
                log.error("ExplainItem Error:" + e.toString());
            }finally {
                ReferenceCountUtil.release(stnBuf);
                ReferenceCountUtil.release(timeBuf);
            }
        }
    }

    protected String toHex(double value) {
        ByteBuf buf = Unpooled.buffer();
        buf.order(BYTE_ORDER);
        buf.writeDouble(value);
        return ByteBufUtil.hexDump(buf).toUpperCase();
    }
}
