package com.gaea.iesms.bm.archive.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.lang.generator.SnowflakeGenerator;
import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.gaea.base.core.R;
import com.gaea.base.error.BizException;
import com.gaea.iesms.bm.archive.mapper.CeCustMapper;
import com.gaea.iesms.bm.archive.model.response.dto.ceCust.CeCustDto;
import com.gaea.iesms.bm.archive.service.CeCustService;
import com.gaea.iesms.bm.archive.service.CePointMeterService;
import com.gaea.iesms.bm.archive.service.CePointService;
import com.gaea.iesms.bm.archive.service.CeResourceService;
import com.gaea.iesms.core.feign.request.archive.AddCeCustRequest;
import com.gaea.iesms.core.feign.request.archive.QueryUserRequest;
import com.gaea.iesms.core.feign.request.archive.UpdateCeCustRequest;
import com.gaea.iesms.core.feign.request.archive.UserPageRequest;
import com.gaea.iesms.core.feign.response.archive.CeCustVo;
import com.gaea.iesms.core.feign.service.RemoteIotService;
import com.gaea.iesms.core.model.entity.archive.CeCust;
import com.gaea.iesms.core.model.entity.archive.CePoint;
import com.gaea.iesms.core.model.entity.archive.CePointMeter;
import com.gaea.iesms.core.model.entity.archive.CeResource;
import com.gaea.iesms.core.model.entity.iot.GmDevMeter;
import com.gaea.iesms.core.model.jsonobject.archive.CeCustPropsJsonObject;
import com.gaea.iesms.core.model.jsonobject.archive.CePointPropsJsonObject;
import com.gaea.spring.cloud.starter.util.AuthUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 针对表【ce_cust(用能客户 用能资源子表)】的数据库操作Service实现
 *
 * @author JIE
 */
@Slf4j
@Service
public class CeCustServiceImpl extends ServiceImpl<CeCustMapper, CeCust>
        implements CeCustService {

    @Resource
    private RemoteIotService remoteIotService;

    @Resource
    private CeCustMapper ceCustMapper;

    @Resource
    private CeResourceService ceResourceService;

    @Resource
    private CePointService cePointService;

    @Resource
    private CePointMeterService cePointMeterService;

    private static final SnowflakeGenerator SNOWFLAKE_GENERATOR = new SnowflakeGenerator();

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean addCeCust(AddCeCustRequest addUserRequest) {
        CeCust ceCust = new CeCust();
        BeanUtils.copyProperties(addUserRequest, ceCust);

        CeResource ceResource = new CeResource();
        BeanUtils.copyProperties(addUserRequest, ceResource);
        ceResource.setCeResClass(1);
        ceResource.setCeResStatus("20");

        String orgNo = addUserRequest.getOrgNo();
        if (StrUtil.isBlank(orgNo)) {
            orgNo = AuthUtils.getOrgNo();
        }

        // 增加用能资源表
        Long next = SNOWFLAKE_GENERATOR.next();
        ceResource.setId(next);
        ceResource.setOrgNo(orgNo);
        ceResourceService.addCeResource(ceResource, null);

        // 新增用能点
        CePoint cePoint = new CePoint();
        cePoint.setOrgNo(orgNo);
        cePoint.setCeCustId(next);
        cePoint.setCeResId(next);
        cePoint.setCeResClass(1);
        cePoint.setCePointLevel(1);
        cePoint.setCePointNo(addUserRequest.getCeResNo());
        cePoint.setCePointName(ceResource.getCeResName());
        cePoint.setIsSettlement(0);
        cePoint.setCePointType("");
        cePoint.setParentId(0L);
        cePoint.setCePointProps(new CePointPropsJsonObject());

        CeCustPropsJsonObject ceCustProps = addUserRequest.getCeCustProps();
        if (ceCustProps != null) {
            List<String> meterList = ceCustProps.getMeterList();
            if (CollectionUtil.isNotEmpty(meterList)) {
                cePoint.setMeasPointIdList(meterList);

                if (CollectionUtil.isNotEmpty(meterList)) {
                    List<CePointMeter> cePointMeters = new ArrayList<>();
                    for (String aLong : meterList) {
                        CePointMeter cePointMeter = new CePointMeter();
                        cePointMeter.setCePointId(cePoint.getId());
                        cePointMeter.setDevMeterId(Long.valueOf(aLong));
                        cePointMeters.add(cePointMeter);
                    }
                    cePointMeterService.saveBatch(cePointMeters);
                }
            }
        }
        cePointService.save(cePoint);
        // 增加用户档案表
        ceCust.setId(ceResource.getId());
        return save(ceCust);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean addCeCustBatch(List<AddCeCustRequest> addUserRequest) {
        if (CollectionUtil.isEmpty(addUserRequest)) {
            return false;
        }
        boolean flag = false;
        for (AddCeCustRequest addCeCustRequest : addUserRequest) {
            flag = addCeCust(addCeCustRequest);
        }
        return flag;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public List<String> addVppCeCustBatch(List<AddCeCustRequest> addUserRequest) {
        List<String> list = new ArrayList<>();
        for (AddCeCustRequest addCeCustRequest : addUserRequest) {
            processCeCustRequest(addCeCustRequest, list);
        }
        return list;
    }

    /**
     * 新增客户信息
     *
     * @param addCeCustRequest 新增客户请求
     * @param list             添加成功的客户ID列表
     * @author JIE
     */
    private void processCeCustRequest(AddCeCustRequest addCeCustRequest, List<String> list) {
        // 创建客户对象，并从请求对象中复制属性
        CeCust ceCust = new CeCust();
        BeanUtils.copyProperties(addCeCustRequest, ceCust);

        // 根据添加客户的请求创建资源对象
        CeResource ceResource = createCeResource(addCeCustRequest);
        // 添加资源到系统
        ceResourceService.addVppCeResource(ceResource);

        // 创建点对象，并关联资源
        CePoint cePoint = createCePoint(addCeCustRequest, ceResource);
        // 保存点的计量信息
        saveCePointMeters(addCeCustRequest, cePoint);

        // 保存点信息到系统
        cePointService.save(cePoint);

        // 将资源ID设置给客户对象，表示客户和资源的关联
        ceCust.setId(ceResource.getId());
        // 保存客户信息到系统，并检查保存是否成功
        boolean save = save(ceCust);
        // 如果保存成功，将客户ID添加到列表中
        if (save) {
            list.add(ceCust.getId().toString());
        }
    }

    /**
     * 根据添加客户请求创建用能资源对象
     *
     * @param addCeCustRequest 添加客户的请求对象，包含所需属性。
     * @return 创建的CE资源对象。
     */
    private CeResource createCeResource(AddCeCustRequest addCeCustRequest) {
        // 初始化资源对象
        CeResource ceResource = new CeResource();
        // 复制请求对象的属性到CE资源对象
        BeanUtils.copyProperties(addCeCustRequest, ceResource);
        // 设置资源的类别和状态
        ceResource.setCeResClass(1);
        ceResource.setCeResStatus("20");

        // 获取组织编号，如果请求中没有提供，则使用认证信息中的组织编号
        String orgNo = addCeCustRequest.getOrgNo();
        if (StrUtil.isBlank(orgNo)) {
            orgNo = AuthUtils.getOrgNo();
        }

        Long id = addCeCustRequest.getId();
        if (id == null) {
            // 生成资源的唯一ID
            Long next = SNOWFLAKE_GENERATOR.next();
            ceResource.setId(next);
        }else {
            ceResource.setId(id);
        }

        // 设置资源的组织编号
        ceResource.setOrgNo(orgNo);
        return ceResource;
    }

    /**
     * 根据添加客户请求和资源信息创建量测点
     *
     * @param addCeCustRequest 添加客户请求对象，包含客户资源编号等信息。
     * @param ceResource       客户资源对象，包含资源编号、资源名称等信息。
     * @return 返回创建的客户积分点对象。
     */
    private CePoint createCePoint(AddCeCustRequest addCeCustRequest, CeResource ceResource) {
        CePoint cePoint = new CePoint();
        cePoint.setOrgNo(ceResource.getOrgNo());
        cePoint.setCeCustId(ceResource.getId());
        cePoint.setCeResId(ceResource.getId());
        cePoint.setCeResClass(1);
        cePoint.setCePointLevel(1);
        cePoint.setCePointNo(addCeCustRequest.getCeResNo());
        cePoint.setCePointName(ceResource.getCeResName());
        cePoint.setIsSettlement(0);
        cePoint.setCePointType("");
        cePoint.setParentId(0L);
        cePoint.setCePointProps(new CePointPropsJsonObject());

        return cePoint;
    }

    /**
     * 保存量测点与表计的关系
     *
     * @param addCeCustRequest 添加客户的请求对象，包含客户属性。
     * @param cePoint          量测点信息
     */
    private void saveCePointMeters(AddCeCustRequest addCeCustRequest, CePoint cePoint) {
        // 从添加客户请求中获取客户属性
        CeCustPropsJsonObject ceCustProps = addCeCustRequest.getCeCustProps();
        // 检查客户属性是否存在
        if (ceCustProps != null) {
            // 获取客户属性中的计量点列表
            List<String> meterList = ceCustProps.getMeterList();
            // 检查计量点列表是否为空
            if (CollectionUtil.isNotEmpty(meterList)) {
                // 将计量点列表设置到客户热点对象中
                cePoint.setMeasPointIdList(meterList);
                // 使用流处理每个计量点，创建并保存客户热点与计量点的关联信息
                List<CePointMeter> cePointMeters = meterList.stream()
                        .map(meterId -> createCePointMeter(cePoint.getId(), meterId))
                        .collect(Collectors.toList());
                // 批量保存客户热点与计量点的关联信息
                cePointMeterService.saveBatch(cePointMeters);
            }
        }
    }

    /**
     * 创建一个用能点量测表计关系对象
     * <p>
     *
     * @param cePointId 量测点id
     * @param meterId   表计id
     * @return 返回创建的用能点量测表计关系对象
     */
    private CePointMeter createCePointMeter(Long cePointId, String meterId) {
        // 创建一个新的CE点计数器实例
        CePointMeter cePointMeter = new CePointMeter();
        // 设置CE点的ID
        cePointMeter.setCePointId(cePointId);
        // 将计数器ID字符串转换为长整型并设置到CE点计数器对象中
        cePointMeter.setDevMeterId(Long.valueOf(meterId));
        // 返回创建的CE点计数器对象
        return cePointMeter;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean updateUserProfile(UpdateCeCustRequest ceCustRequest) {
        // 验证id是否为空
        if (StrUtil.isBlank(ceCustRequest.getId())) {
            throw BizException.of("id不能为空");
        }

        // 复制请求信息到用户档案对象并更新
        CeCust ceCust = new CeCust();
        BeanUtils.copyProperties(ceCustRequest, ceCust);
        ceCust.setId(Long.valueOf(ceCustRequest.getId()));
        updateById(ceCust);

        // 复制请求信息到用能资源对象并更新
        CeResource ceResource = new CeResource();
        BeanUtils.copyProperties(ceCustRequest, ceResource);
        ceResource.setId(Long.valueOf(ceCustRequest.getId()));
        ceResourceService.updateCeResource(ceResource, null);

        // 根据id查询对应的用能点
        LambdaQueryWrapper<CePoint> pointQueryWrapper = new LambdaQueryWrapper<>();
        pointQueryWrapper
                .eq(CePoint::getCeResId, ceCustRequest.getId());
        CePoint cePoint = cePointService.getOne(pointQueryWrapper);

        // 更新或设置用能点的量测表计信息
        CeCustPropsJsonObject ceCustProps = ceCustRequest.getCeCustProps();
        if (ceCustProps != null) {
            List<String> meterList = ceCustProps.getMeterList();
            if (CollectionUtil.isNotEmpty(meterList)) {
                cePoint.setMeasPointIdList(meterList);

                // 批量保存新的量测表计关系
                if (CollectionUtil.isNotEmpty(meterList)) {
                    List<CePointMeter> cePointMeters = new ArrayList<>();
                    for (String aLong : meterList) {
                        CePointMeter cePointMeter = new CePointMeter();
                        cePointMeter.setCePointId(cePoint.getId());
                        cePointMeter.setDevMeterId(Long.valueOf(aLong));
                        cePointMeters.add(cePointMeter);
                    }
                    cePointMeterService.saveBatch(cePointMeters);
                }
            }
        }

        // 更新用能点的编号和名称
        String ceResNo = ceCustRequest.getCeResNo();
        String ceResName = ceResource.getCeResName();
        if (StrUtil.isNotEmpty(ceResNo)) {
            cePoint.setCePointNo(ceResNo);
        }
        if (StrUtil.isNotEmpty(ceResName)) {
            cePoint.setCePointName(ceResName);
        }
        if (cePoint != null) {
            cePointService.updateById(cePoint);
            // 删除并重新添加用能点与量测表计的关系
            cePointMeterService.removeByCePointIds(Collections.singletonList(cePoint.getId()));
        }
        return true;
    }

    @Override
    public CeCustVo queryUserFileAccordConduit(QueryUserRequest queryUserRequest) {
        CeCustDto ceCustDto = ceCustMapper.queryUserFileAccordConduit(queryUserRequest);

        CeCustVo ceCustVo = new CeCustVo();
        if (ceCustDto != null) {
            // 获取并网关联表记
            CeCustPropsJsonObject ceCustProps1 = ceCustDto.getCeCustProps();
            if (ceCustProps1 != null) {
                List<String> meterList = ceCustProps1.getMeterList();
                if (CollectionUtil.isNotEmpty(meterList)) {
                    // 调用iot接口获取表记信息
                    R<List<GmDevMeter>> devMeterIdList = remoteIotService.getDevMeterIdList(meterList);
                    log.info("调用iot接口获取表记信息:{}", devMeterIdList.getCode());
                    if (CollectionUtil.isNotEmpty(devMeterIdList.getData())) {
                        List<GmDevMeter> data = devMeterIdList.getData();
                        List<String> meterNames = data
                                .stream()
                                .map(GmDevMeter::getDevMeterNo)
                                .collect(Collectors.toList());
                        CeCustPropsJsonObject ceCustProps = ceCustDto.getCeCustProps();
                        ceCustProps.setMeterListName(meterNames);
                    }
                }
            }
            BeanUtils.copyProperties(ceCustDto, ceCustVo);
        }
        return ceCustVo;
    }

    @Override
    public IPage<CeCustDto> listPagedUserProfiles(UserPageRequest userPageRequest) {
        Page<CeCustDto> page = new Page<>(userPageRequest.getPageNumber(), userPageRequest.getPageSize());
        String orgNo = userPageRequest.getOrgNo();
        if (StrUtil.isBlank(orgNo)) {
            userPageRequest.setOrgNo(AuthUtils.getOrgNo());
        }
        return ceCustMapper.listPagedUserProfiles(userPageRequest, page);
    }
}




