package com.gaea.iesms.bm.archive.controller;


import com.gaea.base.core.R;
import com.gaea.base.enums.HandleCodeEnum;
import com.gaea.iesms.bm.archive.model.request.ceResource.EnergyResourceTreeRequest;
import com.gaea.iesms.bm.archive.model.request.ceResource.EnergyResourceTreeVo;
import com.gaea.iesms.bm.archive.model.response.vo.ResourceVo;
import com.gaea.iesms.bm.archive.service.CeResourceService;
import com.gaea.iesms.core.model.entity.archive.CeResource;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.validation.Valid;
import java.util.List;

/**
 * @author JIE
 * @version 1.0
 * @description: 用能资源管理
 * @date 2023/4/4 14:33
 */
@Api(tags = "用能资源管理")
@RestController
@RequestMapping("/ceResource")
@Slf4j
public class CeResourceController {

    @Resource
    private CeResourceService ceResourceService;

    /**
     * 查询微电网空间结构
     *
     * @param request 请求参数
     * @author JIE
     * @date 2023/7/17 9:10
     */
    @ApiOperation(value = "获取用能资源树（懒加载）")
    @GetMapping("/getMicrogridSpatialStructure")
    public R<List<EnergyResourceTreeVo>> getMicrogridSpatialStructure(@Valid EnergyResourceTreeRequest request) {
        List<EnergyResourceTreeVo> microgridSpatialStructure = ceResourceService.getMicrogridSpatialStructure(request);
        return R.ok(microgridSpatialStructure);
    }

    @ApiOperation(value = "获取用能资源树（一次性加载）")
    @GetMapping("/getEnergyResourceTree")
    public R<List<EnergyResourceTreeVo>> getEnergyResourceTree(@Valid EnergyResourceTreeRequest request) {
        List<EnergyResourceTreeVo> energyResourceTree = ceResourceService.getEnergyResourceTree(request);
        return R.ok(energyResourceTree);
    }

    @ApiImplicitParams(value = {
            @ApiImplicitParam(name = "id", value = "用能资源id", required = true, dataType = "String", paramType = "query")
    })
    @ApiOperation("获取单个用能资源")
    @GetMapping("/getCeResource")
    public R<CeResource> getCeResource(@RequestParam("id") String id) {
        CeResource ceResource = ceResourceService.getById(id);
        return R.ok(ceResource);
    }

    @ApiImplicitParams({
            @ApiImplicitParam(name = "keyWord", value = "搜索关键词", required = true, dataType = "String"),
    })
    @ApiOperation(value = "搜索用能资源")
    @GetMapping("searchEnergyResources")
    public R<List<ResourceVo>> searchEnergyResources(String keyWord) {
        List<ResourceVo> resourceVos = ceResourceService.searchEnergyResources(keyWord);
        return R.ok(resourceVos);
    }

    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "用能资源ID", required = true, dataType = "String"),
    })
    @ApiOperation(value = "搜索用能资源层级")
    @GetMapping("getEnergyResourceHierarchy")
    public R<List<String>> getEnergyResourceHierarchy(String id) {
        List<String> energyResourceHierarchy = ceResourceService.getEnergyResourceHierarchy(id);
        return R.ok(energyResourceHierarchy);
    }


    @ApiImplicitParams(value = {
            @ApiImplicitParam(name = "ids", value = "用能资源id", required = true, dataType = "List", paramType = "body"),
    })
    @ApiOperation(value = "删除用能资源")
    @DeleteMapping("/deleteCeResource")
    public R<HandleCodeEnum> deleteCeResource(@RequestBody List<String> ids) {
        boolean b = ceResourceService.deleteResourcesById(ids);
        if (!b) {
            return R.enums(HandleCodeEnum.DELETE_ERROR);
        }
        return R.enums(HandleCodeEnum.DELETE_SUCCESS);
    }
}
