package com.gaea.iesms.bm.archive.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.gaea.base.error.BizException;
import com.gaea.iesms.bm.archive.mapper.CeModelMicConfigMapper;
import com.gaea.iesms.bm.archive.model.request.MeasurementItemParamsRequest;
import com.gaea.iesms.bm.archive.model.response.vo.MeasurementItemParamsVo;
import com.gaea.iesms.bm.archive.service.CeModelMicConfigService;
import com.gaea.iesms.bm.archive.service.CePointMeterService;
import com.gaea.iesms.bm.archive.service.CeResourceService;
import com.gaea.iesms.core.feign.dto.DevMeterIdAndCeResIdDto;
import com.gaea.iesms.core.feign.request.AllMeasurementItemsRequest;
import com.gaea.iesms.core.feign.service.RemoteIotService;
import com.gaea.iesms.core.model.entity.archive.CeModelMicConfig;
import com.gaea.iesms.core.model.entity.archive.CeResource;
import com.gaea.iesms.core.model.entity.archive.dto.MeasurementItemDto;
import com.gaea.iesms.core.model.entity.iot.GmDevMeter;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 量测表计设备型号量测项编码配置
 *
 * @author JIE
 * @version 1.0
 * @date 2024/3/29 15:04
 */
@Service
public class CeModelMicConfigServiceImpl extends ServiceImpl<CeModelMicConfigMapper, CeModelMicConfig> implements CeModelMicConfigService {

    @Resource
    private CeModelMicConfigMapper ceModelMicConfigMapper;

    @Resource
    private CeResourceService ceResourceService;

    @Resource
    private CePointMeterService cePointMeterService;

    @Resource
    private RemoteIotService remoteIotService;

    @Override
    public List<MeasurementItemDto> getLeftThreeMeasurementItems(String modelCode) {
        return ceModelMicConfigMapper.getLeftThreeMeasurementItems(modelCode);
    }

    @Override
    public List<CeModelMicConfig> getAllMeasurementItems(AllMeasurementItemsRequest request) {
        LambdaQueryWrapper<CeModelMicConfig> micConfigWrapper = new LambdaQueryWrapper<>();
        micConfigWrapper
                .eq(CeModelMicConfig::getModelCode, request.getModelCode())
                .eq(StrUtil.isNotEmpty(request.getDisplayType()), CeModelMicConfig::getDisplayType, request.getDisplayType())
                .ne(CeModelMicConfig::getMeasItemType, request.getMeasItemType());
        return ceModelMicConfigMapper.selectList(micConfigWrapper);
    }

    @Override
    public List<MeasurementItemParamsVo> getMeasurementItemParams(MeasurementItemParamsRequest request) {
        // 查询用能资源
        CeResource ceResource = ceResourceService.getById(request.getId());
        if (ceResource == null) {
            throw BizException.of("该资源已被删除，请刷新页面重新选择");
        }
        List<Long> ceResIds = Collections.singletonList(ceResource.getId());
        List<String> devMeterIdList = getDevMeterIdAndCeResIds(ceResIds);
        if (CollectionUtil.isEmpty(devMeterIdList)) {
            return null;
        }
        // 根据表计ID列表，远程调用服务，获取对应的量测点列表
        List<GmDevMeter> gmDevMeterList = remoteIotService.getDevMeterIdList(devMeterIdList).getData();
        if (CollectionUtil.isEmpty(gmDevMeterList)) {
            return null;
        }

        GmDevMeter gmDevMeter = gmDevMeterList.get(0);
        // 获取设备型号
        String modelCode = gmDevMeter.getModelCode();
        if (StrUtil.isEmpty(modelCode)) {
            return null;
        }

        List<MeasurementItemParamsVo> measurementItemParamsVos = new ArrayList<>();
        // 获取量测项
        List<CeModelMicConfig> ceModelMicConfigs = getCeModelMicConfigs(modelCode, request.getDisplayType());
        if (CollectionUtil.isNotEmpty(ceModelMicConfigs)) {
            ceModelMicConfigs.forEach(item -> {
                MeasurementItemParamsVo measurementItemParamsVo = new MeasurementItemParamsVo();
                measurementItemParamsVo.setMeasItemAlias(item.getMeasItemAlias());
                measurementItemParamsVo.setMeasItemCode(item.getMeasItemCode());
                measurementItemParamsVo.setMeasItemType(item.getMeasItemType());
                measurementItemParamsVo.setMeasItemUnit(item.getMeasItemUnit());
                measurementItemParamsVos.add(measurementItemParamsVo);
            });
        }
        return measurementItemParamsVos;
    }

    /**
     * 批量获取量测表计
     *
     * @param: ceResIds 用能资源Id 集合
     * @return: java.util.List<java.lang.String>
     * @author JIE
     * @date: 2023/12/6 16:27
     */
    private List<String> getDevMeterIdAndCeResIds(List<Long> ceResIds) {
        List<DevMeterIdAndCeResIdDto> devMeterIdListByCeResIds = cePointMeterService.getDevMeterIdListByCeResIds(ceResIds);
        return devMeterIdListByCeResIds
                .stream()
                .map(DevMeterIdAndCeResIdDto::getDevMeterId)
                .collect(Collectors.toList());
    }

    /**
     * 获取量测项
     *
     * @param: modelCode
     * @param: displayType 1 左边默认 0 - 右侧选择显示
     * @return: [java.lang.String, java.lang.String]
     * @author JIE
     * @date: 2023/12/1 15:55
     */
    private List<CeModelMicConfig> getCeModelMicConfigs(String modelCode, String displayType) {
        LambdaQueryWrapper<CeModelMicConfig> micConfigWrapper = new LambdaQueryWrapper<>();
        micConfigWrapper
                .eq(CeModelMicConfig::getModelCode, modelCode)
                .eq(StrUtil.isNotEmpty(displayType), CeModelMicConfig::getDisplayType, displayType)
                .ne(CeModelMicConfig::getMeasItemType, "GROUP");
        return ceModelMicConfigMapper.selectList(micConfigWrapper);
    }
}
