package com.gaea.iesms.bm.archive.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONObject;
import cn.hutool.json.JSONUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.fasterxml.jackson.databind.JsonNode;
import com.gaea.base.core.R;
import com.gaea.iesms.bm.archive.mapper.CeContainerMapper;
import com.gaea.iesms.bm.archive.model.request.ceContainer.CeContainerRequest;
import com.gaea.iesms.bm.archive.model.request.ceContainer.UpdateCeContainerRequest;
import com.gaea.iesms.bm.archive.model.response.dto.ceContainer.CeContainerDto;
import com.gaea.iesms.bm.archive.model.response.dto.ceResourceSort.CeResourceSortDto;
import com.gaea.iesms.bm.archive.service.*;
import com.gaea.iesms.core.feign.request.archive.PagedContainersRequest;
import com.gaea.iesms.core.feign.response.archive.CeContainerVo;
import com.gaea.iesms.core.feign.service.RemoteIotService;
import com.gaea.iesms.core.model.entity.archive.*;
import com.gaea.iesms.core.model.entity.iot.GmDevMeter;
import com.gaea.iesms.core.model.enums.archive.CeResClassEnum;
import com.gaea.iesms.core.model.jsonobject.archive.CePointPropsJsonObject;
import com.gaea.spring.cloud.starter.util.AuthUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

import static com.gaea.iesms.core.model.entity.archive.factory.CeResourcePropsFactory.createResourceData;

/**
 * @author Administrator
 * @description 针对表【ce_container(用能容器 )】的数据库操作Service实现
 * @createDate 2023-02-20 09:41:47
 */
@Slf4j
@Service
public class CeContainerServiceImpl extends ServiceImpl<CeContainerMapper, CeContainer>
        implements CeContainerService {

    @Resource
    private RemoteIotService remoteIotService;

    @Resource
    private CeContainerMapper ceContainerMapper;

    @Resource
    private CeResourceService ceResourceService;

    @Resource
    private CePartService cePartService;

    @Resource
    private CePointService cePointService;

    @Resource
    private CePointMeterService cePointMeterService;

    @Resource
    private CeResourceSortService ceResourceSortService;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public String addCeContainer(CeContainerRequest ceContainerRequest) {
        String orgNo = AuthUtils.getOrgNo();

        // 1、添加用能容器
        // 1.1 根据上级用能资源标识查询上级资源是什么类型
        Long parentCeResId = ceContainerRequest.getParentCeResId();
        CeResource parCeResource = ceResourceService.getById(parentCeResId);
        // 1.2 判断上级用能资源类型
        Integer ceResClass = parCeResource.getCeResClass();

        // 1.3 设置用能容器
        CeContainer ceContainer = new CeContainer();
        switch (CeResClassEnum.fromCode(ceResClass)) {
            case CUSTOMER:
                ceContainer.setCeCustId(parCeResource.getId());
                ceContainer.setParentCeResClass(CeResClassEnum.CUSTOMER.getValue());
                ceContainer.setParentCeResId(parCeResource.getId());
                ceContainer.setCePartId(0L);
                ceContainer.setParentId(0L);
                break;
            case REGION:
                CePart part = cePartService.getById(parCeResource.getId());
                ceContainer.setCeCustId(part.getCeCustId());
                ceContainer.setCePartId(part.getId());
                ceContainer.setParentCeResId(part.getId());
                ceContainer.setParentCeResClass(CeResClassEnum.REGION.getValue());
                ceContainer.setParentId(0L);
                break;
            case CONTAINER:
                CeContainer container = getById(parCeResource.getId());
                ceContainer.setCeCustId(container.getCeCustId());
                ceContainer.setCePartId(container.getCePartId());
                ceContainer.setParentCeResId(container.getId());
                ceContainer.setParentCeResClass(CeResClassEnum.CONTAINER.getValue());
                ceContainer.setParentId(container.getId());
                break;
            default:
                throw new IllegalArgumentException("Invalid ceResClass: " + ceResClass);
        }
        ceContainer.setCeResSortNo(ceContainerRequest.getCeResSortNo());
        JsonNode ceCntrProps = ceContainerRequest.getCeResourceProps();
        // 1.4 调用工厂方法创建实体类
        CeResourceProps resourceData = createResourceData(ceContainerRequest.getCeResSortNo());
        ceContainer.setCeCntrProps(resourceData.savaConvertFields(ceCntrProps));
        ceContainer.setCeResSortNoId(ceContainerRequest.getCeResSortNoId());
        // 1.5 新增用能容器
        save(ceContainer);

        // 2、将一些公共的字段从json里面抽取出来
        JsonNode ceResourceProps = ceContainerRequest.getCeResourceProps();
        String ceResName = ceResourceProps.get("ceResName").asText();
        JsonNode sortSn1 = ceResourceProps.get("sortSn");
        String sortSn = sortSn1 == null || "".equals(sortSn1.asText()) || "null".equals(sortSn1.asText()) ? "1" : sortSn1.asText();
        JsonNode ceResNo1 = ceResourceProps.get("ceResNo");
        String ceResNo = ceResNo1 == null ? "" : ceResNo1.asText();
        // 获取关联表计
        JsonNode correlTableMeter = ceResourceProps.get("correlTableMeter");
        // 转换成List
        List<String> meterList = new ArrayList<>();
        if (correlTableMeter != null) {
            if (correlTableMeter.isArray()) {
                for (Iterator<JsonNode> it = correlTableMeter.elements(); it.hasNext(); ) {
                    meterList.add(it.next().asText());
                }
            }
        }

        // 3、新增用能点
        CePoint cePoint = new CePoint();
        cePoint.setOrgNo(orgNo);
        cePoint.setCeCustId(ceContainer.getCeCustId());
        cePoint.setCeResId(ceContainer.getId());
        cePoint.setCeResClass(3);
        cePoint.setCePointLevel(1);
        cePoint.setCePointNo(ceResNo);
        if (StrUtil.isBlank(ceResNo)) {
            cePoint.setCePointNo(ceResNo);
        }
        cePoint.setCePointName(ceResName);
        cePoint.setIsSettlement(0);
        // 设置用能类型
        CeResourceSortDto ceResourceSortDto = ceResourceSortService.queryCeResourceSortById(ceContainerRequest.getCeResSortNoId());
        cePoint.setCePointType(ceResourceSortDto.getCeResType());
        cePoint.setParentId(ceContainer.getParentCeResId());
        cePoint.setCePointProps(new CePointPropsJsonObject());
        cePoint.setMeasPointIdList(meterList);
        cePointService.save(cePoint);

        // 4、如果关联表计集合不为空就新增用能点表计关联表
        if (CollectionUtil.isNotEmpty(meterList)) {
            List<CePointMeter> cePointMeters = new ArrayList<>();
            for (String aLong : meterList) {
                CePointMeter cePointMeter = new CePointMeter();
                cePointMeter.setCePointId(cePoint.getId());
                cePointMeter.setDevMeterId(Long.valueOf(aLong));
                cePointMeters.add(cePointMeter);
            }
            cePointMeterService.saveBatch(cePointMeters);
        }

        // 5、新增用能资源
        CeResource ceResource = new CeResource();
        ceResource.setId(ceContainer.getId());
        ceResource.setCeResClass(CeResClassEnum.CONTAINER.getValue());
        ceResource.setCeResStatus("20");
        // 设置用能容器名称
        ceResource.setCeResName(ceResName);
        // 设置排序序号
        ceResource.setSortSn(Integer.valueOf(sortSn));
        // 设置用能容器编号
        ceResource.setCeResNo(ceResNo);
        if (StrUtil.isBlank(ceResNo)) {
            ceResource.setCeResNo(String.valueOf(ceContainer.getId()));
        }
        ceResource.setCeResDesc(ceResName);
        ceResource.setCeResAbbr(ceResName);
        ceResource.setOrgNo(orgNo);
        if (StrUtil.isBlank(ceResNo)) {
            ceResource.setCeResNo(String.valueOf(ceContainer.getId()));
        }
        //  新增用能资源
        ceResourceService.addCeResource(ceResource, ceContainerRequest.getCeResSortNo());
        return ceResource.getId().toString();
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public Boolean updateContainerByProfileId(UpdateCeContainerRequest request) {
        // 1、修改用能容器
        CeContainer ceContainer = new CeContainer();
        BeanUtils.copyProperties(request, ceContainer);
        ceContainer.setId(Long.valueOf(request.getId()));
        JsonNode ceCntrProps = request.getCeResourceProps();
        // 调用工厂方法创建实体类
        CeResourceProps resourceData = createResourceData(ceContainer.getCeResSortNo());
        ceContainer.setCeCntrProps(resourceData.savaConvertFields(ceCntrProps));
        updateById(ceContainer);

        // 2、将一些公共的字段从json里面抽取出来
        JsonNode ceResourceProps = request.getCeResourceProps();
        String ceResName = ceResourceProps.get("ceResName").asText();
        JsonNode sortSn1 = ceResourceProps.get("sortSn");
        String sortSn = sortSn1 == null || "".equals(sortSn1.asText()) || "null".equals(sortSn1.asText()) ? "1" : sortSn1.asText();
        JsonNode ceResNo1 = ceResourceProps.get("ceResNo");
        String ceResNo = ceResNo1 == null ? "" : ceResNo1.asText();
        // 获取关联表计
        JsonNode correlTableMeter = ceResourceProps.get("correlTableMeter");
        // 转换成List
        // 将correlTableMeter 转为 LIST<Long>
        List<String> meterList = new ArrayList<>();
        if (correlTableMeter != null) {
            if (correlTableMeter.isArray()) {
                for (Iterator<JsonNode> it = correlTableMeter.elements(); it.hasNext(); ) {
                    meterList.add(it.next().asText());
                }
            }
        }

        // 3、查询用能点
        LambdaQueryWrapper<CePoint> pointQueryWrapper = new LambdaQueryWrapper<>();
        pointQueryWrapper
                .eq(CePoint::getCeResId, request.getId());
        CePoint cePoint = cePointService.getOne(pointQueryWrapper);
        // 4、修改用能点
        cePoint.setCePointNo(ceResNo);
        if (StrUtil.isBlank(ceResNo)) {
            cePoint.setCePointNo(request.getId());
        }
        cePoint.setCePointName(ceResName);
        cePoint.setMeasPointIdList(meterList);
        cePointService.updateById(cePoint);

        // 5、先删除用能点量测表计关系
        cePointMeterService.removeByCePointIds(Collections.singletonList(cePoint.getId()));
        // 6、重新添加用能点量测表计关系
        if (CollectionUtil.isNotEmpty(meterList)) {
            List<CePointMeter> cePointMeters = new ArrayList<>();
            for (String aLong : meterList) {
                CePointMeter cePointMeter = new CePointMeter();
                cePointMeter.setCePointId(cePoint.getId());
                cePointMeter.setDevMeterId(Long.valueOf(aLong));
                cePointMeters.add(cePointMeter);
            }
            cePointMeterService.saveBatch(cePointMeters);
        }

        // 7、修改用能资源
        CeResource ceResource = new CeResource();
        BeanUtils.copyProperties(request, ceResource);
        ceResource.setId(Long.valueOf(request.getId()));
        // 设置用能资源名称
        ceResource.setCeResName(ceResName);
        // 设置排序序号
        ceResource.setSortSn(Integer.valueOf(sortSn));
        // 设置用能资源编号
        ceResource.setCeResNo(ceResNo);
        if (StrUtil.isBlank(ceResNo)) {
            ceResource.setCeResNo(request.getId());
        }
        ceResource.setSortSn(Integer.valueOf(sortSn));
        ceResource.setCeResClass(CeResClassEnum.CONTAINER.getValue());
        return ceResourceService.updateCeResource(ceResource, request.getCeResSortNo());
    }

    @Override
    public CeContainerVo getContainerById(String id) {
        CeContainer ceContainer = ceContainerMapper.selectById(id);
        CeResource ceResource = ceResourceService.getById(id);
        CeContainerDto ceContainerDto = new CeContainerDto();
        // 使用copyProperties方法，避免重复的赋值
        BeanUtils.copyProperties(ceContainer, ceContainerDto);
        BeanUtils.copyProperties(ceResource, ceContainerDto);

        // 获取并网关联表记
        CeResourceProps ceDevProps = ceContainerDto.getCeCntrProps();
        JSONObject jsonObject = JSONUtil.parseObj(ceDevProps);
        List<String> correlTableMeter = (List<String>) jsonObject.get("correlTableMeter");
        if (CollectionUtil.isNotEmpty(correlTableMeter)) {
            // 调用iot接口获取表记信息
            R<List<GmDevMeter>> devMeterIdList = remoteIotService.getDevMeterIdList(correlTableMeter);
            log.info("调用iot接口获取表记信息:{}", devMeterIdList.getCode());
            if (CollectionUtil.isNotEmpty(devMeterIdList.getData())) {
                List<GmDevMeter> data = devMeterIdList.getData();
                // 使用map方法，简化映射
                List<String> devMeterNos = data
                        .stream()
                        .map(GmDevMeter::getDevMeterNo)
                        .collect(Collectors.toList());
                jsonObject.set("correlTableMeterName", devMeterNos);
            }
        }

        CeContainerVo ceContainerVo = new CeContainerVo();
        BeanUtils.copyProperties(ceContainerDto, ceContainerVo);
        ceContainerVo.setCeResourceProps(jsonObject);
        return ceContainerVo;
    }

    @Override
    public IPage<CeContainerVo> listPagedContainers(PagedContainersRequest request) {
        IPage<CeContainerVo> ceContainerVoIPage = new Page<>();

        // 根据请求参数创建分页对象并查询容器信息
        Page<CeContainer> page = page(new Page<>(request.getPageNumber(), request.getPageSize()));

        // 判断查询结果是否不为空
        if (CollectionUtil.isNotEmpty(page.getRecords())) {
            List<CeContainerVo> ceContainerVoList = new ArrayList<>();
            // 遍历查询结果，转换为Vo对象
            for (CeContainer ceContainer : page.getRecords()) {
                CeContainerVo ceContainerVo = new CeContainerVo();
                BeanUtils.copyProperties(ceContainer, ceContainerVo);
                ceContainerVoList.add(ceContainerVo);
            }
            // 设置转换后的Vo对象列表到分页对象
            ceContainerVoIPage.setRecords(ceContainerVoList);

            // 根据Vo列表的ID查询关联的资源信息
            List<Long> ids = ceContainerVoList.stream().map(CeContainerVo::getId).collect(Collectors.toList());
            List<CeResource> ceResources = ceResourceService.listByIds(ids);

            // 将资源信息转换为Map，以ID为键
            Map<Long, CeResource> ceResourceMap = ceResources.stream().collect(Collectors.toMap(CeResource::getId, ceResource -> ceResource));
            // 将资源信息合并到Vo对象中
            ceContainerVoList.forEach(ceContainerVo -> {
                CeResource ceResource = ceResourceMap.get(ceContainerVo.getId());
                BeanUtils.copyProperties(ceResource, ceContainerVo);
            });
        }

        // 设置总记录数到分页对象
        ceContainerVoIPage.setTotal(page.getTotal());
        ceContainerVoIPage.setPages(page.getPages());
        ceContainerVoIPage.setCurrent(page.getCurrent());
        ceContainerVoIPage.setSize(page.getSize());
        return ceContainerVoIPage;
    }
}




