package com.gaea.iesms.bm.archive.service.impl;


import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.gaea.base.error.BizException;
import com.gaea.iesms.bm.archive.mapper.*;
import com.gaea.iesms.bm.archive.model.request.ceContainer.CeContainerByResClassAndTypeQueryRequest;
import com.gaea.iesms.bm.archive.model.request.ceContainer.ContainerListRequest;
import com.gaea.iesms.bm.archive.model.request.ceDeivce.CeDeviceByResClassAndTypeQueryRequest;
import com.gaea.iesms.bm.archive.model.request.ceDeivce.CeDevicePageRequest;
import com.gaea.iesms.bm.archive.model.request.cePart.CePartByResClassAndTypeQueryRequest;
import com.gaea.iesms.bm.archive.model.request.cePart.QueryPartRequest;
import com.gaea.iesms.bm.archive.model.request.ceResource.EnergyResourceTreeRequest;
import com.gaea.iesms.bm.archive.model.request.ceResource.EnergyResourceTreeVo;
import com.gaea.iesms.bm.archive.model.request.ceResource.IsNumberExistsRequest;
import com.gaea.iesms.bm.archive.model.request.ceResourceDetail.GetCeResourceDetailRequest;
import com.gaea.iesms.bm.archive.model.response.dto.ceContainer.CeContainerDto;
import com.gaea.iesms.bm.archive.model.response.dto.ceDevice.CeDeviceDto;
import com.gaea.iesms.bm.archive.model.response.dto.cePart.CePartDto;
import com.gaea.iesms.bm.archive.model.response.dto.ceResource.EnergyResourceTypeDto;
import com.gaea.iesms.bm.archive.model.response.dto.ceResourceDetail.CeResourceDetailDto;
import com.gaea.iesms.bm.archive.model.response.dto.ceResourceDetail.EnergyResourceDto;
import com.gaea.iesms.bm.archive.model.response.vo.ResourceVo;
import com.gaea.iesms.bm.archive.service.*;
import com.gaea.iesms.core.model.entity.archive.*;
import com.gaea.iesms.core.model.enums.archive.CeResClassEnum;
import com.gaea.spring.cloud.starter.util.AuthUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 针对表【ce_resource(用能资源 )】的数据库操作Service实现
 *
 * @author Administrator
 * @createDate 2023-02-15 17:25:43
 */
@Service
public class CeResourceServiceImpl extends ServiceImpl<CeResourceMapper, CeResource>
        implements CeResourceService {

    @Resource
    private CeResourceDetailService ceResourceDetailService;

    @Resource
    private CePartMapper partMapper;

    @Resource
    private CeContainerMapper containerMapper;

    @Resource
    private CeDeviceMapper deviceMapper;

    @Resource
    private CeCustMapper ceCustMapper;

    @Resource
    private CeResourceMapper ceResourceMapper;

    @Resource
    private CePointService cePointService;

    @Resource
    private CePointMeterService cePointMeterService;

    @Resource
    private CeResourceSortService ceResourceSortService;

    @Override
    public Boolean addCeResource(CeResource ceResource, String ceResSortNo) {
        // 判断编号
        String ceResNo = ceResource.getCeResNo();
        if (StrUtil.isNotEmpty(ceResNo) && !"null".equals(ceResNo)) {
            // 查询编号是否重复
            IsNumberExistsRequest request = new IsNumberExistsRequest();
            request.setId(ceResource.getId());
            request.setCeResNo(ceResource.getCeResNo());
            request.setCeResClass(ceResource.getCeResClass());
            request.setOrgNo(AuthUtils.getOrgNo());
            request.setCeResSortNo(ceResSortNo);
            int numberExists = ceResourceMapper.isNumberExists(request);
            if (numberExists > 0) {
                throw BizException.of("编号重复，请重新输入");
            }
        }
        return save(ceResource);
    }

    @Override
    public Boolean addVppCeResource(CeResource ceResource) {
        return save(ceResource);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public Boolean deleteResourcesById(List<String> ids) {

        if (CollectionUtil.isEmpty(ids)) {
            throw BizException.of("id不能为空");
        }

        LambdaQueryWrapper<CeResource> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.in(CeResource::getId, ids);

        List<CeResource> ceResources = list(queryWrapper);
        // 如果用能资源对象为null，抛出业务异常
        if (CollectionUtil.isEmpty(ceResources)) {
            throw BizException.of("该用能资源已被删除");
        }

        // 创建用于存储要删除的用能资源ID列表
        List<Long> ceResIds = new ArrayList<>();
        for (CeResource ceResource : ceResources) {
            // 创建用于存储要删除的用能资源ID列表
            ceResIds.add(ceResource.getId());
            // 获取用能资源的类别
            Integer ceResClass = ceResource.getCeResClass();
            // 将类别转换为枚举类型
            CeResClassEnum ceResClassEnum = CeResClassEnum.fromCode(ceResClass);
            // 处理用能资源的类别，执行相应的操作
            handleCeResClass(ceResClassEnum, ceResource, ceResIds);
        }

        // 删除与用能资源相关的能源点信息
        deleteEnergyPoints(ceResIds);
        // 根据用能资源ID列表批量删除用能资源记录
        return removeByIds(ceResIds);
    }

    @Override
    public Boolean updateCeResource(CeResource ceResource, String ceResSortNo) {
        return updateById(ceResource);
    }

    @Override
    public List<EnergyResourceTreeVo> getMicrogridSpatialStructure(EnergyResourceTreeRequest request) {
        List<EnergyResourceTreeVo> energyResourceTreeVos = new ArrayList<>();

        List<CeResourceDetailDto> resourceDetailDtos = new ArrayList<>();

        // 如果存在上级资源id，则查询该资源的子资源明细
        if (request.getSuperiorId() != null) {
            GetCeResourceDetailRequest resourceDetailRequest = buildResourceDetailRequest(request);
            resourceDetailDtos = ceResourceDetailService.querySubResourcesBySuperResource(resourceDetailRequest);
        }

        List<EnergyResourceDto> resourceDtos = new ArrayList<>();
        // 查询用能资源详情并处理
        processResourceDetails(request, resourceDtos, resourceDetailDtos);

        // 将用能资源转换为能源资源树节点
        if (CollectionUtil.isNotEmpty(resourceDtos)) {
            energyResourceTreeVos = resourceDtos.stream()
                    .map(resourceDto -> {
                        EnergyResourceTreeVo energyResourceTreeVo = new EnergyResourceTreeVo();
                        energyResourceTreeVo.setTreeId(String.valueOf(resourceDto.getId()));
                        energyResourceTreeVo.setTreeName(resourceDto.getCeResName());
                        energyResourceTreeVo.setCeResClass(resourceDto.getCeResClass());
                        energyResourceTreeVo.setCeResType(resourceDto.getCeResType());
                        energyResourceTreeVo.setHasChildren(resourceDto.getHasChildren());
                        energyResourceTreeVo.setCeResSortNoId(resourceDto.getCeResSortNoId());
                        energyResourceTreeVo.setCeResSortNo(resourceDto.getCeResSortNo());
                        energyResourceTreeVo.setTreeIcon(resourceDto.getCeResSortIcon());
                        energyResourceTreeVo.setCorrelTableMeter(resourceDto.getCorrelTableMeter());
                        return energyResourceTreeVo;
                    })
                    .collect(Collectors.toList());
        }
        return energyResourceTreeVos;
    }

    @Override
    public List<EnergyResourceTreeVo> getEnergyResourceTree(EnergyResourceTreeRequest request) {
        List<EnergyResourceTreeVo> energyResourceTreeVos = new ArrayList<>();

        List<CeResourceDetailDto> resourceDetailDtos = new ArrayList<>();

        // 如果存在上级资源id，则查询该资源的子资源明细
        if (request.getSuperiorId() != null) {
            GetCeResourceDetailRequest resourceDetailRequest = buildResourceDetailRequest(request);
            resourceDetailDtos = ceResourceDetailService.querySubResourcesBySuperResource(resourceDetailRequest);
        }

        List<EnergyResourceDto> resourceDtos = new ArrayList<>();

        // 查询用能资源详情并处理
        processResourceDetails(request, resourceDtos, resourceDetailDtos);

        // 将用能资源转换为能源资源树节点
        if (CollectionUtil.isNotEmpty(resourceDtos)) {
            energyResourceTreeVos = resourceDtos.stream()
                    .map(resourceDto -> {
                        EnergyResourceTreeVo energyResourceTreeVo = new EnergyResourceTreeVo();
                        energyResourceTreeVo.setTreeId(String.valueOf(resourceDto.getId()));
                        energyResourceTreeVo.setTreeName(resourceDto.getCeResName());
                        energyResourceTreeVo.setCeResClass(resourceDto.getCeResClass());
                        energyResourceTreeVo.setCeResType(resourceDto.getCeResType());
                        energyResourceTreeVo.setHasChildren(resourceDto.getHasChildren());
                        energyResourceTreeVo.setCeResSortNoId(resourceDto.getCeResSortNoId());
                        energyResourceTreeVo.setCorrelTableMeter(resourceDto.getCorrelTableMeter());
                        energyResourceTreeVo.setCeResSortNo(resourceDto.getCeResSortNo());
                        energyResourceTreeVo.setTreeIcon(resourceDto.getCeResSortIcon());

                        // 递归查询子节点
                        if (resourceDto.getHasChildren()) {
                            EnergyResourceTreeRequest subRequest = new EnergyResourceTreeRequest();
                            subRequest.setTreeId(request.getTreeId());
                            subRequest.setSuperiorId(String.valueOf(resourceDto.getId()));
                            subRequest.setSuperiorResourceType(resourceDto.getCeResClass());
                            subRequest.setSupperCeResSortNoId(resourceDto.getCeResSortNoId());
                            subRequest.setCeResType(request.getCeResType());
                            List<EnergyResourceTreeVo> subTreeVos = getEnergyResourceTree(subRequest);
                            energyResourceTreeVo.setTreeVos(subTreeVos);
                        }

                        return energyResourceTreeVo;
                    })
                    .collect(Collectors.toList());
        }
        return energyResourceTreeVos;
    }

    @Override
    public List<ResourceVo> searchEnergyResources(String keyWord) {
        LambdaQueryWrapper<CeResource> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper
                .like(CeResource::getCeResName, keyWord)
                .eq(CeResource::getOrgNo, AuthUtils.getOrgNo());
        List<CeResource> list = list(queryWrapper);
        if (CollectionUtil.isEmpty(list)) {
            return new ArrayList<>();
        }
        return list.stream()
                .map(ceResource -> {
                    ResourceVo resourceVo = new ResourceVo();
                    resourceVo.setId(String.valueOf(ceResource.getId()));
                    resourceVo.setCeResName(ceResource.getCeResName());
                    return resourceVo;
                })
                .collect(Collectors.toList());
    }

    @Override
    public List<String> getEnergyResourceHierarchy(String id) {
        List<String> returnList = new ArrayList<>();
        getEnergyResourceHierarchyRecursive(id, returnList);
        return returnList;
    }

    /**
     * 递归获取用能资源层级结构
     *
     * @param id         能源资源ID
     * @param resultList 用于存储结果的列表
     */
    private void getEnergyResourceHierarchyRecursive(String id, List<String> resultList) {
        // 根据ID获取能源资源对象
        CeResource ceResource = getById(id);
        // 获取能源资源类型
        Integer ceResClass = ceResource.getCeResClass();
        // 根据用能资源类型进行不同的处理
        switch (CeResClassEnum.fromCode(ceResClass)) {
            case CUSTOMER:
                // 客户类型无需处理
                break;
            case REGION:
                // 区域类型，根据ID查询对应的区域对象
                CePart cePart = partMapper.selectById(id);
                // 如果区域对象关联了客户ID，则递归获取客户的用能资源层级结构
                if (cePart.getCeCustId() != null && cePart.getCeCustId() != 0) {
                    getEnergyResourceHierarchyRecursive(String.valueOf(cePart.getCeCustId()), resultList);
                }
                break;
            case CONTAINER:
                // 容器类型，根据ID查询对应的容器对象
                CeContainer container = containerMapper.selectById(id);
                // 如果容器对象有上级用能资源ID，则递归获取上级用能资源的层级结构
                if (container.getParentCeResId() != null && container.getParentCeResId() != 0) {
                    getEnergyResourceHierarchyRecursive(String.valueOf(container.getParentCeResId()), resultList);
                }
                break;
            case DEVICE:
                // 设备类型，根据ID查询对应的设备对象
                CeDevice ceDevice = deviceMapper.selectById(id);
                // 如果设备对象有上级用能资源ID，则递归获取上级用能资源的层级结构
                if (ceDevice.getParentCeResId() != null && ceDevice.getParentCeResId() != 0) {
                    getEnergyResourceHierarchyRecursive(String.valueOf(ceDevice.getParentCeResId()), resultList);
                }
                break;
            default:
                throw new IllegalArgumentException("Invalid ceResClass: " + ceResClass);
        }

        // 添加非重复的ID到结果列表中
        addUnique(resultList, id);
    }

    /**
     * 添加一个唯一的元素到列表中
     *
     * @param list  要添加元素的列表
     * @param value 要添加的元素
     */
    private void addUnique(List<String> list, String value) {
        if (!list.contains(value)) {
            list.add(value);
        }
    }

    /**
     * 根据传入的 EnergyResourceTreeRequest 对象构建 GetCeResourceDetailRequest 对象。
     *
     * @param request EnergyResourceTreeRequest 对象，包含了构建 GetCeResourceDetailRequest 所需的信息。
     * @return 返回构建好的 GetCeResourceDetailRequest 对象。
     */
    private GetCeResourceDetailRequest buildResourceDetailRequest(EnergyResourceTreeRequest request) {
        // 使用 GetCeResourceDetailRequest 的 builder 构造方法，
        // 并利用 request 对象中的相应字段构建 GetCeResourceDetailRequest 对象。
        return GetCeResourceDetailRequest
                .builder()
                .treeId(request.getTreeId())
                .supperCeResSortNo(request.getSupperCeResSortNoId())
                .supperCeResClass(request.getSuperiorResourceType())
                .build();
    }

    /**
     * 根据传入的 EnergyResourceTreeRequest 对象和 CeResourceDetailDto 列表处理资源详细信息，
     * 并将处理结果添加到 resourceDtos 列表中。
     *
     * @param request            EnergyResourceTreeRequest 对象，包含用能资源树请求的信息。
     * @param resourceDtos       用于存放处理结果的 EnergyResourceDto 列表。
     * @param resourceDetailDtos 需要处理的 CeResourceDetailDto 列表。
     */
    private void processResourceDetails(EnergyResourceTreeRequest request,
                                        List<EnergyResourceDto> resourceDtos,
                                        List<CeResourceDetailDto> resourceDetailDtos) {

        if (CollectionUtil.isNotEmpty(resourceDetailDtos)) {
            // 根据 CeResourceDetailDto 列表中的下级用能资源类型进行分组，
            // 然后将分组结果以用能资源类型为键，用能资源编号列表为值的形式存放到 groupedCeResourceMap 中。
            Map<Integer, List<String>> groupedCeResourceMap = resourceDetailDtos.stream()
                    .collect(Collectors.groupingBy(CeResourceDetailDto::getLowerCeResClass,
                            Collectors.mapping(CeResourceDetailDto::getLowerCeResSortNo, Collectors.toList())));

            // 根据 groupedCeResourceMap 中的数据分别处理区域、容器和设备的信息。
            // 区域
            List<String> ceResSortNos = groupedCeResourceMap.getOrDefault(2, Collections.emptyList());
            processArea(request, resourceDtos, ceResSortNos);

            // 容器
            List<String> containerIds = groupedCeResourceMap.getOrDefault(3, Collections.emptyList());
            processContainer(request, resourceDtos, containerIds);

            // 设备
            List<String> deviceIds = groupedCeResourceMap.getOrDefault(4, Collections.emptyList());
            processDevice(request, resourceDtos, deviceIds);
        } else {
            // 用户
            processCust(resourceDtos, request.getTreeId());
        }
    }

    /**
     * 查询用户数据
     *
     * @param resourceDtos 用于存放处理结果的 EnergyResourceDto 列表。
     * @param treeId       树id
     * @author JIE
     * @date 2024/2/26 10:28
     */
    private void processCust(List<EnergyResourceDto> resourceDtos, String treeId) {
        List<EnergyResourceTypeDto> energyResourceTypeDtos = ceCustMapper.listEnergyResourceTypeDto(AuthUtils.getOrgNo());

        if (CollectionUtil.isNotEmpty(energyResourceTypeDtos)) {
            for (EnergyResourceTypeDto cust : energyResourceTypeDtos) {
                // 构建 EnergyResourceDto 对象
                EnergyResourceDto energyResourceDto = buildEnergyResourceDto(cust);

                // 构建 GetCeResourceDetailRequest 对象
                GetCeResourceDetailRequest custResNo = GetCeResourceDetailRequest.builder()
                        .supperCeResClass(1)
                        .treeId(treeId)
                        .build();

                // 查询子资源
                List<CeResourceDetailDto> custResNos = ceResourceDetailService.querySubResourcesBySuperResource(custResNo);
                if (CollectionUtil.isNotEmpty(custResNos)) {
                    // 对子资源进行分组
                    Map<Integer, List<String>> listMap = custResNos.stream()
                            .collect(Collectors.groupingBy(CeResourceDetailDto::getLowerCeResClass,
                                    Collectors.mapping(CeResourceDetailDto::getLowerCeResSortNo, Collectors.toList())));

                    // 检查并设置是否有子节点
                    // 区域
                    checkAndSetHasChildren(energyResourceDto, listMap.getOrDefault(2, Collections.emptyList()), 2);
                    if (!energyResourceDto.getHasChildren()) {
                        // 容器
                        checkAndSetHasChildren(energyResourceDto, listMap.getOrDefault(3, Collections.emptyList()), 3);
                        if (!energyResourceDto.getHasChildren()) {
                            // 设备
                            checkAndSetHasChildren(energyResourceDto, listMap.getOrDefault(4, Collections.emptyList()), 4);
                        }
                    }
                } else {
                    energyResourceDto.setHasChildren(false);
                }
                // 将处理结果添加到 resourceDtos 列表中
                resourceDtos.add(energyResourceDto);
            }
        }
    }

    /**
     * 查询区域数据
     * 并将处理结果添加到 resourceDtos 列表中。
     *
     * @param request        EnergyResourceTreeRequest 对象，包含用能资源树请求的信息。
     * @param resourceDtos   用于存放处理结果的 EnergyResourceDto 列表。
     * @param ceResSortNoIds 用能资源分类编号
     */
    private void processArea(EnergyResourceTreeRequest request, List<EnergyResourceDto> resourceDtos, List<String> ceResSortNoIds) {
        // 检查 ceResSortNos 是否非空
        if (CollectionUtil.isNotEmpty(ceResSortNoIds)) {
            // 根据 ceResSortNos 构建 CePartByResClassAndTypeQueryRequest
            CePartByResClassAndTypeQueryRequest partRequest = CePartByResClassAndTypeQueryRequest.builder()
                    .ceResSortNoIds(ceResSortNoIds)
                    .ceCustId(Long.valueOf(request.getSuperiorId()))
                    .build();

            // 查询下级用能资源
            List<EnergyResourceTypeDto> cePartDtos = partMapper.queryCePartByCeResClassAndCeResType(partRequest);

            // 检查 cePartDtos 是否非空
            if (CollectionUtil.isNotEmpty(cePartDtos)) {
                for (EnergyResourceTypeDto cePart : cePartDtos) {
                    // 构建 EnergyResourceDto 对象
                    EnergyResourceDto energyResourceDto = buildEnergyResourceDto(cePart);

                    // 构建 GetCeResourceDetailRequest 对象
                    GetCeResourceDetailRequest partResNo = GetCeResourceDetailRequest.builder()
                            .supperCeResSortNo(cePart.getCeResSortNoId())
                            .lowerCeResType(request.getCeResType())
                            .supperCeResClass(2)
                            .treeId(request.getTreeId())
                            .build();

                    // 查询子资源
                    List<CeResourceDetailDto> partResNos = ceResourceDetailService.querySubResourcesBySuperResource(partResNo);
                    if (CollectionUtil.isNotEmpty(partResNos)) {
                        // 对子资源进行分组
                        Map<Integer, List<String>> listMap = partResNos.stream()
                                .collect(Collectors.groupingBy(CeResourceDetailDto::getLowerCeResClass,
                                        Collectors.mapping(CeResourceDetailDto::getLowerCeResSortNo, Collectors.toList())));

                        // 检查并设置是否有子节点
                        // 容器
                        checkAndSetHasChildren(energyResourceDto, listMap.getOrDefault(3, Collections.emptyList()), 3);
                        // 如果已经为TRUE了，就不需要再查询设备层级的了
                        if (!energyResourceDto.getHasChildren()) {
                            // 设备
                            checkAndSetHasChildren(energyResourceDto, listMap.getOrDefault(4, Collections.emptyList()), 4);
                        }
                    } else {
                        energyResourceDto.setHasChildren(false);
                    }
                    // 将处理结果添加到 resourceDtos 列表中
                    resourceDtos.add(energyResourceDto);
                }
            }
        }
    }

    /**
     * 查询容器数据
     * 并将处理结果添加到 resourceDtos 列表中。
     *
     * @param request        EnergyResourceTreeRequest 对象，包含用能资源树请求的信息。
     * @param resourceDtos   用于存放处理结果的 EnergyResourceDto 列表。
     * @param ceResSortNoIds 用能资源分类Id
     */
    private void processContainer(EnergyResourceTreeRequest request, List<EnergyResourceDto> resourceDtos, List<String> ceResSortNoIds) {
        // 检查 ceResSortNos 是否非空
        if (CollectionUtil.isNotEmpty(ceResSortNoIds)) {
            // 根据 ceResSortNos 构建 CeContainerByResClassAndTypeQueryRequest
            CeContainerByResClassAndTypeQueryRequest containerRequest = CeContainerByResClassAndTypeQueryRequest.builder()
                    .parentCeResId(Long.valueOf(request.getSuperiorId()))
                    .parentCeResClass(request.getSuperiorResourceType())
                    .ceResSortNoIds(ceResSortNoIds)
                    .build();

            // 查询下级用能资源
            List<EnergyResourceTypeDto> ceContainerDtos = containerMapper.queryCeContainerByCeResClassAndCeResType(containerRequest);

            // 检查 ceContainerDtos 是否非空
            if (CollectionUtil.isNotEmpty(ceContainerDtos)) {
                for (EnergyResourceTypeDto containerDto : ceContainerDtos) {
                    // 构建 EnergyResourceDto 对象
                    EnergyResourceDto energyResourceDto = buildEnergyResourceDto(containerDto);

                    // 构建 GetCeResourceDetailRequest 对象
                    GetCeResourceDetailRequest partResNo = GetCeResourceDetailRequest.builder()
                            .supperCeResSortNo(containerDto.getCeResSortNoId())
                            .lowerCeResType(request.getCeResType())
                            .supperCeResClass(3)
                            .treeId(request.getTreeId())
                            .build();

                    // 查询子资源
                    List<CeResourceDetailDto> partResNos = ceResourceDetailService.querySubResourcesBySuperResource(partResNo);
                    if (CollectionUtil.isNotEmpty(partResNos)) {
                        // 对子资源进行分组
                        Map<Integer, List<String>> listMap = partResNos.stream()
                                .collect(Collectors.groupingBy(CeResourceDetailDto::getLowerCeResClass,
                                        Collectors.mapping(CeResourceDetailDto::getLowerCeResSortNo, Collectors.toList())));

                        // 检查并设置是否有子节点
                        // 容器
                        checkAndSetHasChildren(energyResourceDto, listMap.getOrDefault(3, Collections.emptyList()), 3);
                        // 如果已经为TRUE了，就不需要再查询设备层级的了
                        if (!energyResourceDto.getHasChildren()) {
                            // 设备
                            checkAndSetHasChildren(energyResourceDto, listMap.getOrDefault(4, Collections.emptyList()), 4);
                        }
                    } else {
                        energyResourceDto.setHasChildren(false);
                    }

                    // 将处理结果添加到 resourceDtos 列表中
                    resourceDtos.add(energyResourceDto);
                }
            }
        }
    }

    /**
     * 查询设备数据
     * 并将处理结果添加到 resourceDtos 列表中。
     *
     * @param request        EnergyResourceTreeRequest 对象，包含用能资源树请求的信息。
     * @param resourceDtos   用于存放处理结果的 EnergyResourceDto 列表。
     * @param ceResSortNoIds 用能资源分类Id
     */
    private void processDevice(EnergyResourceTreeRequest request, List<EnergyResourceDto> resourceDtos, List<String> ceResSortNoIds) {
        // 检查 ceResSortNos 是否非空
        if (CollectionUtil.isNotEmpty(ceResSortNoIds)) {
            // 根据 ceResSortNos 构建 CeDeviceByResClassAndTypeQueryRequest
            CeDeviceByResClassAndTypeQueryRequest deviceRequest = CeDeviceByResClassAndTypeQueryRequest.builder()
                    .parentCeResClass(request.getSuperiorResourceType())
                    .parentCeResId(Long.valueOf(request.getSuperiorId()))
                    .ceResSortNoIds(ceResSortNoIds)
                    .build();

            // 查询下级用能资源
            List<EnergyResourceTypeDto> ceDeviceDtos = deviceMapper.queryCeDeviceByCeResClassAndCeResType(deviceRequest);

            // 检查 ceDeviceDtos 是否非空
            if (CollectionUtil.isNotEmpty(ceDeviceDtos)) {
                for (EnergyResourceTypeDto deviceDto : ceDeviceDtos) {
                    // 构建 EnergyResourceDto 对象
                    EnergyResourceDto energyResourceDto = buildEnergyResourceDto(deviceDto);

                    // 构建 GetCeResourceDetailRequest 对象
                    GetCeResourceDetailRequest partResNo = GetCeResourceDetailRequest.builder()
                            .supperCeResSortNo(deviceDto.getCeResSortNoId())
                            .lowerCeResType(request.getCeResType())
                            .supperCeResClass(4)
                            .treeId(request.getTreeId())
                            .build();

                    // 查询子资源
                    List<CeResourceDetailDto> partResNos = ceResourceDetailService.querySubResourcesBySuperResource(partResNo);
                    if (CollectionUtil.isNotEmpty(partResNos)) {
                        // 对子资源进行分组
                        Map<Integer, List<String>> listMap = partResNos.stream()
                                .collect(Collectors.groupingBy(CeResourceDetailDto::getLowerCeResClass,
                                        Collectors.mapping(CeResourceDetailDto::getLowerCeResSortNo, Collectors.toList())));

                        // 检查并设置是否有子节点
                        // 设备
                        checkAndSetHasChildren(energyResourceDto, listMap.getOrDefault(4, Collections.emptyList()), 4);
                    } else {
                        energyResourceDto.setHasChildren(false);
                    }

                    // 将处理结果添加到 resourceDtos 列表中
                    resourceDtos.add(energyResourceDto);
                }
            }
        }
    }

    /**
     * 根据传入的 EnergyResourceTypeDto 对象、资源类别和资源类型构建 EnergyResourceDto 对象。
     *
     * @param resourceTypeDto EnergyResourceTypeDto 对象，包含了构建 EnergyResourceDto 所需的信息。
     * @return 返回构建好的 EnergyResourceDto 对象。
     */
    private EnergyResourceDto buildEnergyResourceDto(EnergyResourceTypeDto resourceTypeDto) {
        // 使用 EnergyResourceDto 的 builder 构造方法，
        // 并利用 resourceTypeDto 对象中的相应字段和传入的资源类别和资源类型构建 EnergyResourceDto 对象。
        return EnergyResourceDto.builder()
                .ceResClass(Integer.valueOf(resourceTypeDto.getCeResClass()))
                .ceResName(resourceTypeDto.getCeResName())
                .ceResType(resourceTypeDto.getCeResType())
                .ceResSortNo(resourceTypeDto.getCeResSortNo())
                .ceResSortNoId(resourceTypeDto.getCeResSortNoId())
                .ceResSortIcon(resourceTypeDto.getCeResSortIcon())
                .correlTableMeter(resourceTypeDto.getCorrelTableMeter())
                .id(resourceTypeDto.getId())
                .sortSn(resourceTypeDto.getSortSn())
                .build();
    }

    /**
     * 检查并设置 resourceDto 对象的 hasChildren 属性。
     * 如果 childIds 列表非空，并且其中的 ID 在数据库中对应的记录数量大于 0，那么将 hasChildren 属性设置为 true，
     * 否则将 hasChildren 属性设置为 false。
     *
     * @param resourceDto    需要设置的 EnergyResourceDto 对象。
     * @param ceResSortNoIds 用能资源分类id
     * @param childClass     用能资源类别
     */
    private void checkAndSetHasChildren(EnergyResourceDto resourceDto, List<String> ceResSortNoIds, int childClass) {
        // 检查 childIds 是否非空
        if (CollectionUtil.isNotEmpty(ceResSortNoIds)) {
            int count;
            // 根据 childClass 的值选择不同的查询方式
            switch (childClass) {
                // 区域
                case 2:
                    CePartByResClassAndTypeQueryRequest partRequest = CePartByResClassAndTypeQueryRequest.builder()
                            .ceCustId(resourceDto.getId())
                            .ceResSortNoIds(ceResSortNoIds)
                            .build();

                    // 查询下级用能资源
                    count = partMapper.queryCePartByCeResClassAndCeResType(partRequest).size();
                    break;
                // 容器
                case 3:
                    CeContainerByResClassAndTypeQueryRequest containerRequest = CeContainerByResClassAndTypeQueryRequest.builder()
                            .parentCeResId(resourceDto.getId())
                            .parentCeResClass(resourceDto.getCeResClass())
                            .ceResSortNoIds(ceResSortNoIds)
                            .build();
                    count = containerMapper.queryCeContainerByCeResClassAndCeResType(containerRequest).size();
                    break;
                // 设备
                case 4:
                    // 根据 ceResSortNos 构建 CeDeviceByResClassAndTypeQueryRequest
                    CeDeviceByResClassAndTypeQueryRequest deviceRequest = CeDeviceByResClassAndTypeQueryRequest.builder()
                            .parentCeResClass(resourceDto.getCeResClass())
                            .parentCeResId(resourceDto.getId())
                            .ceResSortNoIds(ceResSortNoIds)
                            .build();
                    count = deviceMapper.queryCeDeviceByCeResClassAndCeResType(deviceRequest).size();
                    break;
                default:
                    count = 0;
            }
            // 如果 count 大于 0，那么设置 hasChildren 为 true，否则设置为 false
            resourceDto.setHasChildren(count > 0);
        } else {
            // 如果 childIds 为空，那么设置 hasChildren 为 false
            resourceDto.setHasChildren(false);
        }
    }

    /**
     * 处理用能资源的类别，执行相应的操作以获取相关资源的ID列表并添加到给定的ID列表中。
     *
     * @param ceResClassEnum 用能资源的类别枚举
     * @param ceResource     给定的用能资源
     * @throws IllegalArgumentException 如果传递的用能资源类别无效，则抛出非法参数异常
     */
    private void handleCeResClass(CeResClassEnum ceResClassEnum, CeResource ceResource, List<Long> ceResIds) {
        // 根据用能资源的类别执行不同的操作
        switch (ceResClassEnum) {
            case CUSTOMER:
                // 区域
                QueryPartRequest queryPartRequest = new QueryPartRequest();
                queryPartRequest.setCeCustId(ceResource.getId());
                List<CePartDto> cePartDtos = partMapper.listAllEnergyRegions(queryPartRequest);
                // 使用流操作将能源区域记录中的ID提取到ID列表中并返回
                ceResIds.addAll(cePartDtos.stream().map(CePartDto::getId).collect(Collectors.toList()));

                // 容器
                ContainerListRequest containerListRequest = new ContainerListRequest();
                containerListRequest.setCustId(ceResource.getId().toString());
                Page<CeContainerDto> containerDtoPage = new Page<>();
                List<CeContainerDto> containerDtos = containerMapper.listPageContainer(containerListRequest, containerDtoPage).getRecords();
                ceResIds.addAll(containerDtos.stream().map(CeContainerDto::getId).collect(Collectors.toList()));

                // 设备
                CeDevicePageRequest ceDevicePageRequest = new CeDevicePageRequest();
                ceDevicePageRequest.setCustId(ceResource.getId().toString());
                Page<CeDeviceDto> deviceDtoPage = new Page<>();
                List<CeDeviceDto> deviceDtos = deviceMapper.listCeDevcePageDto(ceDevicePageRequest, deviceDtoPage).getRecords();
                ceResIds.addAll(deviceDtos.stream().map(CeDeviceDto::getId).collect(Collectors.toList()));
                break;
            case REGION:
                // 容器
                ContainerListRequest containerListRequest2 = new ContainerListRequest();
                containerListRequest2.setPartId(ceResource.getId().toString());
                Page<CeContainerDto> containerDtoPage2 = new Page<>();
                List<CeContainerDto> containerDtos2 = containerMapper.listPageContainer(containerListRequest2, containerDtoPage2).getRecords();
                ceResIds.addAll(containerDtos2.stream().map(CeContainerDto::getId).collect(Collectors.toList()));

                // 设备
                CeDevicePageRequest ceDevicePageRequest2 = new CeDevicePageRequest();
                ceDevicePageRequest2.setPartId(ceResource.getId().toString());
                Page<CeDeviceDto> deviceDtoPage2 = new Page<>();
                List<CeDeviceDto> deviceDtos2 = deviceMapper.listCeDevcePageDto(ceDevicePageRequest2, deviceDtoPage2).getRecords();
                ceResIds.addAll(deviceDtos2.stream().map(CeDeviceDto::getId).collect(Collectors.toList()));
                break;
            case CONTAINER:
            case DEVICE:
                // 如果类别为CONTAINER或DEVICE，则获取所有容器和设备的ID并添加到ID列表中
                ceResIds.addAll(retrieveAllContainerIds(Collections.singletonList(ceResource.getId())));
                ceResIds.addAll(retrieveAllDeviceIds(Collections.singletonList(ceResource.getId())));
                break;
            default:
                // 如果类别无效，则抛出非法参数异常
                throw new IllegalArgumentException("Invalid ceResClass: " + ceResClassEnum);
        }
    }

    /**
     * 递归检索与给定ID相关的所有容器的ID列表。
     *
     * @param ids 给定的ID
     * @return 与给定ID相关的所有容器的ID列表
     */
    public List<Long> retrieveAllContainerIds(List<Long> ids) {
        // 创建用于存储所有容器ID的列表
        List<Long> ceResIds = new ArrayList<>();

        // 获取与给定ID相关的容器ID列表
        List<Long> containerIds = fetchContainerIds(ids);

        // 如果容器ID列表为空，直接返回空列表
        if (containerIds.isEmpty()) {
            return ceResIds;
        } else {
            if (CollectionUtil.isNotEmpty(containerIds)) {
                ceResIds.addAll(retrieveAllDeviceIds(containerIds));
            }
            retrieveAllContainerIds(containerIds);
            // 递归地检索每个容器中的子容器ID，并将它们添加到结果列表中
            ceResIds.addAll(containerIds);
            return ceResIds;
        }
    }

    /**
     * 根据给定的ID获取相关容器的ID列表。
     *
     * @param ids 给定的ID
     * @return 相关容器的ID列表
     */
    private List<Long> fetchContainerIds(List<Long> ids) {
        // 创建容器列表请求对象(ContainerListRequest)
        ContainerListRequest containerListRequest = new ContainerListRequest();

        // 设置请求的ID列表，仅包含给定的ID
        containerListRequest.setParentCeResIds(ids);

        // 获取容器映射器(ContainerMapper)中符合条件的容器记录
        Page<CeContainerDto> page = new Page<>();
        List<CeContainerDto> records = containerMapper.listPageContainer(containerListRequest, page).getRecords();

        // 使用流操作将容器记录中的ID提取到ID列表中并返回
        return records.stream().map(CeContainerDto::getId).collect(Collectors.toList());
    }

    /**
     * 递归检索与给定ID相关的所有设备的ID列表。
     *
     * @param ids 上级用能资源ID
     * @return 与给定ID相关的所有设备的ID列表
     */
    public List<Long> retrieveAllDeviceIds(List<Long> ids) {
        // 创建用于存储所有设备ID的列表
        List<Long> ceResIds = new ArrayList<>();

        // 获取与给定ID相关的设备ID列表
        List<Long> deviceIds = fetchDeviceIds(ids);
        // 如果设备ID列表为空，直接返回空列表
        if (deviceIds.isEmpty()) {
            return ceResIds;
        } else {
            // 将当前层级的设备ID添加到结果列表中
            ceResIds.addAll(deviceIds);
            // 递归地检索每个设备中的子设备ID，并将它们添加到结果列表中
            List<Long> subDeviceIds = retrieveAllDeviceIds(deviceIds);
            ceResIds.addAll(subDeviceIds);
            return ceResIds;
        }
    }

    /**
     * 根据给定的ID获取相关设备的ID列表。
     *
     * @param ids 上级用能资源ID
     * @return 相关设备的ID列表
     */
    private List<Long> fetchDeviceIds(List<Long> ids) {
        // 创建设备分页请求对象(CeDevicePageRequest)
        CeDevicePageRequest ceDevicePageRequest = new CeDevicePageRequest();

        // 设置请求的CE部件ID列表，仅包含给定的ID
        ceDevicePageRequest.setParentCeResIds(ids);

        // 获取设备映射器(DeviceMapper)中符合条件的设备记录
        Page<CeDeviceDto> page = new Page<>();
        List<CeDeviceDto> deviceDtos = deviceMapper.listCeDevcePageDto(ceDevicePageRequest, page).getRecords();

        // 使用流操作将设备记录中的ID提取到ID列表中并返回
        return deviceDtos.stream().map(CeDeviceDto::getId).collect(Collectors.toList());
    }

    /**
     * 删除与给定用能资源ID相关的能源点信息及其关联数据。
     *
     * @param ceResIds 给定的用能资源ID列表
     */
    private void deleteEnergyPoints(List<Long> ceResIds) {
        // 创建Lambda查询包装器，以查询与给定用能资源ID相关的能源点信息
        LambdaQueryWrapper<CePoint> pointQueryWrapper = new LambdaQueryWrapper<>();
        pointQueryWrapper.in(CePoint::getCeResId, ceResIds);

        // 获取符合条件的能源点信息列表
        List<CePoint> cePoints = cePointService.list(pointQueryWrapper);

        // 从能源点信息列表中提取能源点ID列表
        List<Long> pointIds = cePoints.stream().map(CePoint::getId).collect(Collectors.toList());

        if (CollectionUtil.isNotEmpty(pointIds)) {
            // 根据能源点ID列表批量删除能源点信息记录
            cePointService.removeByIds(pointIds);

            // 创建Lambda查询包装器，以查询与能源点ID相关的能源点计量信息
            LambdaQueryWrapper<CePointMeter> pointMeterQueryWrapper = new LambdaQueryWrapper<>();
            pointMeterQueryWrapper.in(CePointMeter::getCePointId, pointIds);

            // 获取符合条件的能源点计量信息列表
            List<CePointMeter> cePointMeters = cePointMeterService.list(pointMeterQueryWrapper);

            // 从能源点计量信息列表中提取能源点计量信息ID列表
            List<Long> cePointMeterIds = cePointMeters.stream().map(CePointMeter::getId).collect(Collectors.toList());

            // 根据能源点计量信息ID列表批量删除能源点计量信息记录
            if (CollectionUtil.isNotEmpty(cePointMeterIds)) {
                cePointMeterService.removeByCePointIds(cePointMeterIds);
            }
        }
    }
}
