package com.gaea.iesms.bm.standalone.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.lang.Dict;
import cn.hutool.core.map.MapUtil;
import com.gaea.base.error.BizException;
import com.gaea.iesms.bm.standalone.mapper.StatDataMapper;
import com.gaea.iesms.bm.standalone.model.dto.StandaloneDataInfoDTO;
import com.gaea.iesms.bm.standalone.model.dto.StatisticsDateDTO;
import com.gaea.iesms.bm.standalone.model.enums.StandaloneDataTypeEnum;
import com.gaea.iesms.bm.standalone.model.enums.StandaloneDateTypeEnum;
import com.gaea.iesms.bm.standalone.service.StatCecntrEconsService;
import com.gaea.iesms.bm.standalone.service.StatDataService;
import com.gaea.iesms.bm.standalone.util.StatisticsDateUtils;
import com.gaea.iesms.core.feign.request.StandaloneDataCurveBatchRangeRequest;
import com.gaea.iesms.core.feign.request.StandaloneDataCurveBatchRequest;
import com.gaea.iesms.core.feign.request.StandaloneDataCurveRequest;
import com.gaea.iesms.core.feign.request.StandaloneDataInfoBatchRequest;
import com.gaea.iesms.core.feign.response.*;
import com.gaea.iesms.core.model.entity.standalone.StatCecntrEconsDay;
import com.gaea.iesms.core.model.entity.standalone.StatCecntrEconsMonth;
import com.gaea.iesms.core.model.entity.standalone.StatCecntrEconsYear;
import com.gaea.utils.ConvertUtils;
import com.gaea.utils.DateUtils;
import com.gaea.utils.NumberUtils;
import com.gaea.utils.StringUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author wuyuxuan
 * @version 1.0.0
 * @since 2023/12/4
 */
@Service
public class StatDataServiceImpl implements StatDataService {

    private static final String[] DATA_TYPES = {"ECONS", "ELOAD", "EQHOURS", "EXPENSE", "ECONS1", "ECONS2", "ECONS3", "ECONS4"};
    private static final String[] DATE_TYPES = {"DAY", "MONTH", "YEAR"};
    private static final String[] LEVELS = {"DEVICE", "CONTAINER", "PART", "CUST", "ORG"};

    @Resource
    private StatDataMapper dataMapper;

    @Resource
    private StatCecntrEconsService statCecntrEconsService;

    @Override
    public List<BigDecimal> getCurve(StandaloneDataCurveRequest request) {
        StandaloneDataCurveBatchRequest req = new StandaloneDataCurveBatchRequest();
        req.setKeys(Collections.singletonList(request.getKey()));
        req.setDataType(request.getDataType());
        req.setDateType(request.getDateType());
        req.setDateValue(request.getDateValue());
        req.setLevel(request.getLevel());
        return batchGetCurve(req).get(request.getKey());
    }

    @Override
    public Map<String, List<BigDecimal>> batchGetCurve(StandaloneDataCurveBatchRequest request) {
        if (CollectionUtil.isEmpty(request.getKeys()) || StringUtils.isBlank(request.getLevel()) || StringUtils.isBlank(request.getDataType()) || StringUtils.isBlank(request.getDateType())) {
            throw BizException.of("参数有误");
        }
        if (!StringUtils.equalsAny(request.getDataType(), DATA_TYPES)) {
            throw BizException.of("数据类型参数有误");
        }
        if (!StringUtils.equalsAny(request.getDateType(), DATE_TYPES)) {
            throw BizException.of("日期类型参数有误");
        }
        if (!StringUtils.equalsAny(request.getLevel(), LEVELS)) {
            throw BizException.of("数据层级参数有误");
        }
        StatisticsDateDTO dateDTO;
        if (StringUtils.equals(request.getDateType(), DATE_TYPES[0])) {
            dateDTO = StatisticsDateUtils.date(request.getDateValue());
        } else if (StringUtils.equals(request.getDateType(), DATE_TYPES[1])) {
            dateDTO = StatisticsDateUtils.date(request.getDateValue() + "-01");
        } else {
            dateDTO = StatisticsDateUtils.date(request.getDateValue() + "-01-01");
        }
        Map<String, List<BigDecimal>> result = MapUtil.newHashMap();

        if (StringUtils.equals(request.getLevel(), "DEVICE")) {
            // TODO 设备
            if (StringUtils.equals(request.getDateType(), DATE_TYPES[0])) {
                // TODO 设备 电量 日
                List<Dict> list = dataMapper.getDeviceCurveDay(request.getKeys(), request.getDataType(), dateDTO.getDayDateDTO().getDate());
                formatValue(result, list, 96);
            } else if (StringUtils.equals(request.getDateType(), DATE_TYPES[1])) {
                // TODO 设备 电量 月
                List<Dict> list = dataMapper.getDeviceCurveMonth(request.getKeys(), request.getDataType(), dateDTO.getMonthDateDTO().getStart(), dateDTO.getMonthDateDTO().getEnd());
                formatValue(result, list, dateDTO.getMonthDateDTO().getDayCountOfMonth());
            } else if (StringUtils.equals(request.getDateType(), DATE_TYPES[2])) {
                // TODO 设备 电量 年
                List<Dict> list = dataMapper.getDeviceCurveYear(request.getKeys(), request.getDataType(), dateDTO.getYearDateDTO().getStart(), dateDTO.getYearDateDTO().getEnd());
                formatValue(result, list, dateDTO.getYearDateDTO().getMonthCountOfYear());
            }
        } else if (StringUtils.equals(request.getLevel(), "CONTAINER")) {
            // TODO 容器
            if (StringUtils.equals(request.getDateType(), DATE_TYPES[0])) {
                // TODO 日
                List<Dict> list = dataMapper.getContainerCurveDay(request.getKeys(), request.getDataType(), dateDTO.getDayDateDTO().getDate());
                formatValue(result, list, 96);
            } else if (StringUtils.equals(request.getDateType(), DATE_TYPES[1])) {
                // TODO 月
                List<Dict> list = dataMapper.getContainerCurveMonth(request.getKeys(), request.getDataType(), dateDTO.getMonthDateDTO().getStart(), dateDTO.getMonthDateDTO().getEnd());
                formatValue(result, list, dateDTO.getMonthDateDTO().getDayCountOfMonth());
            } else if (StringUtils.equals(request.getDateType(), DATE_TYPES[2])) {
                // TODO 年
                List<Dict> list = dataMapper.getContainerCurveYear(request.getKeys(), request.getDataType(), dateDTO.getYearDateDTO().getStart(), dateDTO.getYearDateDTO().getEnd());
                formatValue(result, list, dateDTO.getYearDateDTO().getMonthCountOfYear());
            }
        } else if (StringUtils.equals(request.getLevel(), "CUST")) {
            // TODO 用户
            if (StringUtils.equals(request.getDateType(), DATE_TYPES[0])) {
                // TODO 日
                List<Dict> list = dataMapper.getCustCurveDay(request.getKeys(), request.getDataType(), dateDTO.getDayDateDTO().getDate());
                formatValue(result, list, 96);
            } else if (StringUtils.equals(request.getDateType(), DATE_TYPES[1])) {
                // TODO 月
                List<Dict> list = dataMapper.getCustCurveMonth(request.getKeys(), request.getDataType(), dateDTO.getMonthDateDTO().getStart(), dateDTO.getMonthDateDTO().getEnd());
                formatValue(result, list, dateDTO.getMonthDateDTO().getDayCountOfMonth());
            } else if (StringUtils.equals(request.getDateType(), DATE_TYPES[2])) {
                // TODO 年
                List<Dict> list = dataMapper.getCustCurveYear(request.getKeys(), request.getDataType(), dateDTO.getYearDateDTO().getStart(), dateDTO.getYearDateDTO().getEnd());
                formatValue(result, list, dateDTO.getYearDateDTO().getMonthCountOfYear());
            }
        } else {
            throw BizException.of("此业务暂无统计");
        }
        request.getKeys().forEach(key -> {
            if (!result.containsKey(key)) {
                result.put(key, new ArrayList<>());
            } else {
                result.put(key, result.get(key).stream().map(item -> {
                    if (item == null) {
                        return null;
                    }
                    return item.setScale(2, BigDecimal.ROUND_HALF_UP).stripTrailingZeros();
                }).collect(Collectors.toList()));
            }
        });
        return result;
    }

    private static List<String> fromDate(StandaloneDateTypeEnum dateTypeEnum, String start, String end) {
        List<String> dateList = new ArrayList<>();
        Calendar calendar = Calendar.getInstance();
        Date startDate = DateUtil.parse(start, dateTypeEnum.getPattern());
        Date endDate = DateUtil.parse(end, dateTypeEnum.getPattern());
        calendar.setTime(startDate);
        if (dateTypeEnum.equals(StandaloneDateTypeEnum.DAY)) {
            while (calendar.getTime().before(endDate) || calendar.getTime().equals(endDate)) {
                dateList.add(DateUtil.formatDate(calendar.getTime()));
                calendar.add(Calendar.DAY_OF_MONTH, 1);
            }
        } else if (dateTypeEnum.equals(StandaloneDateTypeEnum.MONTH)) {
            while (calendar.getTime().before(endDate) || calendar.getTime().equals(endDate)) {
                dateList.add(DateUtil.format(calendar.getTime(), "yyyyMM"));
                calendar.add(Calendar.MONTH, 1);
            }
        } else if (dateTypeEnum.equals(StandaloneDateTypeEnum.YEAR)) {
            while (calendar.getTime().before(endDate) || calendar.getTime().equals(endDate)) {
                dateList.add(DateUtil.format(calendar.getTime(), "yyyy"));
                calendar.add(Calendar.YEAR, 1);
            }
        }
        return dateList;
    }

    @Override
    public Map<String, List<BigDecimal>> batchGetRangeCurve(StandaloneDataCurveBatchRangeRequest request) {
        if (CollectionUtil.isEmpty(request.getKeys())) {
            throw BizException.of("数据标识不能为空");
        }
        if (StringUtils.isBlank(request.getStartDateValue()) || StringUtils.isBlank(request.getEndDateValue())) {
            throw BizException.of("日期值不能为空");
        }

        String itemType = request.getLevel() + request.getDataType() + request.getDateType();
        Map<String, Map<String, BigDecimal>> dataMap;

        if (StringUtils.equals(StandaloneDataTypeEnum.CONTAINER_ECONS_DAY.getValue(), itemType)) {
            dataMap = statCecntrEconsService.getRangeDay(request.getKeys(), request.getStartDateValue(), request.getEndDateValue(), StatCecntrEconsDay::getEconsValueDay);
        } else if (StringUtils.equals(StandaloneDataTypeEnum.CONTAINER_ECONS_MONTH.getValue(), itemType)) {
            dataMap = statCecntrEconsService.getRangeMonth(request.getKeys(), request.getStartDateValue(), request.getEndDateValue(), StatCecntrEconsMonth::getEconsValueMonth);
        } else if (StringUtils.equals(StandaloneDataTypeEnum.CONTAINER_ECONS_YEAR.getValue(), itemType)) {
            dataMap = statCecntrEconsService.getRangeYear(request.getKeys(), request.getStartDateValue(), request.getEndDateValue(), StatCecntrEconsYear::getEconsValueYear);
        } else if (StringUtils.equals(StandaloneDataTypeEnum.CONTAINER_ECONS1_DAY.getValue(), itemType)) {
            dataMap = statCecntrEconsService.getRangeDay(request.getKeys(), request.getStartDateValue(), request.getEndDateValue(), StatCecntrEconsDay::getEconsValueDr1);
        } else if (StringUtils.equals(StandaloneDataTypeEnum.CONTAINER_ECONS1_MONTH.getValue(), itemType)) {
            dataMap = statCecntrEconsService.getRangeMonth(request.getKeys(), request.getStartDateValue(), request.getEndDateValue(), StatCecntrEconsMonth::getEconsValueMr1);
        } else if (StringUtils.equals(StandaloneDataTypeEnum.CONTAINER_ECONS1_YEAR.getValue(), itemType)) {
            dataMap = statCecntrEconsService.getRangeYear(request.getKeys(), request.getStartDateValue(), request.getEndDateValue(), StatCecntrEconsYear::getEconsValueYr1);
        } else if (StringUtils.equals(StandaloneDataTypeEnum.CONTAINER_ECONS2_DAY.getValue(), itemType)) {
            dataMap = statCecntrEconsService.getRangeDay(request.getKeys(), request.getStartDateValue(), request.getEndDateValue(), StatCecntrEconsDay::getEconsValueDr2);
        } else if (StringUtils.equals(StandaloneDataTypeEnum.CONTAINER_ECONS2_MONTH.getValue(), itemType)) {
            dataMap = statCecntrEconsService.getRangeMonth(request.getKeys(), request.getStartDateValue(), request.getEndDateValue(), StatCecntrEconsMonth::getEconsValueMr2);
        } else if (StringUtils.equals(StandaloneDataTypeEnum.CONTAINER_ECONS2_YEAR.getValue(), itemType)) {
            dataMap = statCecntrEconsService.getRangeYear(request.getKeys(), request.getStartDateValue(), request.getEndDateValue(), StatCecntrEconsYear::getEconsValueYr2);
        } else if (StringUtils.equals(StandaloneDataTypeEnum.CONTAINER_ECONS3_DAY.getValue(), itemType)) {
            dataMap = statCecntrEconsService.getRangeDay(request.getKeys(), request.getStartDateValue(), request.getEndDateValue(), StatCecntrEconsDay::getEconsValueDr3);
        } else if (StringUtils.equals(StandaloneDataTypeEnum.CONTAINER_ECONS3_MONTH.getValue(), itemType)) {
            dataMap = statCecntrEconsService.getRangeMonth(request.getKeys(), request.getStartDateValue(), request.getEndDateValue(), StatCecntrEconsMonth::getEconsValueMr3);
        } else if (StringUtils.equals(StandaloneDataTypeEnum.CONTAINER_ECONS3_YEAR.getValue(), itemType)) {
            dataMap = statCecntrEconsService.getRangeYear(request.getKeys(), request.getStartDateValue(), request.getEndDateValue(), StatCecntrEconsYear::getEconsValueYr3);
        } else if (StringUtils.equals(StandaloneDataTypeEnum.CONTAINER_ECONS4_DAY.getValue(), itemType)) {
            dataMap = statCecntrEconsService.getRangeDay(request.getKeys(), request.getStartDateValue(), request.getEndDateValue(), StatCecntrEconsDay::getEconsValueDr4);
        } else if (StringUtils.equals(StandaloneDataTypeEnum.CONTAINER_ECONS4_MONTH.getValue(), itemType)) {
            dataMap = statCecntrEconsService.getRangeMonth(request.getKeys(), request.getStartDateValue(), request.getEndDateValue(), StatCecntrEconsMonth::getEconsValueMr4);
        } else if (StringUtils.equals(StandaloneDataTypeEnum.CONTAINER_ECONS4_YEAR.getValue(), itemType)) {
            dataMap = statCecntrEconsService.getRangeYear(request.getKeys(), request.getStartDateValue(), request.getEndDateValue(), StatCecntrEconsYear::getEconsValueYr4);
        } else {
            throw BizException.of("此业务暂无统计");
        }

        Map<String, List<BigDecimal>> result = new HashMap<>();
        List<String> strings = fromDate(Objects.requireNonNull(StandaloneDateTypeEnum.getByValue(request.getDateType())), request.getStartDateValue(), request.getEndDateValue());
        for (String key : request.getKeys()) {
            if (!dataMap.containsKey(key)) {
                result.put(key, new ArrayList<>());
            } else {
                Map<String, BigDecimal> stringBigDecimalMap = dataMap.get(key);
                List<BigDecimal> list = new ArrayList<>();
                for (String string : strings) {
                    list.add(NumberUtils.round(stringBigDecimalMap.get(string), 2));
                }
                result.put(key, list);
            }
        }


//        if (!StringUtils.equalsAny(request.getDataType(), DATA_TYPES)) {
//            throw BizException.of("数据类型参数有误");
//        }
//        if (!StringUtils.equalsAny(request.getDateType(), DATE_TYPES)) {
//            throw BizException.of("日期类型参数有误");
//        }
//        if (!StringUtils.equalsAny(request.getLevel(), LEVELS)) {
//            throw BizException.of("数据层级参数有误");
//        }
//
//        Map<String, List<BigDecimal>> result = MapUtil.newHashMap();
//        if (StringUtils.equals(request.getDateType(), DATE_TYPES[0])) {
//            // TODO 日
//            List<String> dateList = new ArrayList<>();
//            Date startDate = DateUtil.parseDate(request.getStartDateValue());
//            Date endDate = DateUtil.parseDate(request.getEndDateValue());
//            Calendar calendar = Calendar.getInstance();
//            calendar.setTime(startDate);
//            while (calendar.getTime().before(endDate) || calendar.getTime().equals(endDate)) {
//                dateList.add(DateUtil.formatDate(calendar.getTime()));
//                calendar.add(Calendar.DAY_OF_MONTH, 1);
//            }
//            if (StringUtils.equals(request.getLevel(), LEVELS[1])) {
//                // TODO 容器
//                List<Dict> list = dataMapper.getContainerRangeCurveDay(request.getKeys(), request.getDataType(), startDate, endDate, dateList);
//                formatValue(result, list, dateList);
//            }
//        } else if (StringUtils.equals(request.getDateType(), DATE_TYPES[1])) {
//            // TODO 月
//            if (StringUtils.equals(request.getLevel(), LEVELS[1])) {
//                List<String> dateList = new ArrayList<>();
//                Date startDate = DateUtil.parse(request.getStartDateValue(), "yyyy-MM");
//                Date endDate = DateUtil.parse(request.getEndDateValue(), "yyyy-MM");
//                Calendar calendar = Calendar.getInstance();
//                calendar.setTime(startDate);
//                while (calendar.getTime().before(endDate) || calendar.getTime().equals(endDate)) {
//                    dateList.add(DateUtil.format(calendar.getTime(), "yyyyMM"));
//                    calendar.add(Calendar.MONTH, 1);
//                }
//                // TODO 容器
//                List<Dict> list = dataMapper.getContainerRangeCurveMonth(request.getKeys(), request.getDataType(), request.getStartDateValue().replace("-", ""), request.getEndDateValue().replace("-", ""), dateList);
//                formatValue(result, list, dateList);
//            }
//        } else {
//            throw BizException.of("此业务暂无统计");
//        }
//        request.getKeys().forEach(key -> {
//            if (!result.containsKey(key)) {
//                result.put(key, new ArrayList<>());
//            } else {
//                result.put(key, result.get(key).stream().map(item -> {
//                    if (item == null) {
//                        return null;
//                    }
//                    return item.setScale(2, BigDecimal.ROUND_HALF_UP).stripTrailingZeros();
//                }).collect(Collectors.toList()));
//            }
//        });
        return result;
    }

    private void formatValue(Map<String, List<BigDecimal>> result, List<Dict> dataList, int count) {
        dataList.forEach(data -> {
            List<BigDecimal> list = new ArrayList<>();
            for (int i = 1; i <= count; i++) {
                list.add(data.getBigDecimal("value" + i));
            }
            result.put(data.getStr("id"), list);
        });
    }

//    private void formatValue(Map<String, List<BigDecimal>> result, List<Dict> dataList, List<String> dateList) {
//        dataList.forEach(data -> {
//            List<BigDecimal> list = new ArrayList<>();
//            for (String date : dateList) {
//                list.add(data.getBigDecimal(date));
//            }
//            result.put(data.getStr("id"), list);
//        });
//    }

    @Override
    public Map<String, StandaloneDataInfoVo> batchGetInfo(StandaloneDataInfoBatchRequest request) {
        if (request.getKeys() == null || request.getKeys().size() == 0 || StringUtils.isBlank(request.getLevel())) {
            throw BizException.of("参数有误");
        }

        if (StringUtils.isBlank(request.getDateStat())) {
            request.setDateStat(DateUtils.today());
        }

        List<StandaloneDataInfoDTO> infoVoList;
        if (StringUtils.equals(request.getLevel(), "DEVICE")) {
            infoVoList = dataMapper.getDeviceInfo(request.getKeys(), request.getDateStat());
        } else if (StringUtils.equals(request.getLevel(), "CONTAINER")) {
            infoVoList = dataMapper.getContainerInfo(request.getKeys(), request.getDateStat());
        } else if (StringUtils.equals(request.getLevel(), "PART")) {
            infoVoList = dataMapper.getPartInfo(request.getKeys(), request.getDateStat());
        } else if (StringUtils.equals(request.getLevel(), "CUST")) {
            infoVoList = dataMapper.getCustInfo(request.getKeys(), request.getDateStat());
        } else {
            throw BizException.of("此业务暂无数据");
        }

        Map<String, StandaloneDataInfoVo> result = new HashMap<>();

        Map<String, StandaloneDataInfoDTO> collect = infoVoList.stream().collect(Collectors.toMap(StandaloneDataInfoDTO::getKey, Function.identity(), (oldVal, newVal) -> newVal));

        request.getKeys().forEach(key -> {

            StandaloneDataInfoVo dataInfoVo = new StandaloneDataInfoVo();
            dataInfoVo.setKey(key);

            StandaloneDataInfoDTO item = collect.get(key);
            if (item == null) {
                dataInfoVo.setDay(new StandaloneDayDataInfoVo());
                dataInfoVo.setMonth(new StandaloneMonthDataInfoVo());
                dataInfoVo.setYear(new StandaloneYearDataInfoVo());
                dataInfoVo.setAccum(new StandaloneAccumDataInfoVo());
            } else {
                dataInfoVo.setOpsStatus(item.getOpsStatus());
                dataInfoVo.setDay(ConvertUtils.convert(StandaloneDayDataInfoVo.class, item));
                dataInfoVo.setMonth(ConvertUtils.convert(StandaloneMonthDataInfoVo.class, item));
                dataInfoVo.setYear(ConvertUtils.convert(StandaloneYearDataInfoVo.class, item));
                dataInfoVo.setAccum(ConvertUtils.convert(StandaloneAccumDataInfoVo.class, item));
            }
            result.put(key, dataInfoVo);
        });
        return result;
    }
}
