package com.gaea.iesms.bm.standalone.util;

import cn.hutool.core.collection.ListUtil;
import cn.hutool.core.lang.Dict;
import cn.hutool.core.lang.generator.SnowflakeGenerator;
import com.gaea.iesms.bm.standalone.model.dto.AvgMaxMinValueDTO;
import com.gaea.iesms.core.model.core.IAvgMaxMinValue;
import com.gaea.iesms.core.model.entity.standalone.StatCepartEconsDay;
import com.gaea.utils.*;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;

/**
 * @author wuyuxuan
 * @version 1.0.0
 * @since 2023/4/17
 */
public class StatisticsDataUtils {

    private static final SnowflakeGenerator SNOWFLAKE = new SnowflakeGenerator();
    private static final int NUM = 96;
    private static final String ELOAD_NAME = "eloadValue";

    public static <T, R> BigDecimal computeAddByList(List<T> list, Function<? super T, BigDecimal> mapper) {
        return list.stream().map(mapper).filter(Objects::nonNull).reduce(BigDecimal.ZERO, BigDecimal::add);
    }

    public static <T> T computeAdd96PointEconsValue(T... objs) {
        return computeAdd96PointEconsValue(ListUtil.of(objs));
    }

    public static <T> T computeAdd96PointEloadValue(T... objs) {
        return computeAdd96PointEloadValue(ListUtil.of(objs));
    }

    public static <T> T computeAdd96PointEloadValue(List<T> objs) {
        return computeAdd96PointValue(objs, "eloadValue");
    }

    public static <T> T computeAdd96PointEconsValue(List<T> objs) {
        return computeAdd96PointValue(objs, "econsValue");
    }

    /**
     * 将多个 96 点中的曲线相加合并成一条曲线
     *
     * @param objs
     * @param nameVal
     * @param <T>
     * @return
     */
    public static <T> T computeAdd96PointValue(List<T> objs, String nameVal) {
        if (objs == null || objs.size() == 0) {
            throw new RuntimeException();
        }
        Dict data = Dict.create();
        T target = null;
        for (T obj : objs) {
            if (obj == null) {
                continue;
            }
            target = obj;
            Dict dict = ConvertUtils.convert(Dict.class, obj);
            for (int i = 1; i <= NUM; i++) {
                String name = nameVal + StringUtils.fillZero(i, 2);
                dict.put(name, addBigDecimal(dict.getBigDecimal(name), dict.getBigDecimal(name)));
            }
        }
        ObjectUtils.copyProperties(data, target);
        return target;
    }

    /**
     * 计算 96 点中最大值、最小值、平均值
     *
     * @param obj
     * @param date
     * @param <T>
     * @return
     */
    public static <T> T computeMaxAndMinValue(T obj, Date date) {
        int count = 0;
        BigDecimal total = null;
        BigDecimal curValue = null;
        Date curValueTime = null;
        BigDecimal maxValue = null;
        Date maxValueTime = null;
        BigDecimal minValue = null;
        Date minValueTime = null;
        Dict dict = ConvertUtils.convert(Dict.class, obj);
        for (int i = 1; i <= NUM; i++) {
            BigDecimal value = dict.getBigDecimal(ELOAD_NAME + StringUtils.fillZero(i, 2));
            if (value == null) {
                continue;
            }
            curValue = value;
            curValueTime = DateUtils.offsetMinute(date, (i - 1) * 15);

            if (maxValue == null) {
                maxValue = curValue;
                maxValueTime = curValueTime;
            }

            if (minValue == null) {
                minValue = curValue;
                minValueTime = curValueTime;
            }

            if (NumberUtils.gte(curValue, maxValue)) {
                maxValue = curValue;
                maxValueTime = curValueTime;
            }
            if (NumberUtils.lt(curValue, minValue)) {
                minValue = curValue;
                minValueTime = curValueTime;
            }
            total = NumberUtils.add(total, curValue);
            count++;
        }
        AvgMaxMinValueDTO valueDTO = new AvgMaxMinValueDTO();
        valueDTO.setAvgValue(count == 0 ? null : NumberUtils.div(total, count));
        valueDTO.setCurValue(curValue);
        valueDTO.setCurValueTime(curValueTime);
        valueDTO.setMaxValue(maxValue);
        valueDTO.setMaxValueTime(maxValueTime);
        valueDTO.setMinValue(minValue);
        valueDTO.setMinValueTime(minValueTime);
        ObjectUtils.copyProperties(valueDTO, obj);
        return obj;
    }

    /**
     * 计算 list 集合中最大值、最小值、平均值
     *
     * @param list
     * @param <T>
     * @return
     */
    public static <T> AvgMaxMinValueDTO computeMaxAndMinValue(List<T> list) {
        AvgMaxMinValueDTO valueDTO = new AvgMaxMinValueDTO();
        if (list == null || list.size() == 0) {
            return valueDTO;
        }
        BigDecimal total = null;
        int count = 0;
        BigDecimal maxValue = null;
        Date maxValueTime = null;
        BigDecimal minValue = null;
        Date minValueTime = null;
        for (T obj : list) {
            IAvgMaxMinValue item = (IAvgMaxMinValue) obj;

            if (NumberUtils.gt(item.getMaxValue(), maxValue)) {
                maxValue = item.getMaxValue();
                maxValueTime = item.getMaxValueTime();
            }

            if (NumberUtils.lt(item.getMinValue(), minValue)) {
                minValue = item.getMinValue();
                minValueTime = item.getMinValueTime();
            }

            if (item.getAvgValue() != null) {
                total = NumberUtils.add(total, item.getAvgValue());
                count++;
            }
        }
        valueDTO.setAvgValue(count == 0 ? null : NumberUtils.div(total, count));
        valueDTO.setMaxValue(maxValue);
        valueDTO.setMaxValueTime(maxValueTime);
        valueDTO.setMinValue(minValue);
        valueDTO.setMinValueTime(minValueTime);
        return valueDTO;
    }

    /**
     * 数值相加，两者为 null 相加依然是 null
     *
     * @param a
     * @param b
     * @return
     */
    public static BigDecimal addBigDecimal(BigDecimal a, BigDecimal b) {
        if (a == null && b == null) {
            return null;
        }
        return NumberUtils.add(a, b);
    }

    /**
     * 初始化集合数据
     *
     * @param dataList
     * @param <T>
     */
    public static <T> List<T> initData(List<T> dataList) {
        if (dataList == null) {
            return null;
        }
        for (T data : dataList) {
            initData(data);
        }
        return dataList;
    }

    /**
     * 初始化对象数据
     *
     * @param data
     * @param <T>
     */
    public static <T> T initData(T data) {
        if (data == null) {
            return null;
        }
        Dict init = Dict.create()
            .set("id", SNOWFLAKE.next())
            .set("gmtCreate", System.currentTimeMillis())
            .set("gmtModified", System.currentTimeMillis())
            .set("version", 1);
        ObjectUtils.copyProperties(init, data);
        return data;
    }
}
